package ru.yandex.direct.ess.router.models.rule;

import com.google.common.base.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.ess.common.models.BaseLogicObject;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.binlog.model.Operation.INSERT;
import static ru.yandex.direct.dbschema.ppc.Tables.ESS_ADDITIONAL_OBJECTS;

class AdditionalObjectsHandler<T extends BaseLogicObject> {
    private static final Logger logger = LoggerFactory.getLogger(AdditionalObjectsHandler.class);

    private final Class<? extends BaseLogicObject> logicObjectClass;
    private final TableChangesHandler<T> tableChangesHandler;
    private final String logicProcessName;

    AdditionalObjectsHandler(String logicProcessName, Class<? extends BaseLogicObject> logicObjectClass) {
        this.logicProcessName = logicProcessName;
        this.logicObjectClass = logicObjectClass;
        tableChangesHandler = new TableChangesHandler<>();
        tableChangesHandler.addTableChange(new TableChange.Builder<T>()
                .setTable(ESS_ADDITIONAL_OBJECTS)
                .setOperation(INSERT)
                .setValuesFilter(this::isSameLogicProcessor)
                .setMapper(this::mapToLogicObject)
                .build());
    }

    AdditionalObjectsProcessed<T> processAdditionalObjects(BinlogEvent binlogEvent) {
        var additionalObjects = tableChangesHandler.processChanges(binlogEvent);
        var additionalObjectsWithSkippedSize = additionalObjects.size();
        additionalObjects.removeIf(java.util.Objects::isNull);
        var skippedObjectsSize = additionalObjectsWithSkippedSize - additionalObjects.size();
        return new AdditionalObjectsProcessed<>(additionalObjects, skippedObjectsSize);

    }

    private boolean isSameLogicProcessor(ProceededChange proceededChange) {
        return Objects.equal(logicProcessName, proceededChange.getAfter(ESS_ADDITIONAL_OBJECTS.LOGIC_PROCESS_NAME));
    }

    @SuppressWarnings("unchecked")
    private T mapToLogicObject(ProceededChange proceededChange) {
        String serializedLogicObject = proceededChange.getAfter(ESS_ADDITIONAL_OBJECTS.LOGIC_OBJECT);
        T object;
        try {
            object = (T) JsonUtils.fromJson(serializedLogicObject, logicObjectClass);
        } catch (RuntimeException e) {
            object = null;
            logger.error("Failed to parse additional logic object " + serializedLogicObject + ". Skip it.", e);
        }
        return object;
    }
}
