package ru.yandex.direct.ess.router.rules.aggregatedstatuses;

import java.math.BigInteger;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;

import com.google.protobuf.InvalidProtocolBufferException;
import org.jooq.Field;
import org.jooq.Named;
import org.jooq.impl.TableImpl;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.ess.config.aggregatedstatuses.AggregatedStatusesConfig;
import ru.yandex.direct.ess.logicobjects.aggregatedstatuses.AggregatedStatusEventObject;
import ru.yandex.direct.ess.router.models.TEssEvent;
import ru.yandex.direct.ess.router.models.rule.AbstractRule;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.ess.router.utils.GrutTableChange;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;
import ru.yandex.grut.objects.proto.client.Schema.EObjectType;
import ru.yandex.grut.objects.proto.client.Schema.TBannerCandidateMeta;
import ru.yandex.grut.watchlog.Watch.TEvent;

import static ru.yandex.direct.binlog.model.Operation.DELETE;
import static ru.yandex.direct.binlog.model.Operation.INSERT;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_BUTTONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_DISPLAY_HREFS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_LOGOS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_MULTICARD_SETS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.PROMOACTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.AdditionsItemCallouts.ADDITIONS_ITEM_CALLOUTS;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupsDynamic.ADGROUPS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupsPerformance.ADGROUPS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImages.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.BannersAdditions.BANNERS_ADDITIONS;
import static ru.yandex.direct.dbschema.ppc.tables.BannersMinusGeo.BANNERS_MINUS_GEO;
import static ru.yandex.direct.dbschema.ppc.tables.BannersPerformance.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.tables.Bids.BIDS;
import static ru.yandex.direct.dbschema.ppc.tables.BidsArc.BIDS_ARC;
import static ru.yandex.direct.dbschema.ppc.tables.BidsBase.BIDS_BASE;
import static ru.yandex.direct.dbschema.ppc.tables.BidsDynamic.BIDS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.tables.BidsPerformance.BIDS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.tables.BidsRetargeting.BIDS_RETARGETING;
import static ru.yandex.direct.dbschema.ppc.tables.CampOperationsQueue.CAMP_OPERATIONS_QUEUE;
import static ru.yandex.direct.dbschema.ppc.tables.CampOptions.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignPromoactions.CAMPAIGN_PROMOACTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignsCpmPrice.CAMPAIGNS_CPM_PRICE;
import static ru.yandex.direct.dbschema.ppc.tables.Images.IMAGES;
import static ru.yandex.direct.dbschema.ppc.tables.ModerateBannerPages.MODERATE_BANNER_PAGES;
import static ru.yandex.direct.dbschema.ppc.tables.PerfCreatives.PERF_CREATIVES;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.ess.router.utils.ColumnsChangeType.ANY;
import static ru.yandex.grut.libs.proto.event_type.Event.EEventType.ET_OBJECT_REMOVED;

@EssRule(AggregatedStatusesConfig.class)
public class AggregatedStatusesRule extends AbstractRule<AggregatedStatusEventObject> {
    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(AggregatedStatusesRule.class);

    private final TableChangesHandler<AggregatedStatusEventObject> tableChangesHandler = new TableChangesHandler<>();

    private final List<Named> bannersColumns = List.of(BANNERS.STATUS_ACTIVE, BANNERS.STATUS_MODERATE,
            BANNERS.STATUS_POST_MODERATE, BANNERS.STATUS_SITELINKS_MODERATE, BANNERS.PHONEFLAG, BANNERS.STATUS_ARCH,
            BANNERS.STATUS_SHOW, BANNERS.FLAGS);
    private final List<Named> bannersPerformanceColumns = List.of(BANNERS_PERFORMANCE.STATUS_MODERATE);
    private final List<Named> bannerLogosColumns = List.of(BANNER_LOGOS.STATUS_MODERATE);
    private final List<Named> bannerButtonsColumns = List.of(BANNER_BUTTONS.STATUS_MODERATE);
    private final List<Named> bannerMulticardSetsColumns = List.of(BANNER_MULTICARD_SETS.STATUS_MODERATE);
    private final List<Named> perfCreativesColumns = List.of(PERF_CREATIVES.STATUS_MODERATE);
    private final List<Named> bannersMinusGeoColumns = List.of(BANNERS_MINUS_GEO.MINUS_GEO);
    private final List<Named> moderateBannerPagesColumns = List.of(MODERATE_BANNER_PAGES.STATUS_MODERATE,
            MODERATE_BANNER_PAGES.STATUS_MODERATE_OPERATOR, MODERATE_BANNER_PAGES.IS_REMOVED);
    private final List<Named> bannerDisplayHrefs = List.of(BANNER_DISPLAY_HREFS.STATUS_MODERATE);
    private final List<Named> bannerTurbolandings = List.of(BANNER_TURBOLANDINGS.STATUS_MODERATE);
    private final List<Named> bidsColumns = List.of(BIDS.STATUS_MODERATE, BIDS.IS_SUSPENDED);
    private final List<Named> bidsArcColumns = List.of(BIDS_ARC.STATUS_MODERATE, BIDS_ARC.IS_SUSPENDED);
    private final List<Named> bidsBaseColumns = List.of(BIDS_BASE.OPTS);
    private final List<Named> bidsDynamicColumns = List.of(BIDS_DYNAMIC.OPTS);
    private final List<Named> bidsRetargetingsColumns = List.of(BIDS_RETARGETING.IS_SUSPENDED);
    private final List<Named> bidsPerformanceColumns = List.of(BIDS_PERFORMANCE.IS_SUSPENDED,
            BIDS_PERFORMANCE.IS_DELETED);
    private final List<Named> phrasesColumns = List.of(PHRASES.STATUS_MODERATE, PHRASES.STATUS_MODERATE,
            PHRASES.STATUS_POST_MODERATE, PHRASES.IS_BS_RARELY_LOADED, PHRASES.STATUS_AUTOBUDGET_SHOW,
            PHRASES.STATUS_SHOWS_FORECAST);
    private final List<Named> adgroupPerformanceColumns = List.of(ADGROUPS_PERFORMANCE.STATUS_BL_GENERATED); // ловить удаление фида
    private final List<Named> adgroupDynamicColumns = List.of(ADGROUPS_DYNAMIC.STATUS_BL_GENERATED);

    private final List<Named> campaignsColumns = List.of(
            CAMPAIGNS.ARCHIVED,
            CAMPAIGNS.CONTEXT_LIMIT,
            CAMPAIGNS.CURRENCY,
            CAMPAIGNS.CURRENCY_CONVERTED,
            CAMPAIGNS.DAY_BUDGET,
            CAMPAIGNS.DAY_BUDGET_SHOW_MODE,
            CAMPAIGNS.FINISH_TIME,
            CAMPAIGNS.OPTS,
            CAMPAIGNS.ORDER_ID,
            CAMPAIGNS.PLATFORM,
            CAMPAIGNS.SHOWS,
            CAMPAIGNS.START_TIME,
            CAMPAIGNS.STATUS_ACTIVE,
            CAMPAIGNS.STATUS_BS_SYNCED,
            CAMPAIGNS.STATUS_EMPTY,
            CAMPAIGNS.STATUS_NO_PAY,
            CAMPAIGNS.STATUS_SHOW,
            CAMPAIGNS.SUM,
            CAMPAIGNS.SUM_LAST,
            CAMPAIGNS.SUM_SPENT,
            CAMPAIGNS.SUM_TO_PAY,
            CAMPAIGNS.SUM_UNITS,
            CAMPAIGNS.SUM_SPENT_UNITS,
            CAMPAIGNS.STRATEGY_DATA,
            CAMPAIGNS.TIMEZONE_ID,
            CAMPAIGNS.TIME_TARGET);
    private final List<Named> campOptionsColumns = List.of(CAMP_OPTIONS.STATUS_METRICA_CONTROL,
            CAMP_OPTIONS.STATUS_POST_MODERATE, CAMP_OPTIONS.STATUS_CLICK_TRACK, CAMP_OPTIONS.DAY_BUDGET_STOP_TIME,
            CAMP_OPTIONS.DAY_BUDGET_DAILY_CHANGE_COUNT, CAMP_OPTIONS.STOP_TIME);
    private final List<Named> campaignsCpmPriceColumns = List.of(CAMPAIGNS_CPM_PRICE.STATUS_CORRECT,
            CAMPAIGNS_CPM_PRICE.STATUS_APPROVE);
    private final List<Named> campOperationsQueue = List.of(CAMP_OPERATIONS_QUEUE.OPERATION);
    private final List<Named> calloutsColumns = Collections.singletonList(ADDITIONS_ITEM_CALLOUTS.LAST_MODERATION_STATUS);
    private final List<Named> imagesColumns = List.of(IMAGES.STATUS_MODERATE);
    private final List<Named> bannerImagesColumns = List.of(BANNER_IMAGES.STATUS_MODERATE, BANNER_IMAGES.STATUS_SHOW);
    private final List<Named> promoactionsColumns = List.of(PROMOACTIONS.STATUS_MODERATE);
    private final List<Named> campaignPromoactionsColumns = List.of(CAMPAIGN_PROMOACTIONS.PROMOACTION_ID);

    public AggregatedStatusesRule() {
        tableChangesHandler.addGrutTableChange(
                new GrutTableChange.Builder<AggregatedStatusEventObject>()
                        .setObjectType(EObjectType.OT_BANNER_CANDIDATE)
                        .setWatchlogMapper(AggregatedStatusesRule::mapGrutWatchlogChangeToObject)
                        .build());

        addAnyOpRuleToHandler(tableChangesHandler, BANNERS,
                AggregatedStatusesRule::mapBannerChangeToObject, bannersColumns);
        addAnyOpRuleToHandler(tableChangesHandler, BANNERS_PERFORMANCE,
                AggregatedStatusesRule::mapBannerChangeToObject, bannersPerformanceColumns);
        addAnyOpRuleToHandler(tableChangesHandler, BANNER_LOGOS,
                AggregatedStatusesRule::mapBannerChangeToObject, bannerLogosColumns);
        addAnyOpRuleToHandler(tableChangesHandler, BANNER_BUTTONS,
                AggregatedStatusesRule::mapBannerChangeToObject, bannerButtonsColumns);
        addAnyOpRuleToHandler(tableChangesHandler, BANNER_MULTICARD_SETS,
                AggregatedStatusesRule::mapBannerChangeToObject, bannerMulticardSetsColumns);
        addAnyOpRuleToHandler(tableChangesHandler, PERF_CREATIVES,
                AggregatedStatusesRule::mapPerfCreativesChangeToObject, perfCreativesColumns);
        addAnyOpRuleToHandler(tableChangesHandler, MODERATE_BANNER_PAGES,
                AggregatedStatusesRule::mapModerateBannerPagesChangeToObject, moderateBannerPagesColumns);
        addAnyOpRuleToHandler(tableChangesHandler, BANNER_DISPLAY_HREFS,
                AggregatedStatusesRule::mapBannerDisplayHrefsToObject, bannerDisplayHrefs);
        addAnyOpRuleToHandler(tableChangesHandler, BANNER_TURBOLANDINGS,
                AggregatedStatusesRule::mapBannerTurbolandingsChangeToObject, bannerTurbolandings);


        // Для фраз не обрабатываем DELETE т.к. это единственных объект для которого, в результате
        // архивации/разархивации после удаления может случится INSERT с тем же PRIMARY KEY
        addInsertUpdateOpRuleToHandler(tableChangesHandler, BIDS,
                AggregatedStatusesRule::mapKeywordsChangeToObject, bidsColumns);

        addAnyOpRuleToHandler(tableChangesHandler, BIDS_BASE,
                AggregatedStatusesRule::mapBidsBaseChangeToObject, bidsBaseColumns);
        addAnyOpRuleToHandler(tableChangesHandler, BIDS_RETARGETING,
                AggregatedStatusesRule::mapBidsRetargetingChangeToObject, bidsRetargetingsColumns);

        // не имеют собственного статуса, но влияют на статус группы
        addAnyOpRuleToHandler(tableChangesHandler, BIDS_DYNAMIC,
                AggregatedStatusesRule::mapBidsDynamicChangeToObject, bidsDynamicColumns);
        addAnyOpRuleToHandler(tableChangesHandler, BIDS_PERFORMANCE,
                AggregatedStatusesRule::mapBidsPerformanceChangeToObject, bidsPerformanceColumns);

        addAnyOpRuleToHandler(tableChangesHandler, PHRASES,
                AggregatedStatusesRule::mapAdGroupsChangeToObject, phrasesColumns);


        addAnyOpRuleToHandler(tableChangesHandler, CAMPAIGNS,
                AggregatedStatusesRule::mapCampaignsChangeToObject, campaignsColumns);
        addAnyOpRuleToHandler(tableChangesHandler, CAMP_OPTIONS,
                AggregatedStatusesRule::mapCampaignsChangeToObject, campOptionsColumns);
        addAnyOpRuleToHandler(tableChangesHandler, CAMP_OPERATIONS_QUEUE,
                AggregatedStatusesRule::mapCampaignOperationsQueueChangeToObject, campOperationsQueue);
        addAnyOpRuleToHandler(tableChangesHandler, CAMPAIGNS_CPM_PRICE,
                AggregatedStatusesRule::mapCampaignsChangeToObject, campaignsCpmPriceColumns);


        addInsertRuleToHandler(tableChangesHandler, BANNERS_ADDITIONS,
                AggregatedStatusesRule::mapBannerAdditionChangeToObject);
        addDeleteRuleToHandler(tableChangesHandler, BANNERS_ADDITIONS,
                AggregatedStatusesRule::mapBannerAdditionChangeToObject);

        addUpdateRuleToHandler(tableChangesHandler, ADDITIONS_ITEM_CALLOUTS,
                AggregatedStatusesRule::mapCalloutsChangeToObject, calloutsColumns);

        addAnyOpRuleToHandler(tableChangesHandler, IMAGES,
                AggregatedStatusesRule::mapImageChangeToObject, imagesColumns);

        addAnyOpRuleToHandler(tableChangesHandler, BANNER_IMAGES,
                AggregatedStatusesRule::mapBannerImagesChangeToObject, bannerImagesColumns);

        addUpdateRuleToHandler(tableChangesHandler, ADGROUPS_PERFORMANCE,
                AggregatedStatusesRule::mapAdGroupsPerformanceChangeToObject, adgroupPerformanceColumns);
        addUpdateRuleToHandler(tableChangesHandler, ADGROUPS_DYNAMIC,
                AggregatedStatusesRule::mapAdGroupsDynamicChangeToObject, adgroupDynamicColumns);

        addAnyOpRuleToHandler(tableChangesHandler, BANNERS_MINUS_GEO,
                AggregatedStatusesRule::mapBannersMinusGeoChangeToObject, bannersMinusGeoColumns);

        addUpdateRuleToHandler(tableChangesHandler, PROMOACTIONS,
                AggregatedStatusesRule::mapPromoactionsChangeToObject, promoactionsColumns);
        addAnyOpRuleToHandler(tableChangesHandler, CAMPAIGN_PROMOACTIONS,
                AggregatedStatusesRule::mapCampaignPromoactionsChangeToObject, campaignPromoactionsColumns);
    }


    private static void addInsertUpdateOpRuleToHandler(TableChangesHandler<AggregatedStatusEventObject> tableChangesHandler,
                                              TableImpl<?> table,
                                              Function<ProceededChange, AggregatedStatusEventObject> mapper,
                                              List<Named> columns) {

        addInsertRuleToHandler(tableChangesHandler, table, mapper);
        addUpdateRuleToHandler(tableChangesHandler, table, mapper, columns);
    }

    private static void addAnyOpRuleToHandler(TableChangesHandler<AggregatedStatusEventObject> tableChangesHandler,
                                              TableImpl<?> table,
                                              Function<ProceededChange, AggregatedStatusEventObject> mapper,
                                              List<Named> columns) {
        addInsertUpdateOpRuleToHandler(tableChangesHandler, table, mapper, columns);
        addDeleteRuleToHandler(tableChangesHandler, table, mapper);
    }

    private static void addUpdateRuleToHandler(TableChangesHandler<AggregatedStatusEventObject> tableChangesHandler,
                                               TableImpl<?> table,
                                               Function<ProceededChange, AggregatedStatusEventObject> mapper,
                                               List<Named> columns) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<AggregatedStatusEventObject>()
                        .setTable(table)
                        .setOperation(UPDATE)
                        .setColumns(ANY, columns)
                        .setMapper(mapper)
                        .build());
    }

    private static void addInsertRuleToHandler(TableChangesHandler<AggregatedStatusEventObject> tableChangesHandler,
                                               TableImpl<?> table, Function<ProceededChange,
            AggregatedStatusEventObject> mapper) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<AggregatedStatusEventObject>()
                        .setTable(table)
                        .setOperation(INSERT)
                        .setMapper(mapper)
                        .build());
    }

    private static void addDeleteRuleToHandler(TableChangesHandler<AggregatedStatusEventObject> tableChangesHandler,
                                               TableImpl<?> table,
                                               Function<ProceededChange, AggregatedStatusEventObject> mapper) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<AggregatedStatusEventObject>()
                        .setTable(table)
                        .setOperation(DELETE)
                        .setMapper(mapper)
                        .build());
    }

    private static AggregatedStatusEventObject mapGrutWatchlogChangeToObject(TEssEvent event) {
        TBannerCandidateMeta bannerMeta;
        TEvent internalEvent = event.getEvent();
        try {
            bannerMeta = TBannerCandidateMeta.parseFrom(internalEvent.getObjectMeta());
        } catch (InvalidProtocolBufferException ex) {
            logger.error("Failed to batchProcessing protobuf", ex);
            throw new IllegalStateException(ex);
        }

        logger.info("Parsed AggregatedStatuses event from watchlog event with id {}, adgroup id {}, campaign id {}, event type {}",
                bannerMeta.getDirectId(), bannerMeta.getAdGroupId(), bannerMeta.getCampaignId(), internalEvent.getEventType());

        return AggregatedStatusEventObject.aggregateStatusOnBannerChange(
                bannerMeta.getDirectId(),
                bannerMeta.getAdGroupId(),
                bannerMeta.getCampaignId(),
                internalEvent.getEventType() == ET_OBJECT_REMOVED,
                true);
    }

    //поле BANNERS.CID есть не всегда, но пока что нам без разницы
    private static AggregatedStatusEventObject mapBannerChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnBannerChange(extractId(change, BANNERS.BID),
                        extractField(change, BANNERS.PID), extractField(change, BANNERS.CID), change.getOperation() == DELETE);
        logger.debug("ProceededChange to adChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapPerfCreativesChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnPerfCreativesChange(extractId(change,
                        PERF_CREATIVES.CREATIVE_ID));
        logger.debug("ProceededChange to perfCreativesChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapModerateBannerPagesChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject = AggregatedStatusEventObject
                .aggregateStatusOnModerateBannerPagesChange(extractField(change, MODERATE_BANNER_PAGES.BID));
        logger.debug("ProceededChange to adChangeObject by moderate_banner_pages table: {} to {}", change,
                aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapBannerTurbolandingsChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject = AggregatedStatusEventObject
                .aggregateStatusOnBannerTurbolandingsChange(extractId(change, BANNER_TURBOLANDINGS.BID));
        logger.debug("ProceededChange to adChangeObject by banner_turbolandings table: {} to {}", change,
                aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapBannerDisplayHrefsToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject = AggregatedStatusEventObject
                .aggregateStatusOnBannerDisplayHrefsChange(extractId(change, BANNER_DISPLAY_HREFS.BID));
        logger.debug("ProceededChange to adChangeObject by banner_display_hrefs table: {} to {}", change,
                aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapBannerAdditionChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnBannerAdditionChange(extractId(change, BANNERS.BID));
        logger.debug("ProceededChange to adChangeObject by banners_additions table: {} to {}",
                change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapBannersMinusGeoChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnBannersMinusGeoChange(
                        extractId(change, BANNERS_MINUS_GEO.BID));
        logger.debug("ProceededChange to adChangeObject by banners_minus_geo table: {} to {}",
                change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapCalloutsChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnCalloutChange(
                        extractId(change, ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID));
        logger.debug("ProceededChange to calloutChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapImageChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnImageChange(
                        extractField(change, IMAGES.BID));
        logger.debug("ProceededChange to adChangeObject by images table: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapBannerImagesChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnImageChange(
                        extractField(change, BANNER_IMAGES.BID));
        logger.debug("ProceededChange to adChangeObject by banner_images table: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapKeywordsChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnKeywordChange(extractId(change, BIDS.ID),
                        extractField(change, BIDS.PID), extractField(change, BIDS.CID),
                        change.getOperation() == DELETE);
        logger.debug("ProceededChange to keywordChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapKeywordsArcChangeToObject(ProceededChange change) {
        return AggregatedStatusEventObject.aggregateStatusOnArchivedKeywordChange(extractId(change, BIDS_ARC.ID),
                extractId(change, BIDS_ARC.PID), extractId(change, BIDS_ARC.CID), change.getOperation() == DELETE);
    }

    private static AggregatedStatusEventObject mapBidsBaseChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnBidsBaseChange(extractId(change, BIDS_BASE.BID_ID),
                        extractField(change, BIDS_BASE.PID));
        logger.debug("ProceededChange to bidsBaseChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapBidsDynamicChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject = AggregatedStatusEventObject
                .aggregateStatusOnShowConditionChange(extractField(change, BIDS_DYNAMIC.PID));
        logger.debug("ProceededChange to bidsDynamicChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapBidsPerformanceChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject = AggregatedStatusEventObject
                .aggregateStatusOnShowConditionChange(extractField(change, BIDS_PERFORMANCE.PID));
        logger.debug("ProceededChange to bidsPerformanceChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapBidsRetargetingChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject = AggregatedStatusEventObject
                .aggregateStatusOnRetargetingChange(extractId(change, BIDS_RETARGETING.RET_ID),
                        extractField(change, BIDS_RETARGETING.PID), change.getOperation() == DELETE);
        logger.debug("ProceededChange to bidsRetargetingChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapAdGroupsChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnAdGroupChange(extractId(change, PHRASES.PID),
                        extractField(change, PHRASES.CID), change.getOperation() == DELETE);
        logger.debug("ProceededChange to adgroupChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapAdGroupsPerformanceChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnAdGroupChange(extractId(change, PHRASES.PID), null, false);
        logger.debug("ProceededChange to AdGroupsPerformanceChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapAdGroupsDynamicChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnAdGroupChange(extractId(change, PHRASES.PID), null, false);
        logger.debug("ProceededChange to AdGroupsDynamicChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapCampaignsChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnCampaignChange(extractId(change, CAMPAIGNS.CID),
                        change.getOperation() == DELETE);
        logger.debug("ProceededChange to campaignChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    // deleted всегда false, т.к. мы не хотим удалять статус для кампании, когда у кампании удаляется
    // запись в camp_queue
    private static AggregatedStatusEventObject mapCampaignOperationsQueueChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnCampaignChange(extractId(change, CAMPAIGNS.CID),
                        false);
        logger.debug("ProceededChange to campaignOperationsQueueChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapPromoactionsChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnPromoactionChange(extractId(change, PROMOACTIONS.ID));
        logger.debug("ProceededChange to promoactionChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static AggregatedStatusEventObject mapCampaignPromoactionsChangeToObject(ProceededChange change) {
        AggregatedStatusEventObject aggregatedObject =
                AggregatedStatusEventObject.aggregateStatusOnCampaignChange(
                        extractId(change, CAMPAIGN_PROMOACTIONS.CID), false);
        logger.debug("ProceededChange to campaignPromoactionChangeObject: {} to {}", change, aggregatedObject);
        return aggregatedObject;
    }

    private static <T> T extractField(ProceededChange change, Field<T> field) {
        return change.getOperation().equals(DELETE) ? change.getBefore(field) : change.getAfter(field);
    }

    private static Long extractId(ProceededChange change, Field<Long> field) {
        Long pkeyId;
        if (change.getPrimaryKey(field) instanceof BigInteger) {
            // Первичный ключ в MySQL может быть bigint unsigned, его нужно явно приводить к long
            pkeyId = ((BigInteger) change.getPrimaryKey(field)).longValue();
        } else {
            pkeyId = change.getPrimaryKey(field);
        }
        // во второстепенной таблице id объекта может не содержаться в primary key (например bid в banners_performance)
        return pkeyId == null ? extractField(change, field) : pkeyId;
    }

    @Override
    public List<AggregatedStatusEventObject> mapBinlogEvent(BinlogEvent event) {
        return tableChangesHandler.processChanges(event);
    }

    @Override
    public List<AggregatedStatusEventObject> mapWatchlogEvent(TEssEvent event) {
        return tableChangesHandler.processGrutChanges(event);
    }
}
