package ru.yandex.direct.ess.router.rules.brandliftrecalc;

import java.util.List;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Named;

import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.ess.config.brandliftrecalc.BrandLiftRecalcConfig;
import ru.yandex.direct.ess.logicobjects.brandliftrecalc.BrandLiftRecalcObject;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.ess.router.models.rule.StandardRule;
import ru.yandex.direct.ess.router.utils.ColumnsChangeType;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;

import static ru.yandex.direct.binlog.model.Operation.DELETE;
import static ru.yandex.direct.binlog.model.Operation.INSERT;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;

@ParametersAreNonnullByDefault
@EssRule(BrandLiftRecalcConfig.class)
public class BrandLiftRecalcRule  extends StandardRule<BrandLiftRecalcObject> {

    private static final List<Named> whiteListCampaignFields = List.of(
            CAMPAIGNS.START_TIME,
            CAMPAIGNS.FINISH_TIME,
            CAMPAIGNS.STATUS_SHOW,
            CAMPAIGNS.ARCHIVED,
            CAMPAIGNS.AUTOBUDGET,
            CAMPAIGNS.STRATEGY_NAME,
            CAMPAIGNS.STRATEGY_DATA,
            CAMPAIGNS.DONT_SHOW,
            CAMPAIGNS.TIME_TARGET,
            CAMPAIGNS.DISABLED_IPS,
            CAMPAIGNS.DISABLED_SSP,
            CAMPAIGNS.DISABLED_VIDEO_PLACEMENTS,
            CAMPAIGNS.RF,
            CAMPAIGNS.RF_RESET,
            CAMPAIGNS.DAY_BUDGET);

    private static final List<Named> blackListAdGroupFields = List.of(
            PHRASES.GROUP_NAME,
            PHRASES.LAST_CHANGE,
            PHRASES.STATUS_BS_SYNCED,
            PHRASES.IS_BS_RARELY_LOADED,
            PHRASES.FORECAST_DATE
    );

    private static final List<Named> blackListBannerFields = List.of(
            BANNERS.GEOFLAG,
            BANNERS.BANNER_ID,
            BANNERS.LAST_CHANGE,
            BANNERS.STATUS_BS_SYNCED,
            BANNERS.FLAGS
    );

    private static final List<Named> whiteListMultipliersFields = List.of(
            HIERARCHICAL_MULTIPLIERS.LAST_CHANGE
    );

    public BrandLiftRecalcRule() {
        super(tableChangesHandler -> {
            // Кампании
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<BrandLiftRecalcObject>()
                            .setTable(CAMPAIGNS)
                            .setOperation(UPDATE)
                            .setValuesFilter(BrandLiftRecalcRule::isCampaignTypeAllowedForBL)
                            .setColumns(ColumnsChangeType.ANY, whiteListCampaignFields)
                            .setMapper(BrandLiftRecalcRule::mapCampaignChangeToObject)
                            .build());

            // Группы
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<BrandLiftRecalcObject>()
                            .setTable(PHRASES)
                            .setOperation(INSERT)
                            .setValuesFilter(BrandLiftRecalcRule::isAdGroupTypeAllowedForBL)
                            .setMapper(BrandLiftRecalcRule::mapPhrasesChangeToObject)
                            .build());
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<BrandLiftRecalcObject>()
                            .setTable(PHRASES)
                            .setOperation(DELETE)
                            .setValuesFilter(BrandLiftRecalcRule::isAdGroupTypeAllowedForBL)
                            .setMapper(BrandLiftRecalcRule::mapPhrasesChangeToObject)
                            .build());
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<BrandLiftRecalcObject>()
                            .setTable(PHRASES)
                            .setOperation(UPDATE)
                            .setValuesFilter(BrandLiftRecalcRule::isAdGroupTypeAllowedForBL)
                            .setColumns(ColumnsChangeType.ANY_EXCEPT, blackListAdGroupFields)
                            .setMapper(BrandLiftRecalcRule::mapPhrasesChangeToObject)
                            .build());
            // Объявления
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<BrandLiftRecalcObject>()
                            .setTable(BANNERS)
                            .setOperation(INSERT)
                            .setValuesFilter(BrandLiftRecalcRule::isBannerTypeAllowedForBL)
                            .setMapper(BrandLiftRecalcRule::mapBannerChangeToObject)
                            .build());
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<BrandLiftRecalcObject>()
                            .setTable(BANNERS)
                            .setOperation(DELETE)
                            .setValuesFilter(BrandLiftRecalcRule::isBannerTypeAllowedForBL)
                            .setMapper(BrandLiftRecalcRule::mapBannerChangeToObject)
                            .build());
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<BrandLiftRecalcObject>()
                            .setTable(BANNERS)
                            .setOperation(UPDATE)
                            .setValuesFilter(BrandLiftRecalcRule::isBannerTypeAllowedForBL)
                            .setColumns(ColumnsChangeType.ANY_EXCEPT, blackListBannerFields)
                            .setMapper(BrandLiftRecalcRule::mapBannerChangeToObject)
                            .build());

            // Корректировки
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<BrandLiftRecalcObject>()
                            .setTable(HIERARCHICAL_MULTIPLIERS)
                            .setOperation(INSERT)
                            .setColumns(ColumnsChangeType.ANY, whiteListMultipliersFields)
                            .setMapper(BrandLiftRecalcRule::mapMultipliersChangeToObject)
                            .build());

            tableChangesHandler.addTableChange(
                    new TableChange.Builder<BrandLiftRecalcObject>()
                            .setTable(HIERARCHICAL_MULTIPLIERS)
                            .setOperation(DELETE)
                            .setMapper(BrandLiftRecalcRule::mapMultipliersChangeToObject)
                            .build());

            tableChangesHandler.addTableChange(
                    new TableChange.Builder<BrandLiftRecalcObject>()
                            .setTable(HIERARCHICAL_MULTIPLIERS)
                            .setOperation(UPDATE)
                            .setColumns(ColumnsChangeType.ANY, whiteListMultipliersFields)
                            .setMapper(BrandLiftRecalcRule::mapMultipliersChangeToObject)
                            .build());
        });
    }

    private static BrandLiftRecalcObject mapCampaignChangeToObject(ProceededChange change) {
        return new BrandLiftRecalcObject(change.getPrimaryKey(CAMPAIGNS.CID));
    }

    private static BrandLiftRecalcObject mapPhrasesChangeToObject(ProceededChange change) {
        return new BrandLiftRecalcObject(change.getBeforeOrAfter(PHRASES.CID));
    }

    private static BrandLiftRecalcObject mapBannerChangeToObject(ProceededChange change) {
        return new BrandLiftRecalcObject(change.getBeforeOrAfter(BANNERS.CID));
    }

    private static BrandLiftRecalcObject mapMultipliersChangeToObject(ProceededChange change) {
        return new BrandLiftRecalcObject(change.getBeforeOrAfter(HIERARCHICAL_MULTIPLIERS.CID));
    }

    private static boolean isCampaignTypeAllowedForBL(ProceededChange change) {
        var actualTypes = Stream.of(CampaignsType.cpm_banner, CampaignsType.cpm_yndx_frontpage);
        return actualTypes.anyMatch(type -> type.getLiteral().equals(change.getAfter(CAMPAIGNS.TYPE)));
    }

    private static boolean isAdGroupTypeAllowedForBL(ProceededChange change) {
        var actualTypes = Stream.of(PhrasesAdgroupType.cpm_banner, PhrasesAdgroupType.cpm_video, PhrasesAdgroupType.cpm_audio,
                PhrasesAdgroupType.cpm_outdoor, PhrasesAdgroupType.cpm_indoor, PhrasesAdgroupType.cpm_geoproduct,
                PhrasesAdgroupType.cpm_yndx_frontpage);
        return actualTypes.anyMatch(type -> type.getLiteral().equals(change.getBeforeOrAfter(PHRASES.ADGROUP_TYPE)));
    }

    private static boolean isBannerTypeAllowedForBL(ProceededChange change) {
        var actualTypes = Stream.of(BannersBannerType.cpm_banner, BannersBannerType.cpm_audio,
                BannersBannerType.cpm_outdoor, BannersBannerType.cpm_indoor);
        return actualTypes.anyMatch(type -> type.getLiteral().equals(change.getBeforeOrAfter(BANNERS.BANNER_TYPE)));
    }

}
