package ru.yandex.direct.ess.router.rules.bsexport.adgroup.filter

import org.jooq.Field
import org.jooq.Named
import org.jooq.Table
import org.springframework.stereotype.Component
import ru.yandex.direct.binlog.model.Operation
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_INTERNAL
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_MOBILE_CONTENT
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_ADDITIONAL_TARGETINGS
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_BS_TAGS
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_PAGE_TARGETS
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS_MINUS_GEO
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.Tables.BANNER_PERMALINKS
import ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_RETARGETING
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_INTERNAL
import ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS
import ru.yandex.direct.dbschema.ppc.Tables.IMAGES
import ru.yandex.direct.dbschema.ppc.Tables.MOBILE_CONTENT
import ru.yandex.direct.dbschema.ppc.Tables.PHRASES
import ru.yandex.direct.dbschema.ppc.enums.AdgroupAdditionalTargetingsTargetingType
import ru.yandex.direct.dbschema.ppc.enums.BannerPermalinksPermalinkAssignType
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.AdGroupResourceType.SHOW_CONDITIONS
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.BsExportAdGroupObject
import ru.yandex.direct.ess.logicobjects.bsexport.adgroup.ShowConditionsInfo
import ru.yandex.direct.ess.router.rules.bsexport.extractDebugInfo
import ru.yandex.direct.ess.router.utils.ColumnsChangeType
import ru.yandex.direct.ess.router.utils.ProceededChange
import ru.yandex.direct.ess.router.utils.TableChange
import ru.yandex.direct.ess.router.utils.TableChangesHandler

@Component
class AdGroupShowConditionsFilter : AdGroupResourceFilter {

    private val filtersToConfigure = listOf(
        ::configureAdditionalTargetingFilter,
        ::configureGeoFilter,
        ::configureGoalContextFilter,
        ::configureInternalFilter,
        ::configureMobileContentFilter,
        ::configurePageBlocksFilter,
        ::configurePerformanceFilter,
        ::configurePlaceFilter,
        ::configureTargetTagsFilter,
    )

    override fun configure(tableChangesHandler: TableChangesHandler<BsExportAdGroupObject>) =
        filtersToConfigure.forEach { it(tableChangesHandler) }

    private fun configureAdditionalTargetingFilter(tableChangesHandler: TableChangesHandler<BsExportAdGroupObject>) {
        addAllTableChanges(
            tableChangesHandler, ADGROUP_ADDITIONAL_TARGETINGS,
            mapAdGroupToObject(ADGROUP_ADDITIONAL_TARGETINGS.PID, false),
            valuesFilter = filterUnknownTargetingType
        )

        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(MOBILE_CONTENT)
                .setOperation(Operation.UPDATE)
                .setColumns(
                    ColumnsChangeType.ANY, listOf(
                        MOBILE_CONTENT.OS_TYPE,
                        MOBILE_CONTENT.BUNDLE_ID,
                        MOBILE_CONTENT.STORE_CONTENT_ID,
                    )
                )
                .setMapper(mapMobileContentToObject(MOBILE_CONTENT.MOBILE_CONTENT_ID))
                .build()
        )
    }

    private fun configureGeoFilter(tableChangesHandler: TableChangesHandler<BsExportAdGroupObject>) {
        addAllTableChanges(
            tableChangesHandler, PHRASES,
            mapAdGroupToObject(PHRASES.PID, true),
            updateColumns = listOf(PHRASES.GEO, PHRASES.STATUS_POST_MODERATE)
        )
        addAllTableChanges(
            tableChangesHandler, BANNERS,
            mapAdGroupToObject(BANNERS.PID, false),
            updateColumns = listOf(
                BANNERS.STATUS_POST_MODERATE,
                BANNERS.STATUS_SHOW,
                BANNERS.STATUS_ARCH,
                BANNERS.HREF,
                BANNERS.PHONEFLAG,
                BANNERS.VCARD_ID,
            )
        )
        addAllTableChanges(
            tableChangesHandler, BANNERS_MINUS_GEO,
            mapBannerToObject(BANNERS_MINUS_GEO.BID, true),
            updateColumns = listOf(BANNERS_MINUS_GEO.MINUS_GEO)
        )
        addAllTableChanges(
            tableChangesHandler, IMAGES,
            mapBannerToObject(IMAGES.BID, false),
            updateColumns = listOf(IMAGES.STATUS_MODERATE)
        )
        addAllTableChanges(
            tableChangesHandler, BANNERS_PERFORMANCE,
            mapBannerToObject(BANNERS_PERFORMANCE.BID, false),
            updateColumns = listOf(BANNERS_PERFORMANCE.STATUS_MODERATE)
        )
        addAllTableChanges(
            tableChangesHandler, BANNER_TURBOLANDINGS,
            mapBannerToObject(BANNER_TURBOLANDINGS.BID, true),
            updateColumns = listOf(BANNER_TURBOLANDINGS.STATUS_MODERATE)
        )

        addAllTableChanges(
            tableChangesHandler, BANNER_PERMALINKS,
            mapBannerToObject(BANNER_PERMALINKS.BID, true),
            // в коде handler'а интересуют только manual пермалинки
            // фильтровать по полю assign_type может только в INSERT, но этого достаточно,
            // так как уже отсечем довольно много событий
            { proceededChange ->
                proceededChange.operation != Operation.INSERT ||
                    proceededChange.getAfter<String, BannerPermalinksPermalinkAssignType>(
                        BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE
                    ) == BannerPermalinksPermalinkAssignType.manual.literal
            },
            updateColumns = listOf(
                BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE,
                BANNER_PERMALINKS.PREFER_VCARD_OVER_PERMALINK,
            )
        )
    }

    private fun configureGoalContextFilter(tableChangesHandler: TableChangesHandler<BsExportAdGroupObject>) {
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(BIDS_RETARGETING)
                .setOperation(Operation.INSERT)
                .setMapper(mapAdGroupToObject(BIDS_RETARGETING.PID, false))
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(BIDS_RETARGETING)
                .setOperation(Operation.DELETE)
                .setMapper(mapAdGroupToObject(BIDS_RETARGETING.PID, false))
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(ADGROUPS_INTERNAL)
                .setOperation(Operation.DELETE)
                .setMapper(mapAdGroupToObject(ADGROUPS_INTERNAL.PID, true))
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(PHRASES)
                .setOperation(Operation.DELETE)
                .setMapper(mapAdGroupToObject(PHRASES.PID, true))
                .build()
        )
    }

    private fun configureInternalFilter(tableChangesHandler: TableChangesHandler<BsExportAdGroupObject>) {
        addAllTableChanges(
            tableChangesHandler, ADGROUPS_INTERNAL,
            mapAdGroupToObject(ADGROUPS_INTERNAL.PID, true),
            updateColumns = listOf(
                ADGROUPS_INTERNAL.RF,
                ADGROUPS_INTERNAL.RF_RESET,
                ADGROUPS_INTERNAL.START_TIME,
                ADGROUPS_INTERNAL.FINISH_TIME
            )
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(PHRASES)
                .setOperation(Operation.DELETE)
                .setMapper(mapAdGroupToObject(PHRASES.PID, true))
                .build()
        )
    }

    private fun configureMobileContentFilter(tableChangesHandler: TableChangesHandler<BsExportAdGroupObject>) {
        addAllTableChanges(
            tableChangesHandler, ADGROUPS_MOBILE_CONTENT,
            mapAdGroupToObject(ADGROUPS_MOBILE_CONTENT.PID, true),
            updateColumns = listOf(
                ADGROUPS_MOBILE_CONTENT.MIN_OS_VERSION,
                ADGROUPS_MOBILE_CONTENT.NETWORK_TARGETING,
                ADGROUPS_MOBILE_CONTENT.DEVICE_TYPE_TARGETING,
            )
        )

        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(MOBILE_CONTENT)
                .setOperation(Operation.UPDATE)
                .setColumns(
                    ColumnsChangeType.ANY, listOf(
                        MOBILE_CONTENT.OS_TYPE,
                        MOBILE_CONTENT.BUNDLE_ID,
                        MOBILE_CONTENT.STORE_CONTENT_ID,
                        MOBILE_CONTENT.MIN_OS_VERSION,
                    )
                )
                .setMapper(mapMobileContentToObject(MOBILE_CONTENT.MOBILE_CONTENT_ID))
                .build()
        )
    }

    private fun configurePageBlocksFilter(tableChangesHandler: TableChangesHandler<BsExportAdGroupObject>) =
        addAllTableChanges(
            tableChangesHandler, ADGROUP_PAGE_TARGETS,
            mapAdGroupToObject(ADGROUP_PAGE_TARGETS.PID, true),
            updateColumns = listOf(ADGROUP_PAGE_TARGETS.PAGE_BLOCKS)
        )

    private fun configurePerformanceFilter(tableChangesHandler: TableChangesHandler<BsExportAdGroupObject>) =
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(PHRASES)
                .setOperation(Operation.INSERT)
                .setValuesFilter { change: ProceededChange ->
                    val adGroupType: String = change.getBeforeOrAfter(PHRASES.ADGROUP_TYPE)
                    adGroupType == PhrasesAdgroupType.performance.literal
                }.setMapper(mapAdGroupToObject(PHRASES.PID, true))
                .build()
        )

    private fun configurePlaceFilter(tableChangesHandler: TableChangesHandler<BsExportAdGroupObject>) {
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(CAMPAIGNS_INTERNAL)
                .setOperation(Operation.UPDATE)
                .setColumn(CAMPAIGNS_INTERNAL.PLACE_ID)
                .setMapper(mapCampaignToObject)
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(PHRASES)
                .setOperation(Operation.INSERT)
                .setMapper(mapAdGroupToObject(PHRASES.PID, true))
                .build()
        )
    }

    private fun configureTargetTagsFilter(tableChangesHandler: TableChangesHandler<BsExportAdGroupObject>) {
        addAllTableChanges(
            tableChangesHandler, ADGROUP_BS_TAGS,
            mapAdGroupToObject(ADGROUP_BS_TAGS.PID, true),
            updateColumns = listOf(ADGROUP_BS_TAGS.TARGET_TAGS_JSON)
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(PHRASES)
                .setOperation(Operation.INSERT)
                .setMapper(mapAdGroupToObject(PHRASES.PID, true))
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>()
                .setTable(CAMP_OPTIONS)
                .setColumn(CAMP_OPTIONS.PLACEMENT_TYPES)
                .setOperation(Operation.UPDATE)
                .setMapper(mapCampaignToObject)
                .build()
        )
    }

    private fun addAllTableChanges(
        handler: TableChangesHandler<BsExportAdGroupObject>,
        table: Table<*>,
        binlogMapper: ((ProceededChange) -> BsExportAdGroupObject),
        valuesFilter: ((ProceededChange) -> Boolean)? = null,
        updateColumns: List<Named>? = null
    ) {
        handler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>().apply {
                setTable(table)
                setOperation(Operation.INSERT)
                if (valuesFilter != null)
                    setValuesFilter(valuesFilter)
                setMapper(binlogMapper)
            }.build()
        )
        handler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>().apply {
                setTable(table)
                setOperation(Operation.UPDATE)
                if (valuesFilter != null)
                    setValuesFilter(valuesFilter)
                if (updateColumns != null)
                    setColumns(ColumnsChangeType.ANY, updateColumns)
                setMapper(binlogMapper)
            }.build()
        )
        handler.addTableChange(
            TableChange.Builder<BsExportAdGroupObject>().apply {
                setTable(table)
                setOperation(Operation.DELETE)
                if (valuesFilter != null)
                    setValuesFilter(valuesFilter)
                setMapper(binlogMapper)
            }.build()
        )
    }

    companion object {
        private val knownTargetingTypes = AdgroupAdditionalTargetingsTargetingType.values().map { it.name }.toSet()

        val filterUnknownTargetingType =
            { change: ProceededChange ->
                val targetingTypeStr: String =
                    change.getBeforeOrAfter(ADGROUP_ADDITIONAL_TARGETINGS.TARGETING_TYPE)
                knownTargetingTypes.contains(targetingTypeStr)
            }
    }

    private fun mapAdGroupToObject(
        pid: Field<Long>,
        isPrimary: Boolean
    ): ((ProceededChange) -> BsExportAdGroupObject) =
        { change: ProceededChange ->
            BsExportAdGroupObject(
                resourceType = SHOW_CONDITIONS,
                adGroupId = if (isPrimary) change.getLongPrimaryKey(pid) else change.getLongBeforeOrAfter(pid),
                campaignId = null,
                debugInfo = extractDebugInfo(change)
            )
        }

    private val mapCampaignToObject =
        { change: ProceededChange ->
            BsExportAdGroupObject(
                resourceType = SHOW_CONDITIONS,
                adGroupId = null,
                campaignId = change.getLongPrimaryKey(CAMPAIGNS_INTERNAL.CID),
                debugInfo = extractDebugInfo(change)
            )
        }

    private fun mapMobileContentToObject(
        mobContId: Field<Long>
    ): ((ProceededChange) -> BsExportAdGroupObject) =
        { change: ProceededChange ->
            BsExportAdGroupObject(
                resourceType = SHOW_CONDITIONS,
                adGroupId = null,
                campaignId = null,
                debugInfo = extractDebugInfo(change),
                additionalInfo = ShowConditionsInfo(
                    mobileContentId = change.getLongPrimaryKey(mobContId)
                )
            )
        }

    private fun mapBannerToObject(
        bannerId: Field<Long>,
        isPrimary: Boolean
    ): ((ProceededChange) -> BsExportAdGroupObject) =
        { change: ProceededChange ->
            BsExportAdGroupObject(
                resourceType = SHOW_CONDITIONS,
                adGroupId = null,
                campaignId = null,
                debugInfo = extractDebugInfo(change),
                additionalInfo = ShowConditionsInfo(
                    bannerId = if (isPrimary) change.getLongPrimaryKey(bannerId) else change.getLongBeforeOrAfter(
                        bannerId
                    )
                )
            )
        }
}
