package ru.yandex.direct.ess.router.rules.bsexport.bids;

import java.math.BigInteger;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import org.jooq.Named;
import org.jooq.TableField;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType;
import ru.yandex.direct.ess.config.bsexport.bids.BsExportBiddableShowConditionsConfig;
import ru.yandex.direct.ess.logicobjects.bsexport.bids.BidObjectType;
import ru.yandex.direct.ess.logicobjects.bsexport.bids.BsExportBidsObject;
import ru.yandex.direct.ess.router.models.rule.AbstractRule;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.binlog.model.Operation.DELETE;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_BASE;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_HREF_PARAMS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_RETARGETING;
import static ru.yandex.direct.dbschema.ppc.Tables.DYNAMIC_CONDITIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_CONDITIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_GOALS;
import static ru.yandex.direct.dbschema.ppc.tables.Bids.BIDS;
import static ru.yandex.direct.ess.router.utils.TableChangesHandlerUtilKt.addRulesToHandler;

/**
 * BS/ExportWorker.pm см get_prices_for_update %sqls
 */
@EssRule(BsExportBiddableShowConditionsConfig.class)
public class BsExportBiddableShowConditionsRule extends AbstractRule<BsExportBidsObject> {
    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(BsExportBiddableShowConditionsRule.class);

    private final TableChangesHandler<BsExportBidsObject> tableChangesHandler = new TableChangesHandler<>();

    private static final List<Named> BIDS_COLUMNS = List.of(
            BIDS.PRICE,
            BIDS.PRICE_CONTEXT,
            BIDS.STATUS_MODERATE,
            BIDS.IS_SUSPENDED,
            BIDS.SHOWS_FORECAST,
            BIDS.PHRASE);

    // для поля HrefParams
    private static final List<Named> BIDS_HREF_PARAMS_COLUMNS = List.of(
            BIDS_HREF_PARAMS.PARAM1,
            BIDS_HREF_PARAMS.PARAM2
    );

    private static final List<Named> BIDS_RETARGETING_COLUMNS = List.of(
            BIDS_RETARGETING.RET_COND_ID,
            BIDS_RETARGETING.PRICE_CONTEXT,
            BIDS_RETARGETING.IS_SUSPENDED
    );

    // для поля HrefParams
    private static final List<Named> RETARGETING_CONDITION_COLUMNS = List.of(
            RETARGETING_CONDITIONS.CONDITION_JSON,
            RETARGETING_CONDITIONS.PROPERTIES
    );

    private static final List<Named> RETARGETING_GOAL_COLUMNS = List.of(
            RETARGETING_GOALS.IS_ACCESSIBLE
    );

    private static final List<Named> BIDS_DYNAMIC_COLUMNS = List.of(
            BIDS_DYNAMIC.DYN_COND_ID,
            BIDS_DYNAMIC.PRICE,
            BIDS_DYNAMIC.PRICE_CONTEXT,
            BIDS_DYNAMIC.AUTOBUDGET_PRIORITY,
            BIDS_DYNAMIC.OPTS
    );

    // Для поля HrefParams
    private static final List<Named> DYNAMIC_CONDITION_COLUMNS = List.of(
            DYNAMIC_CONDITIONS.CONDITION_NAME,
            DYNAMIC_CONDITIONS.CONDITION_JSON
    );

    private static final List<Named> BIDS_PERFORMANCE_COLUMNS = List.of(
            BIDS_PERFORMANCE.PRICE_CPA,
            BIDS_PERFORMANCE.PRICE_CPC,
            BIDS_PERFORMANCE.TARGET_FUNNEL,
            BIDS_PERFORMANCE.IS_SUSPENDED,
            BIDS_PERFORMANCE.IS_DELETED,
            // ставки могут задаваться на кампанию, а отправлять мы их должны на фильтр
            // если на фильтре ставки нет, а утсновлена в стратегии - Директ сбоасывает statusBsSynced
            // по-хорошему, стоит использовать другое поле, со временем сделаем
            BIDS_PERFORMANCE.STATUS_BS_SYNCED,
            // для поля HrefParams
            BIDS_PERFORMANCE.NAME
    );

    private static final List<Named> BIDS_BASE_COLUMNS = List.of(
            BIDS_BASE.PRICE,
            BIDS_BASE.PRICE_CONTEXT,
            BIDS_BASE.OPTS,
            BIDS_BASE.RELEVANCE_MATCH_CATEGORIES
    );

    static final Map<String, BidObjectType> BID_TYPE_BY_BIDS_BASE_TYPE = Map.of(
            BidsBaseBidType.relevance_match.getLiteral(), BidObjectType.RELEVANCE_MATCH,
            BidsBaseBidType.offer_retargeting.getLiteral(), BidObjectType.OFFER_RETARGETING
    );

    public BsExportBiddableShowConditionsRule() {
        // bids
        addRulesToHandler(tableChangesHandler, BIDS, BsExportBiddableShowConditionsRule::mapBidsChangeObject,
                BIDS_COLUMNS);

        // bids_retargeting
        addRulesToHandler(tableChangesHandler, BIDS_RETARGETING,
                BsExportBiddableShowConditionsRule::mapBidsRetargetingChangeObject,
                BIDS_RETARGETING_COLUMNS);
        addRulesToHandler(tableChangesHandler, RETARGETING_CONDITIONS,
                BsExportBiddableShowConditionsRule::mapRetargetingConditionChangeObject,
                RETARGETING_CONDITION_COLUMNS);
        addRulesToHandler(tableChangesHandler, RETARGETING_GOALS,
                BsExportBiddableShowConditionsRule::mapRetargetingGoalChangeObject,
                RETARGETING_GOAL_COLUMNS);

        // bids_dynamic
        addRulesToHandler(tableChangesHandler, BIDS_DYNAMIC,
                BsExportBiddableShowConditionsRule::mapBidsDynamicChangeObject,
                BIDS_DYNAMIC_COLUMNS);
        addRulesToHandler(tableChangesHandler, DYNAMIC_CONDITIONS,
                BsExportBiddableShowConditionsRule::mapDynamicConditionChangeObject,
                DYNAMIC_CONDITION_COLUMNS);

        // bids_performance
        addRulesToHandler(tableChangesHandler, BIDS_PERFORMANCE,
                BsExportBiddableShowConditionsRule::mapBidsPerformanceChangeObject,
                BIDS_PERFORMANCE_COLUMNS);

        // relevance_match || offer_retargeting
        addRulesToHandler(tableChangesHandler, BIDS_BASE,
                BsExportBiddableShowConditionsRule::mapBidsBaseChangeObject,
                BIDS_BASE_COLUMNS,
                this::filterByBidsType);

        // изменения bids_href_params для BIDS и BIDS_BASE
        addRulesToHandler(tableChangesHandler, BIDS_HREF_PARAMS,
                BsExportBiddableShowConditionsRule::mapBidsHrefParamsChangeObject,
                BIDS_HREF_PARAMS_COLUMNS);
    }

    private boolean filterByBidsType(ProceededChange proceededChange) {
        String bidsBaseType = proceededChange.getBeforeOrAfter(BIDS_BASE.BID_TYPE);
        if (!BID_TYPE_BY_BIDS_BASE_TYPE.containsKey(bidsBaseType)) {
            logger.error("Unsupported bid_base type: {}", bidsBaseType);
        }
        return BID_TYPE_BY_BIDS_BASE_TYPE.containsKey(bidsBaseType);
    }

    @Override
    public List<BsExportBidsObject> mapBinlogEvent(BinlogEvent binlogEvent) {
        return tableChangesHandler.processChanges(binlogEvent);
    }

    private static BsExportBidsObject mapBidsChangeObject(ProceededChange change) {
        BsExportBidsObject bsExportBidsObject = createBidObject(
                change,
                BIDS.CID,
                BIDS.PID,
                BIDS.ID,
                BidObjectType.KEYWORD);
        logger.debug("ProceededChange to bids change object: {} to {}", change, bsExportBidsObject);
        return bsExportBidsObject;
    }

    private static BsExportBidsObject mapBidsRetargetingChangeObject(ProceededChange change) {
        BsExportBidsObject bsExportBidsObject = createBidObject(
                change,
                BIDS_RETARGETING.CID,
                BIDS_RETARGETING.PID,
                BIDS_RETARGETING.RET_ID,
                BidObjectType.RETARGETING);
        logger.debug("ProceededChange to bids_retargeting change object: {} to {}", change, bsExportBidsObject);
        return bsExportBidsObject;
    }

    private static BsExportBidsObject mapRetargetingConditionChangeObject(ProceededChange change) {
        BsExportBidsObject bsExportBidsObject = createBidObject(
                change,
                null,
                null,
                RETARGETING_CONDITIONS.RET_COND_ID,
                BidObjectType.RETARGETING_CONDITION);
        logger.debug("ProceededChange to retargeting_condition change object: {} to {}", change, bsExportBidsObject);
        return bsExportBidsObject;
    }

    private static BsExportBidsObject mapRetargetingGoalChangeObject(ProceededChange change) {
        BsExportBidsObject bsExportBidsObject = createBidObject(
                change,
                null,
                null,
                RETARGETING_GOALS.RET_COND_ID,
                BidObjectType.RETARGETING_CONDITION);
        logger.debug("ProceededChange to retargeting_condition change object: {} to {}", change, bsExportBidsObject);
        return bsExportBidsObject;
    }

    private static BsExportBidsObject mapBidsDynamicChangeObject(ProceededChange change) {
        BsExportBidsObject bsExportBidsObject = createBidObject(
                change,
                null,
                BIDS_DYNAMIC.PID,
                BIDS_DYNAMIC.DYN_ID,
                BidObjectType.DYNAMIC);
        logger.debug("ProceededChange to bids_dynamic change object: {} to {}", change, bsExportBidsObject);
        return bsExportBidsObject;
    }

    private static BsExportBidsObject mapDynamicConditionChangeObject(ProceededChange change) {
        BsExportBidsObject bsExportBidsObject = createBidObject(
                change,
                null,
                null,
                DYNAMIC_CONDITIONS.DYN_COND_ID,
                BidObjectType.DYNAMIC_CONDITION);
        logger.debug("ProceededChange to dynamic_condition change object: {} to {}", change, bsExportBidsObject);
        return bsExportBidsObject;
    }

    private static BsExportBidsObject mapBidsPerformanceChangeObject(ProceededChange change) {
        BsExportBidsObject bsExportBidsObject = createBidObject(
                change,
                null,
                BIDS_PERFORMANCE.PID,
                BIDS_PERFORMANCE.PERF_FILTER_ID,
                BidObjectType.PERFORMANCE);
        logger.debug("ProceededChange to bids_performance change object: {} to {}", change, bsExportBidsObject);
        return bsExportBidsObject;
    }

    static BsExportBidsObject mapBidsBaseChangeObject(ProceededChange change) {
        String bidType = change.getBeforeOrAfter(BIDS_BASE.BID_TYPE);
        BsExportBidsObject bsExportBidsObject = createBidObject(
                change,
                BIDS_BASE.CID,
                BIDS_BASE.PID,
                BIDS_BASE.BID_ID,
                BID_TYPE_BY_BIDS_BASE_TYPE.get(bidType));
        logger.debug("ProceededChange to bids_base change object: {} to {}", change, bsExportBidsObject);
        return bsExportBidsObject;
    }

    private static BsExportBidsObject mapBidsHrefParamsChangeObject(ProceededChange change) {
        BsExportBidsObject bsExportBidsObject = new BsExportBidsObject(
                getPrimaryLong(change, BIDS_HREF_PARAMS.CID),
                null,
                getPrimaryLong(change, BIDS_HREF_PARAMS.ID),
                BidObjectType.BIDS_HREF_PARAMS,
                false);
        fillDebugInfo(change, bsExportBidsObject);

        logger.debug("ProceededChange to bids_href_params change object: {} to {}", change, bsExportBidsObject);
        return bsExportBidsObject;
    }

    static BsExportBidsObject createBidObject(
            ProceededChange change,
            @Nullable TableField<?, Long> cid,
            @Nullable TableField<?, Long> pid,
            TableField<?, Long> id,
            BidObjectType type
    ) {
        logger.debug("createBidObject from change: {}", change);
        BsExportBidsObject bsExportBidsObject = new BsExportBidsObject(
                getFieldLong(change, cid),
                getFieldLong(change, pid),
                getPrimaryLong(change, id),
                type,
                DELETE.equals(change.getOperation()));
        fillDebugInfo(change, bsExportBidsObject);

        return bsExportBidsObject;
    }

    private static Long getPrimaryLong(ProceededChange change, TableField<?, Long> field) {
        return convertIdToLong(change.getPrimaryKey(field));
    }

    @Nullable
    private static Long getFieldLong(ProceededChange change, @Nullable TableField<?, Long> field) {
        if (field == null) {
            return null;
        }
        return convertIdToLong(change.getBeforeOrAfter(field));
    }

    private static long convertIdToLong(Object value) {
        if (value instanceof BigInteger) {
            // Первичный ключ в MySQL может быть bigint unsigned, его нужно явно приводить к long
            return ((BigInteger) value).longValue();
        } else if (value instanceof Long) {
            return (Long) value;
        } else {
            throw new IllegalStateException("Unsuppored type of id: " + value.getClass());
        }
    }

    private static void fillDebugInfo(ProceededChange change, BsExportBidsObject bsExportBidsObject) {
        bsExportBidsObject.setReqid(change.getReqId());
        bsExportBidsObject.setService(change.getService());
        bsExportBidsObject.setMethod(change.getMethod());
    }
}
