package ru.yandex.direct.ess.router.rules.bsexport.campaign.filter;

import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.core.entity.campaign.model.CampaignOpts;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.tables.Campaigns;
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.BsExportCampaignObject;
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.CampaignResourceType;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;

@Component
@ParametersAreNonnullByDefault
public class CampaignAllowedOnAdultContentFilter extends BaseCampaignFilter {
    private static final Set<String> ALLOWED_CAMPAIGNS_TYPES = Set.of(
            CampaignsType.text.getLiteral(),
            CampaignsType.cpm_banner.getLiteral(),
            CampaignsType.cpm_price.getLiteral(),
            CampaignsType.mobile_content.getLiteral());

    @Override
    public void init(TableChangesHandler<BsExportCampaignObject> tableChangesHandler) {
        tableChangesHandler.addTableChange(new TableChange.Builder<BsExportCampaignObject>()
                .setTable(CAMPAIGNS)
                .setOperation(Operation.INSERT)
                .setValuesFilter(CampaignAllowedOnAdultContentFilter::isAllowedOnAdultContentChanged)
                .setMapper(this::mapToCampaignObject)
                .build());

        tableChangesHandler.addTableChange(new TableChange.Builder<BsExportCampaignObject>()
                .setTable(CAMPAIGNS)
                .setOperation(Operation.UPDATE)
                .setColumn(CAMPAIGNS.OPTS)
                .setValuesFilter(CampaignAllowedOnAdultContentFilter::isAllowedOnAdultContentChanged)
                .setMapper(this::mapToCampaignObject)
                .build());
    }

    @Override
    public CampaignResourceType campaignResourceType() {
        return CampaignResourceType.CAMPAIGN_ALLOWED_ON_ADULT_CONTENT;
    }

    private BsExportCampaignObject mapToCampaignObject(ProceededChange proceededChange) {
        var objectBuilder = new BsExportCampaignObject.Builder()
                .setCid(proceededChange.getPrimaryKey(CAMPAIGNS.CID));
        setSystemFields(proceededChange, objectBuilder);
        return objectBuilder.build();
    }

    private static boolean isAllowedOnAdultContentChanged(ProceededChange change) {
        String campaignType = change.getBeforeOrAfter(Campaigns.CAMPAIGNS.TYPE);
        boolean campaignTypeIsAllowed = campaignType != null && ALLOWED_CAMPAIGNS_TYPES.contains(campaignType);

        boolean before = isAllowedOnAdultContent(change.getBefore(Campaigns.CAMPAIGNS.OPTS));
        boolean after = isAllowedOnAdultContent(change.getAfter(Campaigns.CAMPAIGNS.OPTS));

        return campaignTypeIsAllowed && before != after;
    }

    private static boolean isAllowedOnAdultContent(@Nullable String opts) {
        return opts != null && opts.contains(CampaignOpts.IS_ALLOWED_ON_ADULT_CONTENT.getTypedValue());
    }
}
