package ru.yandex.direct.ess.router.rules.bsexport.campaign.filter;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.BsExportCampaignObject;
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.CampaignResourceType;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_INTERNAL;
import static ru.yandex.direct.ess.router.utils.ColumnsChangeType.ANY;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Component
@ParametersAreNonnullByDefault
public class CampaignRfOptionsFilter extends BaseCampaignFilter {

    @SuppressWarnings("ConstantConditions")
    private static final Set<String> ALLOWED_CAMPAIGN_TYPES =
            mapSet(CampaignTypeKinds.INTERNAL, type -> CampaignType.toSource(type).getLiteral());

    @Override
    public void init(TableChangesHandler<BsExportCampaignObject> tableChangesHandler) {
        tableChangesHandler.addTableChange(new TableChange.Builder<BsExportCampaignObject>()
                .setTable(CAMPAIGNS)
                .setOperation(Operation.INSERT)
                .setValuesFilter(this::isSuitableCampaignsWithRfOptions)
                .setMapper(this::mapToInternalCampaignObject)
                .build());

        tableChangesHandler.addTableChange(new TableChange.Builder<BsExportCampaignObject>()
                .setTable(CAMPAIGNS_INTERNAL)
                .setOperation(Operation.INSERT)
                .setValuesFilter(this::isSuitableCampaignsInternalWithRfOptions)
                .setMapper(this::mapToInternalCampaignObject)
                .build());

        tableChangesHandler.addTableChange(new TableChange.Builder<BsExportCampaignObject>()
                .setTable(CAMPAIGNS)
                .setOperation(Operation.UPDATE)
                .setColumns(ANY, List.of(CAMPAIGNS.RF, CAMPAIGNS.RF_RESET))
                // для UPDATE получаем изменения только при изменении столбцов выше, поэтому тут достаточно в фильтре
                // проверить тип кампании
                .setValuesFilter(this::isSuitableCampaignsType)
                .setMapper(this::mapToInternalCampaignObject)
                .build());

        tableChangesHandler.addTableChange(new TableChange.Builder<BsExportCampaignObject>()
                .setTable(CAMPAIGNS_INTERNAL)
                .setOperation(Operation.UPDATE)
                .setColumns(ANY, List.of(
                                CAMPAIGNS_INTERNAL.MAX_CLICKS_COUNT,
                                CAMPAIGNS_INTERNAL.MAX_CLICKS_PERIOD,
                                CAMPAIGNS_INTERNAL.MAX_STOPS_COUNT,
                                CAMPAIGNS_INTERNAL.MAX_STOPS_PERIOD
                        )
                )
                .setMapper(this::mapToInternalCampaignObject)
                .build());
    }

    /**
     * Проверка подходящего типа кампании
     *
     * @see #ALLOWED_CAMPAIGN_TYPES
     */
    boolean isSuitableCampaignsType(ProceededChange proceededChange) {
        String type = proceededChange.getBeforeOrAfter(CAMPAIGNS.TYPE);
        return ALLOWED_CAMPAIGN_TYPES.contains(type);
    }

    /**
     * Проверка подходящего типа кампании и наличия на нем RF'а
     */
    boolean isSuitableCampaignsWithRfOptions(ProceededChange proceededChange) {
        Long rf = proceededChange.getBeforeOrAfter(CAMPAIGNS.RF);
        return isSuitableCampaignsType(proceededChange)
                && rf != null && rf > 0;
    }

    /**
     * Проверка наличия RF'а на полях внутренней кампании
     */
    boolean isSuitableCampaignsInternalWithRfOptions(ProceededChange proceededChange) {
        Long maxClicksCount = proceededChange.getBeforeOrAfter(CAMPAIGNS_INTERNAL.MAX_CLICKS_COUNT);
        Long maxClicksPeriod = proceededChange.getBeforeOrAfter(CAMPAIGNS_INTERNAL.MAX_CLICKS_PERIOD);
        Long maxStopsCount = proceededChange.getBeforeOrAfter(CAMPAIGNS_INTERNAL.MAX_STOPS_COUNT);
        Long maxStopsPeriod = proceededChange.getBeforeOrAfter(CAMPAIGNS_INTERNAL.MAX_STOPS_PERIOD);
        return maxClicksCount != null || maxClicksPeriod != null || maxStopsCount != null || maxStopsPeriod != null;
    }

    @Override
    public CampaignResourceType campaignResourceType() {
        return CampaignResourceType.CAMPAIGN_RF_OPTIONS;
    }

    private BsExportCampaignObject mapToInternalCampaignObject(ProceededChange proceededChange) {
        var objectBuilder = new BsExportCampaignObject.Builder()
                .setCid(proceededChange.getPrimaryKey(CAMPAIGNS.CID));
        setSystemFields(proceededChange, objectBuilder);
        return objectBuilder.build();
    }
}
