package ru.yandex.direct.ess.router.rules.bsexport.campaign.filter;

import java.util.List;

import org.jooq.Field;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.BsExportCampaignObject;
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.CampaignResourceType;
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.MultiplierInfo;
import ru.yandex.direct.ess.logicobjects.bsexport.campaing.RetargetingConditionInfo;
import ru.yandex.direct.ess.router.rules.bsexport.multipliers.MultiplierCommonFilter;
import ru.yandex.direct.ess.router.utils.ColumnsChangeType;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_GOALS;
import static ru.yandex.direct.ess.router.rules.bsexport.multipliers.MultiplierCommonFilter.getLongField;

@Component
public class MultipliersFilter extends BaseCampaignFilter {
    @Override
    public CampaignResourceType campaignResourceType() {
        return CampaignResourceType.MULTIPLIERS;
    }

    @Override
    public void init(TableChangesHandler<BsExportCampaignObject> tableChangesHandler) {
        MultiplierCommonFilter.registerCommonChangeHandlers(tableChangesHandler, this::checkPid,
                this::mapCid, this::mapHierarchicalMultiplierId, this::mapRetargetingConditionId);

        // Корректировка на timeTarget хранится на кампании
        registerTimeTargetChangeHandlers(tableChangesHandler);
    }

    private void registerTimeTargetChangeHandlers(TableChangesHandler<BsExportCampaignObject> tableChangesHandler) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportCampaignObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(Operation.INSERT)
                        .setMapper(this::mapCidFromTimeTargetChange)
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportCampaignObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(Operation.UPDATE)
                        // для автобюджета не нужно применять коэффициенты временного таргетинга
                        .setColumns(ColumnsChangeType.ANY, List.of(CAMPAIGNS.AUTOBUDGET, CAMPAIGNS.TIME_TARGET))
                        .setMapper(this::mapCidFromTimeTargetChange)
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportCampaignObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(Operation.DELETE)
                        .setMapper(this::mapCidFromTimeTargetChange)
                        .build()
        );
    }

    private boolean checkPid(ProceededChange proceededChange) {
        Long adGroupId = proceededChange.getBeforeOrAfter(HIERARCHICAL_MULTIPLIERS.PID);
        return adGroupId == null;
    }

    private BsExportCampaignObject mapCid(ProceededChange proceededChange) {
        var objectBuilder = new BsExportCampaignObject.Builder()
                .setCid(proceededChange.getBeforeOrAfter(HIERARCHICAL_MULTIPLIERS.CID));
        setSystemFields(proceededChange, objectBuilder);
        return objectBuilder.build();
    }

    private BsExportCampaignObject mapHierarchicalMultiplierId(
            ProceededChange proceededChange, Field<Long> field) {
        var multiplierId = getLongField(proceededChange, field);
        var objectBuilder = new BsExportCampaignObject.Builder()
                .setHierarchicalMultiplierId(multiplierId)
                .setAdditionalInfo(new MultiplierInfo(multiplierId));
        setSystemFields(proceededChange, objectBuilder);
        return objectBuilder.build();
    }

    private BsExportCampaignObject mapRetargetingConditionId(ProceededChange proceededChange) {
        Long retargetingId = proceededChange.getPrimaryKey(RETARGETING_GOALS.RET_COND_ID);
        var objectBuilder = new BsExportCampaignObject.Builder()
                .setRetargetingConditionId(retargetingId)
                .setAdditionalInfo(new RetargetingConditionInfo(retargetingId));
        setSystemFields(proceededChange, objectBuilder);
        return objectBuilder.build();
    }

    private BsExportCampaignObject mapCidFromTimeTargetChange(ProceededChange proceededChange) {
        var objectBuilder = new BsExportCampaignObject.Builder()
                .setCid(proceededChange.getPrimaryKey(CAMPAIGNS.CID));
        setSystemFields(proceededChange, objectBuilder);
        return objectBuilder.build();
    }
}
