package ru.yandex.direct.ess.router.rules.bsexport.feeds;

import java.math.BigInteger;
import java.util.List;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.dbschema.ppc.enums.FeedsUpdateStatus;
import ru.yandex.direct.ess.config.bsexport.feeds.BsExportFeedsConfig;
import ru.yandex.direct.ess.logicobjects.bsexport.feeds.BsExportFeedsObject;
import ru.yandex.direct.ess.router.models.rule.AbstractRule;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.binlog.model.Operation.DELETE;
import static ru.yandex.direct.binlog.model.Operation.INSERT;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.FEEDS;

@EssRule(BsExportFeedsConfig.class)
public class BsExportFeedsRule extends AbstractRule<BsExportFeedsObject> {

    private final TableChangesHandler<BsExportFeedsObject> tableChangesHandler;

    public BsExportFeedsRule() {
        tableChangesHandler = new TableChangesHandler<>();
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportFeedsObject>()
                        .setTable(FEEDS)
                        .setOperation(UPDATE)
                        .setColumn(FEEDS.UPDATE_STATUS)
                        .setValuesFilter(this::isFeedReady)
                        .setMapper(this::mapToObject)
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportFeedsObject>()
                        .setTable(FEEDS)
                        .setOperation(INSERT)
                        .setValuesFilter(this::isFeedReady)
                        .setMapper(this::mapToObject)
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportFeedsObject>()
                        .setTable(FEEDS)
                        .setOperation(DELETE)
                        .setMapper(this::mapToDeletedObject)
                        .build()
        );
    }

    @Override
    public List<BsExportFeedsObject> mapBinlogEvent(BinlogEvent binlogEvent) {
        return tableChangesHandler.processChanges(binlogEvent);
    }

    private boolean isFeedReady(ProceededChange proceededChange) {
        return proceededChange.getAfter(FEEDS.UPDATE_STATUS).equals(FeedsUpdateStatus.Done.getLiteral());
    }

    private BsExportFeedsObject mapToObject(ProceededChange proceededChange) {
        return mapToObjectInternal(proceededChange);
    }

    private BsExportFeedsObject mapToDeletedObject(ProceededChange proceededChange) {
        var object = mapToObjectInternal(proceededChange);
        object.setDeleted(true);
        return object;
    }

    private BsExportFeedsObject mapToObjectInternal(ProceededChange proceededChange) {
        BigInteger feedId = proceededChange.getPrimaryKey(FEEDS.FEED_ID);
        var object = new BsExportFeedsObject(feedId.longValue());
        object.setReqid(proceededChange.getReqId());
        object.setService(proceededChange.getService());
        object.setMethod(proceededChange.getMethod());
        return object;
    }

}
