package ru.yandex.direct.ess.router.rules.bsexport.multipliers;

import java.math.BigInteger;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Table;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.ess.config.bsexport.multipliers.BsExportMultipliersConfig;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.AccessibleGoalChangedInfo;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.BsExportMultipliersObject;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.DeleteInfo;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.MultiplierType;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.TimeTargetChangedInfo;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.UpsertInfo;
import ru.yandex.direct.ess.router.models.rule.AbstractRule;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.ess.router.utils.ColumnsChangeType;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.dbschema.ppc.Tables.AB_SEGMENT_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TYPE_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.DEMOGRAPHY_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.GEO_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.Tables.INVENTORY_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_GOALS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.WEATHER_MULTIPLIER_VALUES;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Правила для экспорта корректировок в БК
 */
@EssRule(BsExportMultipliersConfig.class)
public class BsExportMultipliersRule extends AbstractRule<BsExportMultipliersObject> {
    public static final Map<String, MultiplierType> SUPPORTED_TYPES = StreamEx.of(MultiplierType.values())
            .mapToEntry(MultiplierType::getDbTypes, Function.identity())
            .flatMapKeys(Collection::stream)
            .mapKeys(HierarchicalMultipliersType::getLiteral)
            .toMap();
    private final TableChangesHandler<BsExportMultipliersObject> tableChangesHandler;

    public BsExportMultipliersRule() {
        tableChangesHandler = new TableChangesHandler<>();
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportMultipliersObject>()
                        .setTable(HIERARCHICAL_MULTIPLIERS)
                        .setOperation(Operation.INSERT)
                        .setValuesFilter(this::checkSupportedType)
                        .setMapper(this::mapToObject)
                        .build()
        );

        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportMultipliersObject>()
                        .setTable(HIERARCHICAL_MULTIPLIERS)
                        .setOperation(Operation.UPDATE)
                        .setValuesFilter(this::checkSupportedType)
                        .setMapper(this::mapToObject)
                        .build()
        );

        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportMultipliersObject>()
                        .setTable(HIERARCHICAL_MULTIPLIERS)
                        .setOperation(Operation.DELETE)
                        .setValuesFilter(this::checkSupportedType)
                        .setMapper(this::mapToObject)
                        .build()
        );

        registerTableChangeHandler(MultiplierType.DEMOGRAPHY,
                DEMOGRAPHY_MULTIPLIER_VALUES, DEMOGRAPHY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(MultiplierType.RETARGETING,
                RETARGETING_MULTIPLIER_VALUES, RETARGETING_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(MultiplierType.WEATHER,
                WEATHER_MULTIPLIER_VALUES, WEATHER_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(MultiplierType.GEO,
                GEO_MULTIPLIER_VALUES, GEO_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(MultiplierType.DEVICE,
                MOBILE_MULTIPLIER_VALUES, MOBILE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(MultiplierType.INVENTORY,
                BANNER_TYPE_MULTIPLIER_VALUES, BANNER_TYPE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(MultiplierType.INVENTORY,
                INVENTORY_MULTIPLIER_VALUES, INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);

        // Для выгрузки корректировок на АБ-сегменты используется поле retargeting_conditions.condition_json
        // До, кажется, для этой корректоровки это поле никто не изменяет, поэтому роутер не следит за ним
        registerTableChangeHandler(MultiplierType.AB_SEGMENT,
                AB_SEGMENT_MULTIPLIER_VALUES, AB_SEGMENT_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);

        // Изменения в retargeting_goals.is_accessible для целей, входящих в условие ретаргетинга
        // могут повлиять на выгрузку корректировок на условие ретаргетинга и на АБ-сегменты
        registerAccessibleGoalChangeHandlers();

        // Корректировка на timeTarget хранится на кампании
        registerTimeTargetChangeHandlers();
    }

    private void registerTableChangeHandler(MultiplierType multiplierType, Table<?> table,
                                            Field<Long> hierarchicalMultiplierIdField) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportMultipliersObject>()
                        .setTable(table)
                        .setOperation(Operation.INSERT)
                        .setValuesFilter(change -> checkChangeHasField(change, hierarchicalMultiplierIdField))
                        .setMapper(change -> mapToUpsertInfo(change, multiplierType, hierarchicalMultiplierIdField))
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportMultipliersObject>()
                        .setTable(table)
                        .setOperation(Operation.UPDATE)
                        .setValuesFilter(change -> checkChangeHasField(change, hierarchicalMultiplierIdField))
                        .setMapper(change -> mapToUpsertInfo(change, multiplierType, hierarchicalMultiplierIdField))
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportMultipliersObject>()
                        .setTable(table)
                        .setOperation(Operation.DELETE)
                        .setValuesFilter(change -> checkChangeHasField(change, hierarchicalMultiplierIdField))
                        .setMapper(change -> mapToUpsertInfo(change, multiplierType, hierarchicalMultiplierIdField))
                        .build()
        );
    }

    private void registerAccessibleGoalChangeHandlers() {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportMultipliersObject>()
                        .setTable(RETARGETING_GOALS)
                        .setOperation(Operation.UPDATE)
                        .setColumn(RETARGETING_GOALS.IS_ACCESSIBLE)
                        .setMapper(this::mapToAccessibleGoalChangedInfo)
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportMultipliersObject>()
                        .setTable(RETARGETING_GOALS)
                        .setOperation(Operation.DELETE)
                        .setMapper(this::mapToAccessibleGoalChangedInfo)
                        .build()
        );
    }

    private void registerTimeTargetChangeHandlers() {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportMultipliersObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(Operation.INSERT)
                        .setMapper(this::mapToTimeTargetChangeInfo)
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportMultipliersObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(Operation.UPDATE)
                        // для автобюджета не нужно применять коэффициенты временного таргетинга
                        .setColumns(ColumnsChangeType.ANY, List.of(CAMPAIGNS.AUTOBUDGET, CAMPAIGNS.TIME_TARGET))
                        .setMapper(this::mapToTimeTargetChangeInfo)
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportMultipliersObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(Operation.DELETE)
                        .setMapper(this::mapToTimeTargetChangeInfo)
                        .build()
        );
    }

    private boolean checkSupportedType(ProceededChange proceededChange) {
        String beforeOrAfter = proceededChange.getBeforeOrAfter(HIERARCHICAL_MULTIPLIERS.TYPE);
        return SUPPORTED_TYPES.containsKey(beforeOrAfter);
    }

    private boolean checkChangeHasField(ProceededChange proceededChange, Field<?> field) {
        return proceededChange.beforeOrAfterContains(field);
    }

    private BsExportMultipliersObject mapToObject(ProceededChange proceededChange) {
        String dbTypeLiteral = proceededChange.getBeforeOrAfter(HIERARCHICAL_MULTIPLIERS.TYPE);
        MultiplierType multiplierType = SUPPORTED_TYPES.get(dbTypeLiteral);

        Long reqid = nvl(proceededChange.getReqId(), 0L);
        String service = nvl(proceededChange.getService(), "");
        String method = nvl(proceededChange.getMethod(), "");

        if (proceededChange.getOperation().equals(Operation.DELETE)) {
            DeleteInfo deleteInfo = new DeleteInfo(
                    multiplierType,
                    proceededChange.getBeforeOrAfter(HIERARCHICAL_MULTIPLIERS.CID),
                    proceededChange.getBeforeOrAfter(HIERARCHICAL_MULTIPLIERS.PID)
            );
            return BsExportMultipliersObject.delete(deleteInfo, reqid, service, method);
        } else {
            Long hierarchicalMultiplierId = getHierarchicalMultiplierIdPk(
                    proceededChange);

            UpsertInfo upsertInfo = new UpsertInfo(multiplierType, hierarchicalMultiplierId);
            return BsExportMultipliersObject.upsert(upsertInfo, reqid, service, method);
        }
    }

    private BsExportMultipliersObject mapToUpsertInfo(
            ProceededChange proceededChange, MultiplierType multiplierType, Field<Long> field) {
        Long reqid = nvl(proceededChange.getReqId(), 0L);
        String service = nvl(proceededChange.getService(), "");
        String method = nvl(proceededChange.getMethod(), "");

        Long hierarchicalMultiplierId = getLongField(proceededChange, field);

        UpsertInfo upsertInfo = new UpsertInfo(multiplierType, hierarchicalMultiplierId);
        return BsExportMultipliersObject.upsert(upsertInfo, reqid, service, method);
    }

    private BsExportMultipliersObject mapToAccessibleGoalChangedInfo(ProceededChange proceededChange) {
        Long reqid = nvl(proceededChange.getReqId(), 0L);
        String service = nvl(proceededChange.getService(), "");
        String method = nvl(proceededChange.getMethod(), "");
        AccessibleGoalChangedInfo accessibleGoalChangedInfo =
                new AccessibleGoalChangedInfo(proceededChange.getPrimaryKey(RETARGETING_GOALS.RET_COND_ID));
        return BsExportMultipliersObject.accessibleGoalChanged(accessibleGoalChangedInfo, reqid, service, method);
    }

    private BsExportMultipliersObject mapToTimeTargetChangeInfo(ProceededChange proceededChange) {
        Long reqid = nvl(proceededChange.getReqId(), 0L);
        String service = nvl(proceededChange.getService(), "");
        String method = nvl(proceededChange.getMethod(), "");
        TimeTargetChangedInfo timeTargetChangedInfo =
                new TimeTargetChangedInfo(proceededChange.getPrimaryKey(CAMPAIGNS.CID));
        return BsExportMultipliersObject.timeTargetChanged(timeTargetChangedInfo, reqid, service, method);
    }

    @Override
    public List<BsExportMultipliersObject> mapBinlogEvent(BinlogEvent binlogEvent) {
        return tableChangesHandler.processChanges(binlogEvent);
    }

    private Long getHierarchicalMultiplierIdPk(ProceededChange proceededChange) {
        if (proceededChange.getPrimaryKey(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID) instanceof BigInteger) {
            return ((BigInteger) proceededChange.getPrimaryKey(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)).longValue();
        } else {
            return proceededChange.getPrimaryKey(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
        }
    }

    private Long getLongField(ProceededChange proceededChange, Field<Long> field) {
        if (proceededChange.getBeforeOrAfter(field) instanceof BigInteger) {
            return ((BigInteger) proceededChange.getBeforeOrAfter(field)).longValue();
        } else {
            return proceededChange.getBeforeOrAfter(field);
        }
    }
}
