package ru.yandex.direct.ess.router.rules.bsexport.multipliers;

import java.math.BigInteger;
import java.util.Collection;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Table;

import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.ess.common.models.BaseLogicObject;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.MultiplierType;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.dbschema.ppc.Tables.AB_SEGMENT_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TYPE_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.DEMOGRAPHY_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.GEO_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.Tables.INVENTORY_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_GOALS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.WEATHER_MULTIPLIER_VALUES;

/**
 * Общая логика для корректировок, отправляемых вместе с группами/кампаниями
 */
@ParametersAreNonnullByDefault
public class MultiplierCommonFilter {
    public static final Map<String, MultiplierType> SUPPORTED_TYPES = StreamEx.of(MultiplierType.values())
            .mapToEntry(MultiplierType::getDbTypes, Function.identity())
            .flatMapKeys(Collection::stream)
            .mapKeys(HierarchicalMultipliersType::getLiteral)
            .toMap();

    public static <T extends BaseLogicObject> void registerCommonChangeHandlers(
            TableChangesHandler<T> tableChangesHandler,
            Predicate<ProceededChange> checkPid,
            Function<ProceededChange, T> mapCidPid,
            BiFunction<ProceededChange, Field<Long>, T> mapHierarchicalMultiplierId,
            Function<ProceededChange, T> mapRetargetingConditionId) {

        tableChangesHandler.addTableChange(
                new TableChange.Builder<T>()
                        .setTable(HIERARCHICAL_MULTIPLIERS)
                        .setOperation(Operation.INSERT)
                        .setValuesFilter(change -> checkSupportedType(change) && checkPid.test(change))
                        .setMapper(mapCidPid)
                        .build()
        );

        tableChangesHandler.addTableChange(
                new TableChange.Builder<T>()
                        .setTable(HIERARCHICAL_MULTIPLIERS)
                        .setOperation(Operation.UPDATE)
                        .setValuesFilter(change -> checkSupportedType(change) && checkPid.test(change))
                        .setMapper(mapCidPid)
                        .build()
        );

        tableChangesHandler.addTableChange(
                new TableChange.Builder<T>()
                        .setTable(HIERARCHICAL_MULTIPLIERS)
                        .setOperation(Operation.DELETE)
                        .setValuesFilter(change -> checkSupportedType(change) && checkPid.test(change))
                        .setMapper(mapCidPid)
                        .build()
        );

        registerTableChangeHandler(tableChangesHandler, mapHierarchicalMultiplierId,
                DEMOGRAPHY_MULTIPLIER_VALUES, DEMOGRAPHY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(tableChangesHandler, mapHierarchicalMultiplierId,
                RETARGETING_MULTIPLIER_VALUES, RETARGETING_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(tableChangesHandler, mapHierarchicalMultiplierId,
                WEATHER_MULTIPLIER_VALUES, WEATHER_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(tableChangesHandler, mapHierarchicalMultiplierId,
                GEO_MULTIPLIER_VALUES, GEO_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(tableChangesHandler, mapHierarchicalMultiplierId,
                MOBILE_MULTIPLIER_VALUES, MOBILE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(tableChangesHandler, mapHierarchicalMultiplierId,
                BANNER_TYPE_MULTIPLIER_VALUES, BANNER_TYPE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
        registerTableChangeHandler(tableChangesHandler, mapHierarchicalMultiplierId,
                INVENTORY_MULTIPLIER_VALUES, INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);

        // Для транспорта корректировок на АБ-сегменты используется поле retargeting_conditions.condition_json
        // Но, кажется, для этой корректоровки это поле никто не изменяет, поэтому роутер не следит за ним
        registerTableChangeHandler(tableChangesHandler, mapHierarchicalMultiplierId,
                AB_SEGMENT_MULTIPLIER_VALUES, AB_SEGMENT_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);


        // Изменения в retargeting_goals.is_accessible для целей, входящих в условие ретаргетинга
        // могут повлиять на выгрузку корректировок на условие ретаргетинга и на АБ-сегменты
        registerAccessibleGoalChangeHandlers(tableChangesHandler, mapRetargetingConditionId);
    }

    private static <T extends BaseLogicObject> void registerTableChangeHandler(
            TableChangesHandler<T> tableChangesHandler,
            BiFunction<ProceededChange, Field<Long>, T> mapHierarchicalMultiplierId,
            Table<?> table, Field<Long> hierarchicalMultiplierIdField) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<T>()
                        .setTable(table)
                        .setOperation(Operation.INSERT)
                        .setValuesFilter(change -> checkChangeHasField(change, hierarchicalMultiplierIdField))
                        .setMapper(change -> mapHierarchicalMultiplierId.apply(change, hierarchicalMultiplierIdField))
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<T>()
                        .setTable(table)
                        .setOperation(Operation.UPDATE)
                        .setValuesFilter(change -> checkChangeHasField(change, hierarchicalMultiplierIdField))
                        .setMapper(change -> mapHierarchicalMultiplierId.apply(change, hierarchicalMultiplierIdField))
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<T>()
                        .setTable(table)
                        .setOperation(Operation.DELETE)
                        .setValuesFilter(change -> checkChangeHasField(change, hierarchicalMultiplierIdField))
                        .setMapper(change -> mapHierarchicalMultiplierId.apply(change, hierarchicalMultiplierIdField))
                        .build()
        );
    }

    private static <T extends BaseLogicObject> void registerAccessibleGoalChangeHandlers(
            TableChangesHandler<T> tableChangesHandler,
            Function<ProceededChange, T> mapRetargetingConditionId) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<T>()
                        .setTable(RETARGETING_GOALS)
                        .setOperation(Operation.UPDATE)
                        .setColumn(RETARGETING_GOALS.IS_ACCESSIBLE)
                        .setMapper(mapRetargetingConditionId)
                        .build()
        );
        tableChangesHandler.addTableChange(
                new TableChange.Builder<T>()
                        .setTable(RETARGETING_GOALS)
                        .setOperation(Operation.DELETE)
                        .setMapper(mapRetargetingConditionId)
                        .build()
        );
    }

    private static boolean checkSupportedType(ProceededChange proceededChange) {
        String multiplierType = proceededChange.getBeforeOrAfter(HIERARCHICAL_MULTIPLIERS.TYPE);
        return SUPPORTED_TYPES.containsKey(multiplierType);
    }

    private static boolean checkChangeHasField(ProceededChange proceededChange, Field<Long> field) {
        return proceededChange.beforeOrAfterContains(field);
    }

    public static Long getLongField(ProceededChange proceededChange, Field<Long> field) {
        if (proceededChange.getBeforeOrAfter(field) instanceof BigInteger) {
            return ((BigInteger) proceededChange.getBeforeOrAfter(field)).longValue();
        } else {
            return proceededChange.getBeforeOrAfter(field);
        }
    }
}
