package ru.yandex.direct.ess.router.rules.bsexport.resources.filter

import org.springframework.stereotype.Component
import ru.yandex.direct.binlog.model.Operation
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate
import ru.yandex.direct.ess.common.utils.TablesEnum
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BannerResourceType
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BsExportBannerResourcesObject
import ru.yandex.direct.ess.logicobjects.bsexport.resources.DebugInfo
import ru.yandex.direct.ess.router.utils.ColumnsChangeType
import ru.yandex.direct.ess.router.utils.ProceededChange
import ru.yandex.direct.ess.router.utils.TableChange
import ru.yandex.direct.ess.router.utils.TableChangesHandler
import java.math.BigInteger

val SUPPORTED_BANNER_TYPES = setOf(
    BannersBannerType.text.literal,
    BannersBannerType.cpc_video.literal,
    BannersBannerType.image_ad.literal,
    BannersBannerType.mcbanner.literal,
    BannersBannerType.dynamic.literal,
    BannersBannerType.performance.literal,
    BannersBannerType.performance_main.literal,
)

@Component
open class BannerFeedInfoFilter : IBannerResourceFilter {
    fun bannerResourceType() = BannerResourceType.BANNER_FEED_INFO

    override fun init(tableChangesHandler: TableChangesHandler<BsExportBannerResourcesObject>) {
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.ADGROUPS_TEXT)
                .setOperation(Operation.INSERT)
                .setMapper(this::mapToAdGroupTextObject)
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.ADGROUPS_TEXT)
                .setOperation(Operation.UPDATE)
                .setColumn(Tables.ADGROUPS_TEXT.FEED_ID)
                .setMapper(this::mapToAdGroupTextObject)
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.ADGROUPS_TEXT)
                .setOperation(Operation.DELETE)
                .setMapper(this::mapToAdGroupTextObject)
                .build()
        )

        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.ADGROUPS_PERFORMANCE)
                .setOperation(Operation.INSERT)
                .setMapper(this::mapToAdGroupPerformanceObject)
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.ADGROUPS_PERFORMANCE)
                .setOperation(Operation.UPDATE)
                .setColumn(Tables.ADGROUPS_PERFORMANCE.FEED_ID)
                .setMapper(this::mapToAdGroupPerformanceObject)
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.ADGROUPS_PERFORMANCE)
                .setOperation(Operation.DELETE)
                .setMapper(this::mapToAdGroupPerformanceObject)
                .build()
        )

        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.ADGROUPS_DYNAMIC)
                .setOperation(Operation.INSERT)
                .setMapper(this::mapToAdGroupDynamicObject)
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.ADGROUPS_DYNAMIC)
                .setOperation(Operation.UPDATE)
                .setColumn(Tables.ADGROUPS_DYNAMIC.FEED_ID)
                .setMapper(this::mapToAdGroupDynamicObject)
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.ADGROUPS_DYNAMIC)
                .setOperation(Operation.DELETE)
                .setMapper(this::mapToAdGroupDynamicObject)
                .build()
        )

        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.FEEDS)
                .setOperation(Operation.UPDATE)
                // такое изменение может произойти только раз за время жизни фида, когда
                // фид регистрируется в MBI (маркетном api) и для него проставляются market_*_id
                // далее они не меняются
                .setColumns(
                    ColumnsChangeType.ANY, listOf(
                        Tables.FEEDS.MARKET_BUSINESS_ID, Tables.FEEDS.MARKET_SHOP_ID, Tables.FEEDS.MARKET_FEED_ID
                    )
                )
                .setMapper(this::mapToFeedObject)
                .build()
        )

        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.BANNERS)
                .setOperation(Operation.INSERT)
                .setValuesFilter(this::checkBanner)
                .setMapper(this::mapToBannerObject)
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<BsExportBannerResourcesObject>()
                .setTable(Tables.BANNERS)
                .setOperation(Operation.UPDATE)
                .setValuesFilter(this::checkBanner)
                .setMapper(this::mapToBannerObject)
                .build()
        )
    }

    private fun checkBanner(proceededChange: ProceededChange): Boolean {
        val bannerType = proceededChange.getAfter<String, BannersBannerType>(Tables.BANNERS.BANNER_TYPE)
        if (bannerType !in SUPPORTED_BANNER_TYPES) {
            return false
        }
        return !proceededChange.afterContains(Tables.BANNERS.STATUS_MODERATE)
            || proceededChange.getAfter<Any, BannersStatusmoderate>(Tables.BANNERS.STATUS_MODERATE) == BannersStatusmoderate.Yes.literal
    }

    private fun mapToFeedObject(change: ProceededChange): BsExportBannerResourcesObject {
        val feedId: BigInteger = change.getPrimaryKey(Tables.FEEDS.FEED_ID)
        val obj = BsExportBannerResourcesObject.Builder()
            .setAdditionalTable(TablesEnum.FEEDS)
            .setAdditionalId(feedId.toLong())
            .setResourceType(bannerResourceType())
            .build()
        addDebugInfo(obj, change)
        return obj
    }

    private fun mapToAdGroupTextObject(change: ProceededChange): BsExportBannerResourcesObject {
        val adGroupId: Long = change.getPrimaryKey(Tables.ADGROUPS_TEXT.PID)
        val obj = BsExportBannerResourcesObject.Builder()
            .setAdditionalTable(TablesEnum.ADGROUPS_TEXT)
            .setAdditionalId(adGroupId)
            .setResourceType(bannerResourceType())
            .build()
        addDebugInfo(obj, change)
        return obj
    }

    private fun mapToAdGroupPerformanceObject(change: ProceededChange): BsExportBannerResourcesObject {
        val adGroupId: Long = change.getPrimaryKey(Tables.ADGROUPS_PERFORMANCE.PID)
        val obj = BsExportBannerResourcesObject.Builder()
            .setAdditionalTable(TablesEnum.ADGROUPS_PERFORMANCE)
            .setAdditionalId(adGroupId)
            .setResourceType(bannerResourceType())
            .build()
        addDebugInfo(obj, change)
        return obj
    }

    private fun mapToAdGroupDynamicObject(change: ProceededChange): BsExportBannerResourcesObject {
        val adGroupId: Long = change.getPrimaryKey(Tables.ADGROUPS_DYNAMIC.PID)
        val obj = BsExportBannerResourcesObject.Builder()
            .setAdditionalTable(TablesEnum.ADGROUPS_DYNAMIC)
            .setAdditionalId(adGroupId)
            .setResourceType(bannerResourceType())
            .build()
        addDebugInfo(obj, change)
        return obj
    }

    private fun mapToBannerObject(change: ProceededChange): BsExportBannerResourcesObject {
        val obj = BsExportBannerResourcesObject.Builder()
            .setBid(change.getPrimaryKey(Tables.BANNERS.BID))
            .setResourceType(bannerResourceType())
            .build()
        addDebugInfo(obj, change)
        return obj
    }

    private fun addDebugInfo(obj: BsExportBannerResourcesObject, change: ProceededChange) {
        val debugInfo = DebugInfo(change.reqId, change.service, change.method)
        obj.setDebugInfo(debugInfo)
    }
}
