package ru.yandex.direct.ess.router.rules.bsexport.resources.filter;

import java.math.BigInteger;
import java.util.List;
import java.util.Set;

import org.springframework.stereotype.Component;

import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.ess.common.utils.TablesEnum;
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BannerResourceType;
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BsExportBannerResourcesObject;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.binlog.model.Operation.INSERT;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_CONTENT;
import static ru.yandex.direct.ess.router.rules.bsexport.resources.filter.BaseBannerResourceFilter.addDebugInfo;
import static ru.yandex.direct.ess.router.utils.ColumnsChangeType.ANY;

// Фильтр для банеров, рекламирующих мобильные приложения
@Component
public class BannerMobileContentFilter implements IBannerResourceFilter {

    private static final Set<String> MOBILE_CONTENT_BANNER_TYPES = Set.of(
            BannersBannerType.mobile_content.getLiteral(),
            BannersBannerType.cpc_video.getLiteral(),
            BannersBannerType.image_ad.getLiteral()
    );

    @Override
    public void init(TableChangesHandler<BsExportBannerResourcesObject> tableChangesHandler) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportBannerResourcesObject>()
                        .setTable(CAMPAIGNS_MOBILE_CONTENT)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID)
                        .setMapper(this::mapCampaignMobileContentObject)
                        .build());

        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportBannerResourcesObject>()
                        .setTable(BANNERS)
                        .setOperation(INSERT)
                        .setValuesFilter(this::checkBanner)
                        .setMapper(this::mapBannerObject)
                        .build());

        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportBannerResourcesObject>()
                        .setTable(BANNERS)
                        .setOperation(UPDATE)
                        .setColumns(ANY, List.of(BANNERS.STATUS_MODERATE, BANNERS.HREF, BANNERS.LANGUAGE))
                        .setValuesFilter(this::checkBanner)
                        .setMapper(this::mapBannerObject)
                        .build());

        tableChangesHandler.addTableChange(
                new TableChange.Builder<BsExportBannerResourcesObject>()
                        .setTable(MOBILE_CONTENT)
                        .setOperation(UPDATE)
                        .setColumns(ANY, List.of(
                                MOBILE_CONTENT.PUBLISHER_DOMAIN_ID,
                                MOBILE_CONTENT.STORE_CONTENT_ID,
                                MOBILE_CONTENT.BUNDLE_ID))
                        .setMapper(this::mapMobileContentIdObject)
                        .build());
    }

    private BannerResourceType bannerResourceType() {
        return BannerResourceType.BANNER_MOBILE_CONTENT;
    }

    private boolean checkBanner(ProceededChange proceededChange) {
        String bannerType = proceededChange.getAfter(BANNERS.BANNER_TYPE);
        if (!MOBILE_CONTENT_BANNER_TYPES.contains(bannerType)) {
            return false;
        }
        return !proceededChange.afterContains(BANNERS.STATUS_MODERATE)
                || proceededChange.getAfter(BANNERS.STATUS_MODERATE).equals(BannersStatusmoderate.Yes.getLiteral());
    }

    private BsExportBannerResourcesObject mapBannerObject(ProceededChange change) {
        var object = new BsExportBannerResourcesObject.Builder()
                .setBid(change.getPrimaryKey(BANNERS.BID))
                .setPid(change.getAfter(BANNERS.PID))
                .setCid(change.getAfter(BANNERS.CID))
                .setResourceType(bannerResourceType())
                .build();
        addDebugInfo(object, change);
        return object;
    }

    private BsExportBannerResourcesObject mapMobileContentIdObject(ProceededChange change) {
        BigInteger mobileContentId = change.getPrimaryKey(MOBILE_CONTENT.MOBILE_CONTENT_ID);
        var object = new BsExportBannerResourcesObject.Builder()
                .setAdditionalTable(TablesEnum.MOBILE_CONTENT)
                .setAdditionalId(mobileContentId.longValue())
                .setResourceType(bannerResourceType())
                .build();
        addDebugInfo(object, change);
        return object;
    }

    private BsExportBannerResourcesObject mapCampaignMobileContentObject(ProceededChange change) {
        var object = new BsExportBannerResourcesObject.Builder()
                .setAdditionalTable(TablesEnum.CAMPAIGNS_MOBILE_CONTENT)
                .setAdditionalId(change.getPrimaryKey(CAMPAIGNS_MOBILE_CONTENT.CID))
                .setResourceType(bannerResourceType())
                .build();
        addDebugInfo(object, change);
        return object;
    }
}
