package ru.yandex.direct.ess.router.rules.campaignlastchange;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.List;

import com.google.protobuf.InvalidProtocolBufferException;
import org.jooq.Field;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.ess.common.utils.TablesEnum;
import ru.yandex.direct.ess.config.campaignlastchange.CampAggregatedLastchangeConfig;
import ru.yandex.direct.ess.logicobjects.campaignlastchange.CampAggregatedLastchangeObject;
import ru.yandex.direct.ess.router.models.TEssEvent;
import ru.yandex.direct.ess.router.models.rule.AbstractRule;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.ess.router.utils.GrutTableChange;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;
import ru.yandex.grut.objects.proto.client.Schema.EObjectType;
import ru.yandex.grut.objects.proto.client.Schema.TBannerCandidateMeta;
import ru.yandex.grut.watchlog.Watch.TEvent;

import static ru.yandex.direct.binlog.model.Operation.DELETE;
import static ru.yandex.direct.binlog.model.Operation.INSERT;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.ess.router.models.WatchlogEventKt.getUtcTimestamp;
import static ru.yandex.direct.utils.DateTimeUtils.instantToMoscowDateTime;

@EssRule(CampAggregatedLastchangeConfig.class)
public class CampAggregatedLastchangeRule extends AbstractRule<CampAggregatedLastchangeObject> {

    private static final Logger logger = LoggerFactory.getLogger(CampAggregatedLastchangeRule.class);

    private static final TableChangesHandler<CampAggregatedLastchangeObject> tableChangesHandler =
            new TableChangesHandler<>();

    static {
        tableChangesHandler.addGrutTableChange(
                new GrutTableChange.Builder<CampAggregatedLastchangeObject>()
                        .setObjectType(EObjectType.OT_BANNER_CANDIDATE)
                        .setWatchlogMapper(CampAggregatedLastchangeRule::mapGrutBannerCandidateChangeToObject)
                        .build());
        tableChangesHandler.addTableChange(
                new TableChange.Builder<CampAggregatedLastchangeObject>()
                        .setTable(BANNERS)
                        .setOperation(INSERT)
                        .setMapper(CampAggregatedLastchangeRule::mapBannerChangeToObject)
                        .build());
        tableChangesHandler.addTableChange(
                new TableChange.Builder<CampAggregatedLastchangeObject>()
                        .setTable(PHRASES)
                        .setOperation(INSERT)
                        .setMapper(CampAggregatedLastchangeRule::mapPhrasesChangeToObject)
                        .build());
        tableChangesHandler.addTableChange(
                new TableChange.Builder<CampAggregatedLastchangeObject>()
                        .setTable(BANNER_IMAGES)
                        .setOperation(INSERT)
                        .setMapper(CampAggregatedLastchangeRule::mapBannerImagesChangeToObject).build());
        tableChangesHandler.addTableChange(
                new TableChange.Builder<CampAggregatedLastchangeObject>()
                        .setTable(BANNERS)
                        .setOperation(UPDATE)
                        .setColumn(BANNERS.LAST_CHANGE)
                        .setMapper(CampAggregatedLastchangeRule::mapBannerChangeToObject)
                        .build());
        tableChangesHandler.addTableChange(
                new TableChange.Builder<CampAggregatedLastchangeObject>()
                        .setTable(PHRASES)
                        .setOperation(UPDATE)
                        .setColumn(PHRASES.LAST_CHANGE)
                        .setMapper(CampAggregatedLastchangeRule::mapPhrasesChangeToObject)
                        .build());
        tableChangesHandler.addTableChange(
                new TableChange.Builder<CampAggregatedLastchangeObject>()
                        .setTable(BANNER_IMAGES)
                        .setOperation(UPDATE)
                        .setColumn(BANNER_IMAGES.DATE_ADDED)
                        .setMapper(CampAggregatedLastchangeRule::mapBannerImagesChangeToObject)
                        .build());
        tableChangesHandler.addTableChange(
                new TableChange.Builder<CampAggregatedLastchangeObject>()
                        .setTable(BANNERS)
                        .setOperation(DELETE)
                        .setMapper(CampAggregatedLastchangeRule::mapBannerChangeToObject)
                        .build());
        tableChangesHandler.addTableChange(
                new TableChange.Builder<CampAggregatedLastchangeObject>()
                        .setTable(PHRASES)
                        .setOperation(DELETE)
                        .setMapper(CampAggregatedLastchangeRule::mapPhrasesChangeToObject)
                        .build());
        tableChangesHandler.addTableChange(
                new TableChange.Builder<CampAggregatedLastchangeObject>()
                        .setTable(BANNER_IMAGES)
                        .setOperation(DELETE)
                        .setMapper(CampAggregatedLastchangeRule::mapBannerImagesChangeToObject)
                        .build());
    }

    @Override
    public List<CampAggregatedLastchangeObject> mapBinlogEvent(BinlogEvent event) {
        return tableChangesHandler.processChanges(event);
    }

    @Override
    public List<CampAggregatedLastchangeObject> mapWatchlogEvent(TEssEvent watchlogEvent) {
        return tableChangesHandler.processGrutChanges(watchlogEvent);
    }

    private static CampAggregatedLastchangeObject mapBannerChangeToObject(ProceededChange change) {
        return new CampAggregatedLastchangeObject(TablesEnum.BANNERS, change.getPrimaryKey(BANNERS.BID),
                extractId(change, BANNERS.CID), extractLastChange(change, BANNERS.LAST_CHANGE,
                change.getBinlogTimestamp()));
    }

    private static CampAggregatedLastchangeObject mapPhrasesChangeToObject(ProceededChange change) {
        return new CampAggregatedLastchangeObject(TablesEnum.PHRASES, change.getPrimaryKey(PHRASES.PID),
                extractId(change, PHRASES.CID), extractLastChange(change, PHRASES.LAST_CHANGE,
                change.getBinlogTimestamp()));
    }

    private static CampAggregatedLastchangeObject mapBannerImagesChangeToObject(ProceededChange change) {
        return new CampAggregatedLastchangeObject(TablesEnum.BANNER_IMAGES,
                change.getPrimaryKey(BANNER_IMAGES.IMAGE_ID), extractId(change, BANNER_IMAGES.BID),
                extractLastChange(change, BANNER_IMAGES.DATE_ADDED, change.getBinlogTimestamp()));
    }

    private static Long extractId(ProceededChange change, Field<Long> field) {
        if (change.getOperation() != DELETE) {
            return change.getAfter(field);
        } else {
            return change.getBefore(field);
        }
    }

    private static LocalDateTime extractLastChange(ProceededChange change, Field<LocalDateTime> lastChangeField,
                                                   LocalDateTime binlogTime) {
        if (change.getOperation() != DELETE) {
            return change.getAfter(lastChangeField);
        } else {
            return instantToMoscowDateTime(Instant.ofEpochSecond(binlogTime.toEpochSecond(ZoneOffset.UTC)));
        }
    }

    private static CampAggregatedLastchangeObject mapGrutBannerCandidateChangeToObject(TEssEvent event) {
        TBannerCandidateMeta bannerMeta;
        TEvent internalEvent = event.getEvent();
        try {
            bannerMeta = TBannerCandidateMeta.parseFrom(internalEvent.getObjectMeta());
        } catch (InvalidProtocolBufferException ex) {
            logger.error("Failed to batchProcessing protobuf", ex);
            throw new IllegalStateException(ex);
        }
        LocalDateTime moscowTime = instantToMoscowDateTime(Instant.ofEpochSecond(getUtcTimestamp(internalEvent)));

        logger.info("Parsed CampAggregatedLastchange event from watchlog event with id {}, campaign id {}, " +
                        "event type {}, timestamp {}, moscow time {}",
                bannerMeta.getDirectId(), event.getDirectCampaignId(), internalEvent.getEventType(),
                internalEvent.getTimestamp(), moscowTime);

        return new CampAggregatedLastchangeObject(TablesEnum.BANNER_CANDIDATES, bannerMeta.getDirectId(),
                event.getDirectCampaignId(), moscowTime);
    }
}
