package ru.yandex.direct.ess.router.rules.campaignstatuscorrect;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.jooq.Field;

import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.ess.config.campaignstatuscorrect.CampaignStatusCorrectCheckConfig;
import ru.yandex.direct.ess.logicobjects.campaignstatuscorrect.CampaignStatusCorrectCheckObject;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.ess.router.models.rule.StandardRule;
import ru.yandex.direct.ess.router.utils.ColumnsChangeType;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;

import static ru.yandex.direct.binlog.model.Operation.DELETE;
import static ru.yandex.direct.binlog.model.Operation.INSERT;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

@ParametersAreNonnullByDefault
@EssRule(value = CampaignStatusCorrectCheckConfig.class)
public class CampaignStatusCorrectCheckRule extends StandardRule<CampaignStatusCorrectCheckObject> {

    private static final Set<BannersBannerType> BANNERS_NEED_CHECK = ImmutableSet.of(
            BannersBannerType.cpm_banner, BannersBannerType.cpm_audio
    );

    public CampaignStatusCorrectCheckRule() {
        super(tableChangesHandler -> {
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<CampaignStatusCorrectCheckObject>()
                            .setTable(BANNERS)
                            .setOperation(INSERT)
                            .setValuesFilter(CampaignStatusCorrectCheckRule::isBannerNeedCheck)
                            .setMapper(CampaignStatusCorrectCheckRule::mapBannerChangeToObject)
                            .build());
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<CampaignStatusCorrectCheckObject>()
                            .setTable(PHRASES)
                            .setOperation(INSERT)
                            .setMapper(CampaignStatusCorrectCheckRule::mapPhrasesChangeToObject)
                            .build());
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<CampaignStatusCorrectCheckObject>()
                            .setTable(BANNERS)
                            .setOperation(UPDATE)
                            .setColumns(ColumnsChangeType.ANY,
                                    List.of(BANNERS.STATUS_MODERATE, BANNERS.STATUS_POST_MODERATE,
                                            BANNERS.STATUS_SHOW, BANNERS.STATUS_ARCH))
                            .setValuesFilter(CampaignStatusCorrectCheckRule::isBannerNeedCheck)
                            .setMapper(CampaignStatusCorrectCheckRule::mapBannerChangeToObject)
                            .build());
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<CampaignStatusCorrectCheckObject>()
                            .setTable(PHRASES)
                            .setOperation(UPDATE)
                            .setColumns(ColumnsChangeType.ANY,
                                    List.of(PHRASES.STATUS_MODERATE, PHRASES.STATUS_POST_MODERATE))
                            .setMapper(CampaignStatusCorrectCheckRule::mapPhrasesChangeToObject)
                            .build());
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<CampaignStatusCorrectCheckObject>()
                            .setTable(BANNERS)
                            .setOperation(DELETE)
                            .setMapper(CampaignStatusCorrectCheckRule::mapBannerChangeToObject)
                            .build());
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<CampaignStatusCorrectCheckObject>()
                            .setTable(PHRASES)
                            .setOperation(DELETE)
                            .setMapper(CampaignStatusCorrectCheckRule::mapPhrasesChangeToObject)
                            .build());
        });
    }

    private static boolean isBannerNeedCheck(ProceededChange change) {
        return BANNERS_NEED_CHECK.contains(BannersBannerType.valueOf(change.getAfter(BANNERS.BANNER_TYPE)));
    }

    private static CampaignStatusCorrectCheckObject mapBannerChangeToObject(ProceededChange change) {
        return new CampaignStatusCorrectCheckObject(extractCampaignId(change, BANNERS.CID));
    }

    private static CampaignStatusCorrectCheckObject mapPhrasesChangeToObject(ProceededChange change) {
        return new CampaignStatusCorrectCheckObject(extractCampaignId(change, PHRASES.CID));
    }

    private static Long extractCampaignId(ProceededChange change, Field<Long> field) {
        if (change.getOperation() != DELETE) {
            return change.getAfter(field);
        } else {
            return change.getBefore(field);
        }
    }
}
