package ru.yandex.direct.ess.router.rules.campeventlog;

import java.math.BigDecimal;
import java.util.Objects;

import com.fasterxml.jackson.databind.JsonNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.ess.config.campeventlog.CampaignEventLogConfig;
import ru.yandex.direct.ess.logicobjects.campeventlog.CampaignEventLogLogicObject;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.ess.router.models.rule.StandardRule;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.ess.logicobjects.campeventlog.CampaignEventLogLogicObject.CampaignEventType.AVG_CPA_CHANGED;
import static ru.yandex.direct.ess.logicobjects.campeventlog.CampaignEventLogLogicObject.CampaignEventType.DAILY_BUDGET_CHANGED;
import static ru.yandex.direct.ess.logicobjects.campeventlog.CampaignEventLogLogicObject.CampaignEventType.ROI_COEF_CHANGED;
import static ru.yandex.direct.ess.logicobjects.campeventlog.CampaignEventLogLogicObject.CampaignEventType.WEEKLY_BUDGET_CHANGED;

@EssRule(CampaignEventLogConfig.class)
public class CampaignEventLogRule extends StandardRule<CampaignEventLogLogicObject> {

    private static final Logger logger = LoggerFactory.getLogger(CampaignEventLogRule.class);
    public static final String SUM = "sum";
    public static final String AVG_CPA = "avg_cpa";
    public static final String ROI_COEF = "roi_coef";

    public CampaignEventLogRule() {
        super(tableChangesHandler -> {
            addStrategyDataChange(tableChangesHandler, SUM);
            addStrategyDataChange(tableChangesHandler, AVG_CPA);
            addStrategyDataChange(tableChangesHandler, ROI_COEF);

            tableChangesHandler.addTableChange(
                    new TableChange.Builder<CampaignEventLogLogicObject>()
                            .setTable(CAMPAIGNS)
                            .setOperation(UPDATE)
                            .setColumn(CAMPAIGNS.DAY_BUDGET)
                            .setMapper(CampaignEventLogRule::mapDailyBudgetChangeToLogicObject)
                            .build());
        });
    }

    private static void addStrategyDataChange(TableChangesHandler<CampaignEventLogLogicObject> tableChangesHandler,
                                              String fieldName) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<CampaignEventLogLogicObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS.STRATEGY_DATA)
                        .setValuesFilter(change -> filterWeeklyBudgetChange(change, fieldName))
                        .setMapper(change -> mapWeeklyBudgetChangeToLogicObject(change, fieldName))
                        .build());
    }

    private static boolean filterWeeklyBudgetChange(ProceededChange proceededChange, String fieldName) {
        if (proceededChange.getBefore(CAMPAIGNS.STRATEGY_DATA) == null
                || proceededChange.getAfter(CAMPAIGNS.STRATEGY_DATA) == null) {
            Long campaignId = proceededChange.getPrimaryKey(CAMPAIGNS.CID);
            logger.error("Unexpected null in field ppc.campaigns.strategy_data for campaign {}", campaignId);
            return false;
        }

        JsonNode strategyBefore = JsonUtils.fromJson(proceededChange.getBefore(CAMPAIGNS.STRATEGY_DATA));
        JsonNode strategyAfter = JsonUtils.fromJson(proceededChange.getAfter(CAMPAIGNS.STRATEGY_DATA));

        BigDecimal valueBefore = JsonUtils.ifNotNull(strategyBefore.get(fieldName), JsonNode::decimalValue);
        BigDecimal valueAfter = JsonUtils.ifNotNull(strategyAfter.get(fieldName), JsonNode::decimalValue);

        return !Objects.equals(valueBefore, valueAfter);
    }

    private static CampaignEventLogLogicObject mapWeeklyBudgetChangeToLogicObject(ProceededChange proceededChange,
                                                                                  String fieldName) {
        JsonNode strategyBefore = JsonUtils.fromJson(proceededChange.getBefore(CAMPAIGNS.STRATEGY_DATA));
        JsonNode strategyAfter = JsonUtils.fromJson(proceededChange.getAfter(CAMPAIGNS.STRATEGY_DATA));

        BigDecimal valueBefore = JsonUtils.ifNotNull(strategyBefore.get(fieldName), JsonNode::decimalValue);
        BigDecimal valueAfter = JsonUtils.ifNotNull(strategyAfter.get(fieldName), JsonNode::decimalValue);

        CampaignEventLogLogicObject campaignEventLogLogicObject = new CampaignEventLogLogicObject()
                .withEventTime(proceededChange.getBinlogTimestamp())
                .withCampaignId(proceededChange.getPrimaryKey(CAMPAIGNS.CID))
                .withClientId(ClientId.fromLong(proceededChange.getAfter(CAMPAIGNS.CLIENT_ID)));

        switch (fieldName) {
            case SUM:
                return campaignEventLogLogicObject
                        .withCampaignEventType(WEEKLY_BUDGET_CHANGED)
                        .withBudgetBefore(valueBefore)
                        .withBudgetAfter(valueAfter);
            case AVG_CPA:
                return campaignEventLogLogicObject
                        .withCampaignEventType(AVG_CPA_CHANGED)
                        .withAvgCpaBefore(valueBefore)
                        .withAvgCpaAfter(valueAfter);
            case ROI_COEF:
                return campaignEventLogLogicObject
                        .withCampaignEventType(ROI_COEF_CHANGED)
                        .withRoiCoefBefore(valueBefore)
                        .withRoiCoefAfter(valueAfter);
            default:
                throw new IllegalArgumentException("wrong field name " + fieldName);
        }
    }

    private static CampaignEventLogLogicObject mapDailyBudgetChangeToLogicObject(ProceededChange proceededChange) {
        return new CampaignEventLogLogicObject()
                .withEventTime(proceededChange.getBinlogTimestamp())
                .withCampaignId(proceededChange.getPrimaryKey(CAMPAIGNS.CID))
                .withClientId(ClientId.fromLong(proceededChange.getAfter(CAMPAIGNS.CLIENT_ID)))
                .withCampaignEventType(DAILY_BUDGET_CHANGED)
                .withBudgetBefore(proceededChange.getBefore(CAMPAIGNS.DAY_BUDGET))
                .withBudgetAfter(proceededChange.getAfter(CAMPAIGNS.DAY_BUDGET));
    }
}
