package ru.yandex.direct.ess.router.rules.feeds.usagetypes

import com.google.gson.Gson
import com.google.gson.JsonSyntaxException
import com.google.gson.reflect.TypeToken
import org.jooq.Field
import ru.yandex.direct.binlog.model.BinlogEvent
import ru.yandex.direct.binlog.model.Operation
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_TEXT
import ru.yandex.direct.dbschema.ppc.Tables.AGGR_STATUSES_ADGROUPS
import ru.yandex.direct.dbschema.ppc.Tables.AGGR_STATUSES_CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.ess.config.feeds.usagetypes.RecalculateFeedUsageTypesConfig
import ru.yandex.direct.ess.logicobjects.feeds.usagetypes.FeedUsageTypesObject
import ru.yandex.direct.ess.router.models.rule.AbstractRule
import ru.yandex.direct.ess.router.models.rule.EssRule
import ru.yandex.direct.ess.router.utils.ProceededChange
import ru.yandex.direct.ess.router.utils.TableChange
import ru.yandex.direct.ess.router.utils.TableChangesHandler
import java.math.BigInteger

/**
 * Рула следит за всеми изменениями, которые могу повлиять на статус используемости фида
 */
@EssRule(RecalculateFeedUsageTypesConfig::class)
class RecalculateFeedUsageTypesRule : AbstractRule<FeedUsageTypesObject>() {
    companion object {
        // Very similar check is in core FeedUsageService. Keep it corresponding.
        private val STOP_STATUSES = setOf(
            GdSelfStatusEnum.ARCHIVED.name,
            GdSelfStatusEnum.STOP_CRIT.name,
            GdSelfStatusEnum.STOP_OK.name,
            GdSelfStatusEnum.STOP_WARN.name,
            GdSelfStatusEnum.DRAFT.name,
        )

        fun isObjectStatusStopped(selfStatus: Any?): Boolean =
            selfStatus != null && selfStatus is String && STOP_STATUSES.contains(selfStatus)

        val GSON = Gson()
    }

    private val tableChangesHandler = TableChangesHandler<FeedUsageTypesObject>()

    init {
        tableChangesHandler.addTableChange(
            TableChange.Builder<FeedUsageTypesObject>()
                .setTable(ADGROUPS_TEXT)
                .setOperation(Operation.INSERT)
                .setValuesFilter { isTextByFeed(it) }
                .setMapper { obj: ProceededChange -> mapChangeForInsertText(obj) }
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<FeedUsageTypesObject>()
                .setTable(ADGROUPS_DYNAMIC)
                .setOperation(Operation.INSERT)
                .setValuesFilter { isDynamicByFeed(it) }
                .setMapper { obj: ProceededChange -> mapChangeForInsertDynamic(obj) }
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<FeedUsageTypesObject>()
                .setTable(ADGROUPS_PERFORMANCE)
                .setOperation(Operation.INSERT)
                .setMapper { obj: ProceededChange -> mapChangeForInsertPerformance(obj) }
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<FeedUsageTypesObject>()
                .setTable(ADGROUPS_TEXT)
                .setOperation(Operation.DELETE)
                .setMapper { obj: ProceededChange -> mapChangeForDeleteText(obj) }
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<FeedUsageTypesObject>()
                .setTable(ADGROUPS_DYNAMIC)
                .setOperation(Operation.DELETE)
                .setMapper { obj: ProceededChange -> mapChangeForDeleteDynamic(obj) }
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<FeedUsageTypesObject>()
                .setTable(ADGROUPS_PERFORMANCE)
                .setOperation(Operation.DELETE)
                .setMapper { obj: ProceededChange -> mapChangeForDeletePerformance(obj) }
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<FeedUsageTypesObject>()
                .setTable(AGGR_STATUSES_ADGROUPS)
                .setValuesFilter { isAggrStatusAdGroupsSelfStatusChanged(it) }
                .setOperation(Operation.UPDATE)
                .setMapper { obj: ProceededChange -> mapChangeForUpdateAggrAdGroups(obj) }
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<FeedUsageTypesObject>()
                .setTable(AGGR_STATUSES_CAMPAIGNS)
                .setValuesFilter { isAggrStatusCampaignsSelfStatusChanged(it) }
                .setOperation(Operation.UPDATE)
                .setMapper { obj: ProceededChange -> mapChangeForUpdateAggrCampaigns(obj) }
                .build()
        )
    }

    private fun isDynamicByFeed(it: ProceededChange) =
        it.afterContains(ADGROUPS_DYNAMIC.FEED_ID)
            && null != it.getAfter(ADGROUPS_DYNAMIC.FEED_ID)

    private fun isTextByFeed(it: ProceededChange) =
        it.afterContains(ADGROUPS_TEXT.FEED_ID)
            && null != it.getAfter(ADGROUPS_TEXT.FEED_ID)

    private fun selfStatusChanged(before: String?, after: String?): Boolean {
        if (before == null && after == null) {
            return false
        }
        if (before == null || after == null) {
            return true
        }
        return try {
            val mapAfter: Map<String, Any> = GSON.fromJson(after, object : TypeToken<Map<String, Any>>() {}.type)
            val mapBefore: Map<String, Any> = GSON.fromJson(before, object : TypeToken<Map<String, Any>>() {}.type)
            isObjectStatusStopped(mapAfter["s"]) != isObjectStatusStopped(mapBefore["s"])
        } catch (e: JsonSyntaxException) {
            false
        }
    }

    private fun isAggrStatusAdGroupsSelfStatusChanged(it: ProceededChange) =
        it.afterContains(AGGR_STATUSES_ADGROUPS.AGGR_DATA) && it.beforeContains(AGGR_STATUSES_ADGROUPS.AGGR_DATA) &&
            selfStatusChanged(
                it.getBefore(AGGR_STATUSES_ADGROUPS.AGGR_DATA),
                it.getAfter(AGGR_STATUSES_ADGROUPS.AGGR_DATA)
            )

    private fun isAggrStatusCampaignsSelfStatusChanged(it: ProceededChange) =
        it.afterContains(AGGR_STATUSES_CAMPAIGNS.AGGR_DATA) && it.beforeContains(AGGR_STATUSES_CAMPAIGNS.AGGR_DATA) &&
            selfStatusChanged(
                it.getBefore(AGGR_STATUSES_CAMPAIGNS.AGGR_DATA),
                it.getAfter(AGGR_STATUSES_CAMPAIGNS.AGGR_DATA)
            )

    override fun mapBinlogEvent(binlogEvent: BinlogEvent): List<FeedUsageTypesObject> {
        return tableChangesHandler.processChanges(binlogEvent)
    }

    private fun mapChangeForInsertText(change: ProceededChange): FeedUsageTypesObject {
        return FeedUsageTypesObject(
            getAfterLongField(change, ADGROUPS_TEXT.FEED_ID),
            CampaignsType.text,
            null,
            null,
            Operation.INSERT
        )
    }

    private fun mapChangeForInsertDynamic(change: ProceededChange): FeedUsageTypesObject {
        return FeedUsageTypesObject(
            getAfterLongField(change, ADGROUPS_DYNAMIC.FEED_ID),
            CampaignsType.dynamic,
            null,
            null,
            Operation.INSERT
        )
    }

    private fun mapChangeForInsertPerformance(change: ProceededChange): FeedUsageTypesObject {
        return FeedUsageTypesObject(
            getAfterLongField(change, ADGROUPS_PERFORMANCE.FEED_ID),
            CampaignsType.performance,
            null,
            null,
            Operation.INSERT
        )
    }

    private fun mapChangeForDeleteText(change: ProceededChange): FeedUsageTypesObject {
        return FeedUsageTypesObject(
            null,
            CampaignsType.text,
            null,
            change.getLongPrimaryKey(ADGROUPS_TEXT.PID),
            Operation.DELETE
        )
    }

    private fun mapChangeForDeleteDynamic(change: ProceededChange): FeedUsageTypesObject {
        return FeedUsageTypesObject(
            null,
            CampaignsType.dynamic,
            null,
            change.getLongPrimaryKey(ADGROUPS_DYNAMIC.PID),
            Operation.DELETE
        )
    }

    private fun mapChangeForDeletePerformance(change: ProceededChange): FeedUsageTypesObject {
        return FeedUsageTypesObject(
            null,
            CampaignsType.performance,
            null,
            change.getLongPrimaryKey(ADGROUPS_PERFORMANCE.PID),
            Operation.DELETE
        )
    }

    private fun mapChangeForUpdateAggrAdGroups(change: ProceededChange): FeedUsageTypesObject {
        return FeedUsageTypesObject(
            null,
            null,
            null,
            change.getLongPrimaryKey(AGGR_STATUSES_ADGROUPS.PID),
            Operation.UPDATE
        )
    }

    private fun mapChangeForUpdateAggrCampaigns(change: ProceededChange): FeedUsageTypesObject {
        return FeedUsageTypesObject(
            null,
            null,
            change.getLongPrimaryKey(AGGR_STATUSES_CAMPAIGNS.CID),
            null,
            Operation.UPDATE
        )
    }

    private fun getAfterLongField(proceededChange: ProceededChange, field: Field<Long>): Long? {
        return if (proceededChange.getAfter<Any, Long>(field) is BigInteger) {
            (proceededChange.getAfter<Any, Long>(field) as BigInteger).toLong()
        } else {
            proceededChange.getAfter<Long, Long>(field)
        }
    }
}
