package ru.yandex.direct.ess.router.rules.moderation.adgroup;

import java.util.Set;

import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.core.entity.moderation.model.TransportStatus;
import ru.yandex.direct.ess.config.moderation.adgroup.AdGroupModerationConfig;
import ru.yandex.direct.ess.logicobjects.moderation.adgroup.AdGroupModerationEventObject;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.ess.router.rules.moderation.ModerationRule;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;

import static ru.yandex.direct.binlog.model.Operation.INSERT;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.ess.router.rules.moderation.ModerationRulesUtils.isCopy;
import static ru.yandex.direct.ess.router.rules.moderation.ModerationRulesUtils.needModeration;
import static ru.yandex.direct.ess.router.utils.ProceededChangeUtil.getProceededChangeEssTag;
import static ru.yandex.direct.ess.router.utils.ProceededChangeUtil.getProceededChangeTimestamp;

@EssRule(AdGroupModerationConfig.class)
public class AdGroupModerationRule extends ModerationRule<AdGroupModerationEventObject> {

    public AdGroupModerationRule() {
        super(tableChangesHandler -> {
            tableChangesHandler.addTableChange(
                    new TableChange.Builder<AdGroupModerationEventObject>()
                            .setTable(PHRASES)
                            .setOperation(INSERT)
                            .setValuesFilter(change -> needModerationForInsertedAdGroup(change))
                            .setMapper(AdGroupModerationRule::mapChangeToObject)
                            .build()
            );

            tableChangesHandler.addTableChange(
                    new TableChange.Builder<AdGroupModerationEventObject>()
                            .setTable(PHRASES)
                            .setOperation(Operation.UPDATE)
                            .setColumn(PHRASES.STATUS_MODERATE)
                            .setValuesFilter(change -> needModeration(change.getAfter(PHRASES.STATUS_MODERATE), UPDATE))
                            .setMapper(AdGroupModerationRule::mapChangeToObject)
                            .build()
            );

        });
    }

    private static final Set<String> IGNORED_STATUSES_IN_RESHARDING = Set.of(TransportStatus.Yes.name(),
            TransportStatus.No.name());

    private static boolean needModerationForInsertedAdGroup(ProceededChange change) {
        String statusModerate = change.getAfter(PHRASES.STATUS_MODERATE);
        if (change.getIsResharding() && IGNORED_STATUSES_IN_RESHARDING.contains(statusModerate)) {
            return false;
        }
        return needModeration(statusModerate, INSERT);
    }

    private static AdGroupModerationEventObject mapChangeToObject(ProceededChange proceededChange) {
        return new AdGroupModerationEventObject(getProceededChangeEssTag(proceededChange),
                getProceededChangeTimestamp(proceededChange),
                proceededChange.getPrimaryKey(PHRASES.PID),
                isCopy(proceededChange.getAfter(PHRASES.STATUS_MODERATE), proceededChange.getOperation()));
    }
}
