package ru.yandex.direct.ess.router.rules.moderation.asset;

import org.jooq.Table;
import org.jooq.TableField;
import org.jooq.TableRecord;

import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.ess.logicobjects.moderation.asset.BannerAssetModerationEventsObject;
import ru.yandex.direct.ess.router.rules.moderation.ModerationRule;
import ru.yandex.direct.ess.router.rules.moderation.ModerationRulesUtils;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;

import static ru.yandex.direct.ess.router.utils.ProceededChangeUtil.getProceededChangeEssTag;
import static ru.yandex.direct.ess.router.utils.ProceededChangeUtil.getProceededChangeTimestamp;

public abstract class BaseAssetModerationRule<R extends TableRecord> extends ModerationRule<BannerAssetModerationEventsObject> {

    public BaseAssetModerationRule(Table<R> table,
                                   TableField<R, ?> moderationStatusField,
                                   TableField<R, ?> bidField) {
        this(table, moderationStatusField, bidField, true);
    }

    public BaseAssetModerationRule(Table<R> table,
                                   TableField<R, ?> moderationStatusField,
                                   TableField<R, ?> bidField,
                                   boolean isBidPrimaryKey) {
        super(tableChangesHandler -> {
                    tableChangesHandler.addTableChange(
                            new TableChange.Builder<BannerAssetModerationEventsObject>()
                                    .setTable(table)
                                    .setOperation(Operation.INSERT)
                                    .setValuesFilter(e -> isModerationNeeded(e, Operation.INSERT,
                                            moderationStatusField))
                                    .setMapper(e -> mapChangeToObject(
                                            e, Operation.INSERT, moderationStatusField, bidField,
                                            isBidPrimaryKey))
                                    .build()
                    );

                    tableChangesHandler.addTableChange(
                            new TableChange.Builder<BannerAssetModerationEventsObject>()
                                    .setTable(table)
                                    .setOperation(Operation.UPDATE)
                                    .setColumn(moderationStatusField)
                                    .setValuesFilter(e -> isModerationNeeded(e, Operation.UPDATE,
                                            moderationStatusField))
                                    .setMapper(e -> mapChangeToObject(
                                            e, Operation.UPDATE, moderationStatusField, bidField,
                                            isBidPrimaryKey))
                                    .build()
                    );

                }
        );
    }

    private static boolean isModerationNeeded(ProceededChange change, Operation mode,
                                              TableField<?, ?> moderationStatusField) {
        return ModerationRulesUtils.needModeration(change.getAfter(moderationStatusField), mode);
    }

    private static BannerAssetModerationEventsObject mapChangeToObject(ProceededChange change, Operation mode,
                                                                       TableField<?, ?> moderationStatusField,
                                                                       TableField<?, ?> bannerIdField,
                                                                       boolean isBidPrimaryKey) {
        Long bid = isBidPrimaryKey ? change.getPrimaryKey(bannerIdField) : change.getAfter(bannerIdField);
        return new BannerAssetModerationEventsObject(getProceededChangeEssTag(change),
                getProceededChangeTimestamp(change),
                bid,
                ModerationRulesUtils.isCopy(change.getAfter(moderationStatusField), mode));
    }
}
