package ru.yandex.direct.ess.router.rules.moderation.campaign;

import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.direct.ess.config.moderation.campaign.CampaignModerationConfig;
import ru.yandex.direct.ess.logicobjects.moderation.campaign.CampaignModerationEvent;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.ess.router.rules.moderation.ModerationRule;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.ess.router.utils.ProceededChangeUtil.getProceededChangeEssTag;
import static ru.yandex.direct.ess.router.utils.ProceededChangeUtil.getProceededChangeTimestamp;

/**
 * Ess-правило для отправки нотификаций о кампаниях в модерацию:
 * - Стандартная нотификация о модерации по statusModerate=Ready
 * - Нотификация о копированиия кампании (в отдельном формате, в отдельный топик)
 */
@EssRule(CampaignModerationConfig.class)
public class CampaignModerationRule extends ModerationRule<CampaignModerationEvent> {

    public CampaignModerationRule() {
        super(handler -> {
            handler.addTableChange(
                    new TableChange.Builder<CampaignModerationEvent>()
                            .setTable(CAMPAIGNS)
                            .setOperation(Operation.INSERT)
                            .setValuesFilter(change -> isCopyEvent(change) || isCampaignNeedModeration(change))
                            .setMapper(CampaignModerationRule::mapInsertToObject)
                            .build());
            handler.addTableChange(
                    new TableChange.Builder<CampaignModerationEvent>()
                            .setTable(CAMPAIGNS)
                            .setOperation(Operation.UPDATE)
                            .setColumn(CAMPAIGNS.STATUS_MODERATE)
                            .setValuesFilter(CampaignModerationRule::isCampaignNeedModeration)
                            .setMapper(CampaignModerationRule::mapUpdateToObject)
                            .build());
            }
        );
    }

    private static boolean isCopyEvent(ProceededChange change) {
        long copiedFrom = change.getAfter(CAMPAIGNS.COPIED_FROM);
        return copiedFrom > 0;
    }

    private static boolean isCampaignNeedModeration(ProceededChange change) {
        CampaignsStatusmoderate status = CampaignsStatusmoderate.valueOf(change.getAfter(CAMPAIGNS.STATUS_MODERATE));
        return status == CampaignsStatusmoderate.Ready;
    }

    private static CampaignModerationEvent mapInsertToObject(ProceededChange change) {
        boolean needModerationEvent = isCampaignNeedModeration(change);
        return new CampaignModerationEvent(
                getProceededChangeEssTag(change),
                getProceededChangeTimestamp(change),
                change.getPrimaryKey(CAMPAIGNS.CID),
                change.getAfter(CAMPAIGNS.CLIENT_ID),
                change.getLongBeforeOrAfter(CAMPAIGNS.UID),
                change.getAfter(CAMPAIGNS.COPIED_FROM),
                needModerationEvent ? change.getPrimaryKey(CAMPAIGNS.CID) : null,
                needModerationEvent ? change.getAfter(CAMPAIGNS.CLIENT_ID) : null
        );
    }

    private static CampaignModerationEvent mapUpdateToObject(ProceededChange change) {
        boolean needModerationEvent = isCampaignNeedModeration(change);
        return new CampaignModerationEvent(
                getProceededChangeEssTag(change),
                getProceededChangeTimestamp(change),
                null,
                null,
                null,
                null,
                change.getPrimaryKey(CAMPAIGNS.CID),
                change.getAfter(CAMPAIGNS.CLIENT_ID)
        );
    }
}
