package ru.yandex.direct.ess.router.rules.moderation.special;

import java.util.function.Function;
import java.util.function.Predicate;

import org.jooq.Field;
import org.jooq.Record;
import org.jooq.TableField;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.ess.config.moderation.special.ModerationDeletionConfig;
import ru.yandex.direct.ess.logicobjects.moderation.special.ModerationDeletionEvent;
import ru.yandex.direct.ess.router.models.rule.EssRule;
import ru.yandex.direct.ess.router.rules.moderation.ModerationRule;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_BUTTONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_DISPLAY_HREFS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_LOGOS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.ess.router.rules.moderation.special.ModerationDeletionTypeUtil.getDeletedObjectType;
import static ru.yandex.direct.ess.router.utils.ProceededChangeUtil.getProceededChangeEssTag;
import static ru.yandex.direct.ess.router.utils.ProceededChangeUtil.getProceededChangeTimestamp;

@EssRule(ModerationDeletionConfig.class)
public class ModerationDeletionRule extends ModerationRule<ModerationDeletionEvent> {

    @Autowired
    public ModerationDeletionRule() {
        super(handler -> {
            handler.addTableChange(deletedRowWithIdInPrimaryKey(PHRASES.PID));
            handler.addTableChange(deletedRowWithIdInPrimaryKey(BANNERS.BID,
                    ModerationDeletionRule::notPerformanceBanner));
            handler.addTableChange(deletedRowWithIdInField(BANNERS_PERFORMANCE.BID));
            handler.addTableChange(deletedRowWithIdInField(BANNER_IMAGES.BID));
            handler.addTableChange(deletedRowWithIdInPrimaryKey(BANNER_TURBOLANDINGS.BID));
            handler.addTableChange(deletedRowWithIdInPrimaryKey(BANNER_DISPLAY_HREFS.BID));
            handler.addTableChange(deletedRowWithIdInPrimaryKey(BANNER_LOGOS.BID));
            handler.addTableChange(deletedRowWithIdInPrimaryKey(BANNER_BUTTONS.BID));

            handler.addTableChange(deletedFieldValueWithIdInPrimaryKey(BANNERS.BID, BANNERS.VCARD_ID));
            handler.addTableChange(deletedFieldValueWithIdInPrimaryKey(BANNERS.BID, BANNERS.SITELINKS_SET_ID));
        });
    }

    /**
     * Возвращает обработчик изменений (TableChange), описывающий удаление строки из таблицы,
     * которой принадлежит указанное поле. В качестве id в результирующем объекте будет значение поля idField.
     * <p>
     * ВАЖНО: поле idField должно быть Primary Key.
     */
    private static TableChange<ModerationDeletionEvent> deletedRowWithIdInPrimaryKey(TableField<?, Long> idField) {
        return deletedRow(idField, change -> true, change -> mapEventWithIdInPrimaryKey(change, idField));
    }

    /**
     * Возвращает обработчик изменений (TableChange), описывающий удаление строки из таблицы,
     * которой принадлежит указанное поле. В качестве id в результирующем объекте будет значение поля idField.
     * <p>
     * ВАЖНО: поле idField должно быть Primary Key.
     */
    private static TableChange<ModerationDeletionEvent> deletedRowWithIdInPrimaryKey(
            TableField<?, Long> idField, Predicate<ProceededChange> filter) {
        return deletedRow(idField, filter, change -> mapEventWithIdInPrimaryKey(change, idField));
    }

    /**
     * Возвращает обработчик изменений (TableChange), описывающий удаление строки из таблицы,
     * которой принадлежит указанное поле. В качестве id в результирующем объекте будет значение поля idField.
     * <p>
     * ВАЖНО: поле idField должно быть не Primary Key.
     */
    private static TableChange<ModerationDeletionEvent> deletedRowWithIdInField(TableField<?, Long> idField) {
        return deletedRow(idField, change -> true, change -> mapEventWithIdInField(change, idField));
    }

    private static TableChange<ModerationDeletionEvent> deletedRow(
            TableField<?, Long> idField,
            Predicate<ProceededChange> filter,
            Function<ProceededChange, ModerationDeletionEvent> mapper) {
        return new TableChange.Builder<ModerationDeletionEvent>()
                .setTable(idField.getTable())
                .setOperation(Operation.DELETE)
                .setValuesFilter(filter)
                .setMapper(mapper)
                .build();
    }

    private static <R extends Record> TableChange<ModerationDeletionEvent> deletedFieldValueWithIdInPrimaryKey(
            TableField<R, Long> idField, TableField<R, ?> fieldWithDeletedValue) {
        return new TableChange.Builder<ModerationDeletionEvent>()
                .setTable(idField.getTable())
                .setOperation(Operation.UPDATE)
                .setValuesFilter(hasDeletedValuePredicate(fieldWithDeletedValue))
                .setMapper(change -> mapEventWithIdInPrimaryKey(change, idField, fieldWithDeletedValue))
                .build();
    }

    private static ModerationDeletionEvent mapEventWithIdInPrimaryKey(
            ProceededChange change, TableField<?, Long> idField) {
        return mapEventWithIdInPrimaryKey(change, idField, null);
    }

    private static ModerationDeletionEvent mapEventWithIdInPrimaryKey(
            ProceededChange change, TableField<?, Long> idField, Field<?> field) {
        return new ModerationDeletionEvent(
                getProceededChangeEssTag(change),
                getProceededChangeTimestamp(change),
                change.getPrimaryKey(idField),
                getDeletedObjectType(idField.getTable(), field));
    }

    private static ModerationDeletionEvent mapEventWithIdInField(
            ProceededChange change, TableField<?, Long> idField) {
        return mapEventWithIdInField(change, idField, null);
    }

    @SuppressWarnings("SameParameterValue")
    private static ModerationDeletionEvent mapEventWithIdInField(
            ProceededChange change, TableField<?, Long> idField, Field<?> field) {
        return new ModerationDeletionEvent(
                getProceededChangeEssTag(change),
                getProceededChangeTimestamp(change),
                change.getBefore(idField),
                getDeletedObjectType(idField.getTable(), field));
    }

    private static Predicate<ProceededChange> hasDeletedValuePredicate(Field<?> field) {
        return change -> change.afterContains(field) && change.getAfter(field) == null;
    }

    private static boolean notPerformanceBanner(ProceededChange change) {
        return change.beforeContains(BANNERS.BANNER_TYPE) &&
                change.getBefore(BANNERS.BANNER_TYPE) != BannersBannerType.performance;
    }
}
