package ru.yandex.direct.ess.router.rules.mysql2grut

import org.jooq.Field
import org.jooq.impl.TableImpl
import ru.yandex.direct.binlog.model.BinlogEvent
import ru.yandex.direct.binlog.model.Operation
import ru.yandex.direct.dbschema.ppc.Tables.AB_SEGMENT_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.Tables.ADDRESSES
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_INTERNAL
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_MINUS_WORDS
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_MOBILE_CONTENT
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_ADDITIONAL_TARGETINGS
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_BS_TAGS
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUP_PAGE_TARGETS
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS
import ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES
import ru.yandex.direct.dbschema.ppc.Tables.BANNER_TYPE_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.Tables.BIDS
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_BASE
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_DYNAMIC
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_HREF_PARAMS
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_RETARGETING
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_CPM_PRICE
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_CPM_YNDX_FRONTPAGE
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_INTERNAL
import ru.yandex.direct.dbschema.ppc.Tables.CAMP_ADDITIONAL_TARGETINGS
import ru.yandex.direct.dbschema.ppc.Tables.CAMP_AUTOBUDGET_RESTART
import ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS
import ru.yandex.direct.dbschema.ppc.Tables.CAMP_ORDER_TYPES
import ru.yandex.direct.dbschema.ppc.Tables.CLIENTS
import ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS
import ru.yandex.direct.dbschema.ppc.Tables.CLIENT_ADDITIONAL_TARGETINGS
import ru.yandex.direct.dbschema.ppc.Tables.CLIENT_NDS
import ru.yandex.direct.dbschema.ppc.Tables.DEMOGRAPHY_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.Tables.DYNAMIC_CONDITIONS
import ru.yandex.direct.dbschema.ppc.Tables.EXPRESSION_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.Tables.GEO_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS
import ru.yandex.direct.dbschema.ppc.Tables.INTERNAL_AD_PRODUCTS
import ru.yandex.direct.dbschema.ppc.Tables.INVENTORY_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.Tables.MINUS_WORDS
import ru.yandex.direct.dbschema.ppc.Tables.MOBILE_CONTENT
import ru.yandex.direct.dbschema.ppc.Tables.MOBILE_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES
import ru.yandex.direct.dbschema.ppc.Tables.PHRASES
import ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_CONDITIONS
import ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_GOALS
import ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.Tables.STRATEGIES
import ru.yandex.direct.dbschema.ppc.Tables.TABLET_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.Tables.TRAFARET_POSITION_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.Tables.VCARDS
import ru.yandex.direct.dbschema.ppc.Tables.WALLET_CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.Tables.WEATHER_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.Tables.WIDGET_PARTNER_CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.ess.config.mysql2grut.Mysql2GrutReplicationConfig
import ru.yandex.direct.ess.logicobjects.mysql2grut.BidModifierTableType
import ru.yandex.direct.ess.logicobjects.mysql2grut.BiddableShowConditionChangeType
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import ru.yandex.direct.ess.router.models.rule.AbstractRule
import ru.yandex.direct.ess.router.models.rule.EssRule
import ru.yandex.direct.ess.router.rules.bsexport.adgroup.filter.AdGroupShowConditionsFilter
import ru.yandex.direct.ess.router.utils.ColumnsChangeType
import ru.yandex.direct.ess.router.utils.ProceededChange
import ru.yandex.direct.ess.router.utils.TableChange
import ru.yandex.direct.ess.router.utils.TableChangesHandler
import ru.yandex.direct.ess.router.utils.addRulesToHandler
import java.math.BigDecimal

@EssRule(Mysql2GrutReplicationConfig::class, runCondition = NonDevelopmentEnvironment::class)
class Mysql2GrutReplicationRule : AbstractRule<Mysql2GrutReplicationObject>() {
    private val tableChangesHandler = TableChangesHandler<Mysql2GrutReplicationObject>()

    init {
        tableChangesHandler.apply {
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CLIENTS)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapToClientObject(it) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CLIENTS)
                    .setOperation(Operation.UPDATE)
                    .setMapper { mapToClientObject(it) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CLIENTS)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapToClientObject(it) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CLIENTS_OPTIONS)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapToClientObjectWithPrimaryKey(it, CLIENTS_OPTIONS.CLIENT_ID) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CLIENTS_OPTIONS)
                    .setOperation(Operation.UPDATE)
                    .setMapper { mapToClientObjectWithPrimaryKey(it, CLIENTS_OPTIONS.CLIENT_ID) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CLIENTS_OPTIONS)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapToClientObjectWithPrimaryKey(it, CLIENTS_OPTIONS.CLIENT_ID) }
                    .build()
            )
            addRulesToHandler(
                tableChangesHandler,
                CLIENT_ADDITIONAL_TARGETINGS,
                { ch -> Mysql2GrutReplicationObject(clientId = ch.getLongBeforeOrAfter(CLIENT_ADDITIONAL_TARGETINGS.CLIENT_ID)) },
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CLIENTS_OPTIONS)
                    .setOperation(Operation.UPDATE)
                    .setColumns(
                        ColumnsChangeType.ANY,
                        listOf(
                            CLIENTS_OPTIONS.AUTO_OVERDRAFT_LIM,
                            CLIENTS_OPTIONS.DEBT,
                            CLIENTS_OPTIONS.STATUS_BALANCE_BANNED
                        )
                    )
                    .setMapper {
                        Mysql2GrutReplicationObject(
                            clientAutoOverdraftLimitChanged = it.getPrimaryKey(
                                CLIENTS_OPTIONS.CLIENT_ID
                            )
                        )
                    }
                    .build()
            )
            // Для расчета рестарта автобюджета нужно только знать, равно ли поле overdraft_limit нулю https://a.yandex-team.ru/arc/trunk/arcadia/direct/core/src/main/java/ru/yandex/direct/core/entity/campaign/AutoOverdraftUtils.java?rev=r9361848#L53
            // Поэтому, чтобы избежать лишних пересчетов кампаний, следим только за изменениями с нуля или на ноль
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CLIENTS_OPTIONS)
                    .setOperation(Operation.UPDATE)
                    .setColumn(CLIENTS_OPTIONS.OVERDRAFT_LIM)
                    .setValuesFilter { p ->
                        p.getBefore<BigDecimal, BigDecimal>(CLIENTS_OPTIONS.OVERDRAFT_LIM) == BigDecimal.ZERO
                            || p.getAfter<BigDecimal, BigDecimal>(CLIENTS_OPTIONS.OVERDRAFT_LIM) == BigDecimal.ZERO
                    }
                    .setMapper {
                        Mysql2GrutReplicationObject(
                            clientAutoOverdraftLimitChanged = it.getPrimaryKey(
                                CLIENTS_OPTIONS.CLIENT_ID
                            )
                        )
                    }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CLIENT_NDS)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapToClientObjectWithPrimaryKey(it, CLIENT_NDS.CLIENT_ID) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CLIENT_NDS)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapToClientObjectWithPrimaryKey(it, CLIENT_NDS.CLIENT_ID) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CLIENT_NDS)
                    .setOperation(Operation.UPDATE)
                    .setMapper { mapToClientObjectWithPrimaryKey(it, CLIENT_NDS.CLIENT_ID) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(INTERNAL_AD_PRODUCTS)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapToClientObjectWithPrimaryKey(it, INTERNAL_AD_PRODUCTS.CLIENT_ID) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(INTERNAL_AD_PRODUCTS)
                    .setOperation(Operation.UPDATE)
                    .setColumn(INTERNAL_AD_PRODUCTS.PRODUCT_NAME)
                    .setMapper { mapToClientObjectWithPrimaryKey(it, INTERNAL_AD_PRODUCTS.CLIENT_ID) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(INTERNAL_AD_PRODUCTS)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapToClientObjectWithPrimaryKey(it, INTERNAL_AD_PRODUCTS.CLIENT_ID) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMPAIGNS)
                    .setOperation(Operation.INSERT)
                    .setValuesFilter { isBillingAggregateCampaign(it) }
                    .setMapper { mapCampaignToClientObject(it) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMPAIGNS)
                    .setOperation(Operation.DELETE)
                    .setValuesFilter { isBillingAggregateCampaign(it) }
                    .setMapper { mapCampaignToClientObject(it) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMPAIGNS)
                    .setOperation(Operation.UPDATE)
                    .setColumns(ColumnsChangeType.ANY, listOf(CAMPAIGNS.PRODUCT_ID, CAMPAIGNS.WALLET_CID))
                    .setValuesFilter { isBillingAggregateCampaign(it) }
                    .setMapper { mapCampaignToClientObject(it) }
                    .build()
            )

            // campaigns
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMPAIGNS)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapToCampaignObject(it) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMPAIGNS)
                    .setOperation(Operation.UPDATE)
                    .setColumns(
                        ColumnsChangeType.ANY_EXCEPT,
                        listOf(
                            CAMPAIGNS.STATUS_BS_SYNCED,
                            CAMPAIGNS.SHOWS, CAMPAIGNS.SUM_SPENT_UNITS,
                            CAMPAIGNS.LAST_SHOW_TIME,
                            CAMPAIGNS.LAST_CHANGE, CAMPAIGNS.SUM_SPENT
                        )
                    )
                    .setMapper { mapToCampaignObject(it) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMPAIGNS)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapToCampaignObject(it) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMPAIGNS_CPM_PRICE)
                    .setOperation(Operation.UPDATE)
                    .setMapper { Mysql2GrutReplicationObject(campaignId = it.getLongPrimaryKey(CAMPAIGNS_CPM_PRICE.CID)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMPAIGNS_CPM_YNDX_FRONTPAGE)
                    .setOperation(Operation.UPDATE)
                    .setMapper { Mysql2GrutReplicationObject(campaignId = it.getLongPrimaryKey(CAMPAIGNS_CPM_YNDX_FRONTPAGE.CID)) }
                    .build()
            )

            /**
             * Следим за изменениями в таблице рестартов автобюджета, так как некоторые ресчеты рестарта не удалось хорошо реализовать в репликации
             * Например, при изменении sum_spent на кампании, и при этом изменении закончились все деньги на кошельке, то надо переотправить все кампании, привязанные к этому кошельку
             * Для старого транспорта эту работу делает ActiveOrdersImportJob, в репликации такое поддержать будет достаточно дорого
             * Аналогичная ситуация при пополнении кошелька
             *
             * В эти update будут попадать и update от самой репликации, то есть некоторые кампании будут 2 раза отправляться
             * Если это будет сильно нагружать(что вряд ли), то стоит тут отфильтровать по сервису, который сделал обновлние
             */
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMP_AUTOBUDGET_RESTART)
                    .setOperation(Operation.UPDATE)
                    .setColumns(
                        ColumnsChangeType.ANY,
                        listOf(CAMP_AUTOBUDGET_RESTART.RESTART_TIME, CAMP_AUTOBUDGET_RESTART.SOFT_RESTART_TIME)
                    )
                    .setMapper { mapToCampaignObjectWithPrimary(it, CAMP_AUTOBUDGET_RESTART.CID) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(WALLET_CAMPAIGNS)
                    .setOperation(Operation.INSERT)
                    .setMapper { Mysql2GrutReplicationObject(campaignId = it.getLongPrimaryKey(WALLET_CAMPAIGNS.WALLET_CID)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(WALLET_CAMPAIGNS)
                    .setOperation(Operation.UPDATE)
                    .setColumns(
                        ColumnsChangeType.ANY,
                        listOf(WALLET_CAMPAIGNS.IS_SUM_AGGREGATED)
                    )
                    .setMapper { Mysql2GrutReplicationObject(campaignId = it.getLongPrimaryKey(WALLET_CAMPAIGNS.WALLET_CID)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(WIDGET_PARTNER_CAMPAIGNS)
                    .setOperation(Operation.UPDATE)
                    .setMapper { Mysql2GrutReplicationObject(campaignId = it.getLongPrimaryKey(WIDGET_PARTNER_CAMPAIGNS.CID)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMP_OPTIONS)
                    .setOperation(Operation.UPDATE)
                    .setMapper { mapToCampaignObjectWithPrimary(it, CAMP_OPTIONS.CID) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMPAIGNS_INTERNAL)
                    .setOperation(Operation.UPDATE)
                    .setMapper { mapCampaignToClientObject(it) }
                    .build()
            )

            addRulesToHandler(
                tableChangesHandler,
                CAMP_ADDITIONAL_TARGETINGS,
                { ch -> Mysql2GrutReplicationObject(campaignId = ch.getLongBeforeOrAfter(CAMP_ADDITIONAL_TARGETINGS.CID)) },
            )

            /**
             * Следим за выставлением orderType старым транспортом, в репликации не рассчитываем
             * см. https://st.yandex-team.ru/DIRECT-170138
             */
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(CAMP_ORDER_TYPES)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapToCampaignObjectWithPrimary(it, CAMP_ORDER_TYPES.CID) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(PHRASES)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapToAdGroupObject(it) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(PHRASES)
                    .setOperation(Operation.UPDATE)
                    .setColumns(
                        ColumnsChangeType.ANY_EXCEPT, listOf(
                            PHRASES.LAST_CHANGE,
                            PHRASES.STATUS_MODERATE,
                            PHRASES.STATUS_POST_MODERATE,
                            PHRASES.STATUS_BS_SYNCED,
                            PHRASES.STATUS_SHOWS_FORECAST,
                            PHRASES.FORECAST_DATE,
                            PHRASES.STATUS_AUTOBUDGET_SHOW,
                            PHRASES.IS_BS_RARELY_LOADED,
                        )
                    )
                    .setMapper { mapToAdGroupObject(it) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(PHRASES)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapToAdGroupObject(it) }
                    .build()
            )

            // pageGroupTags, targetTags
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUP_BS_TAGS)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUP_BS_TAGS.PID) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUP_BS_TAGS)
                    .setOperation(Operation.UPDATE)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUP_BS_TAGS.PID) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUP_BS_TAGS)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUP_BS_TAGS.PID) }
                    .build()
            )

            //RelevanceMatch
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_DYNAMIC)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_DYNAMIC.PID) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_DYNAMIC)
                    .setOperation(Operation.UPDATE)
                    .setColumn(ADGROUPS_DYNAMIC.RELEVANCE_MATCH_CATEGORIES)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_DYNAMIC.PID) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_DYNAMIC)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_DYNAMIC.PID) }
                    .build()
            )

            // adGroup internalFields
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_INTERNAL)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_DYNAMIC.PID) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_INTERNAL)
                    .setOperation(Operation.UPDATE)
                    .setColumns(
                        ColumnsChangeType.ANY,
                        listOf(
                            ADGROUPS_INTERNAL.LEVEL,
                            ADGROUPS_INTERNAL.RF,
                            ADGROUPS_INTERNAL.RF_RESET,
                            ADGROUPS_INTERNAL.MAX_CLICKS_COUNT,
                            ADGROUPS_INTERNAL.MAX_CLICKS_PERIOD,
                            ADGROUPS_INTERNAL.MAX_STOPS_COUNT,
                            ADGROUPS_INTERNAL.MAX_STOPS_PERIOD
                        )
                    )
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_DYNAMIC.PID) }
                    .build()
            )
            addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_INTERNAL)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_DYNAMIC.PID) }
                    .build()
            )

            // minus-phrases
            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(MINUS_WORDS)
                    .setOperation(Operation.UPDATE)
                    .setColumn(MINUS_WORDS.MW_TEXT)
                    .setMapper { mapToMinusWordsObject(it) }
                    .build()
            )

            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(MINUS_WORDS)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapToMinusWordsObject(it) }
                    .build()
            )

            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(MINUS_WORDS)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapToMinusWordsObject(it, true) }
                    .build()
            )

            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_MINUS_WORDS)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_MINUS_WORDS.PID) }
                    .build()
            )

            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_MINUS_WORDS)
                    .setOperation(Operation.UPDATE)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_MINUS_WORDS.PID) }
                    .build()
            )

            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_MINUS_WORDS)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_MINUS_WORDS.PID) }
                    .build()
            )

            //mobile_content
            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(MOBILE_CONTENT)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapToMobileContentObject(it) }
                    .build()
            )

            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(MOBILE_CONTENT)
                    .setOperation(Operation.UPDATE)
                    .setColumns(
                        ColumnsChangeType.ANY,
                        listOf(
                            MOBILE_CONTENT.NAME,
                            MOBILE_CONTENT.OS_TYPE,
                            MOBILE_CONTENT.BUNDLE_ID,
                            MOBILE_CONTENT.STORE_CONTENT_ID,
                            MOBILE_CONTENT.MIN_OS_VERSION
                        )
                    )
                    .setMapper { mapToMobileContentObject(it) }
                    .build()
            )

            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(MOBILE_CONTENT)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapToMobileContentObject(it, isDeleted = true) }
                    .build()
            )

            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUP_ADDITIONAL_TARGETINGS)
                    .setOperation(Operation.INSERT)
                    .setValuesFilter(AdGroupShowConditionsFilter.filterUnknownTargetingType)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUP_ADDITIONAL_TARGETINGS.PID) }
                    .build()
            )
            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUP_ADDITIONAL_TARGETINGS)
                    .setOperation(Operation.DELETE)
                    .setValuesFilter(AdGroupShowConditionsFilter.filterUnknownTargetingType)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUP_ADDITIONAL_TARGETINGS.PID) }
                    .build()
            )
            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUP_ADDITIONAL_TARGETINGS)
                    .setOperation(Operation.UPDATE)
                    .setValuesFilter(AdGroupShowConditionsFilter.filterUnknownTargetingType)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUP_ADDITIONAL_TARGETINGS.PID) }
                    .build()
            )

            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_MOBILE_CONTENT)
                    .setOperation(Operation.INSERT)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_MOBILE_CONTENT.PID) }
                    .build()
            )

            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_MOBILE_CONTENT)
                    .setOperation(Operation.DELETE)
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_MOBILE_CONTENT.PID) }
                    .build()
            )

            tableChangesHandler.addTableChange(
                TableChange.Builder<Mysql2GrutReplicationObject>()
                    .setTable(ADGROUPS_MOBILE_CONTENT)
                    .setOperation(Operation.UPDATE)
                    .setColumns(
                        ColumnsChangeType.ANY, listOf(
                            ADGROUPS_MOBILE_CONTENT.MIN_OS_VERSION,
                            ADGROUPS_MOBILE_CONTENT.NETWORK_TARGETING,
                            ADGROUPS_MOBILE_CONTENT.DEVICE_TYPE_TARGETING,
                        )
                    )
                    .setMapper { mapAdGroupWithPrimaryKey(it, ADGROUPS_MOBILE_CONTENT.PID) }
                    .build()
            )

            // vcards
            tableChangesHandler.apply {
                addTableChange(
                    TableChange.Builder<Mysql2GrutReplicationObject>()
                        .setTable(VCARDS)
                        .setOperation(Operation.INSERT)
                        .setMapper { mapToVcard(it) }
                        .build()
                )

                addTableChange(
                    TableChange.Builder<Mysql2GrutReplicationObject>()
                        .setTable(VCARDS)
                        .setOperation(Operation.DELETE)
                        .setMapper { mapToVcard(it) }
                        .build()
                )

                addTableChange(
                    TableChange.Builder<Mysql2GrutReplicationObject>()
                        .setTable(ADDRESSES)
                        .setOperation(Operation.UPDATE)
                        .setMapper {
                            Mysql2GrutReplicationObject(
                                vcardAddressId = it.getPrimaryKey(ADDRESSES.AID)
                            )
                        }
                        .build()
                )
            }

            // strategies
            tableChangesHandler.apply {
                addTableChange(
                    TableChange.Builder<Mysql2GrutReplicationObject>()
                        .setTable(STRATEGIES)
                        .setOperation(Operation.INSERT)
                        .setMapper { Mysql2GrutReplicationObject(strategyId = it.getLongPrimaryKey(STRATEGIES.STRATEGY_ID)) }
                        .build()
                )

                addTableChange(
                    TableChange.Builder<Mysql2GrutReplicationObject>()
                        .setTable(STRATEGIES)
                        .setOperation(Operation.UPDATE)
                        .setMapper { Mysql2GrutReplicationObject(strategyId = it.getLongPrimaryKey(STRATEGIES.STRATEGY_ID)) }
                        .build()
                )
            }


            tableChangesHandler.apply {
                addTableChange(
                    TableChange.Builder<Mysql2GrutReplicationObject>()
                        .setTable(PERF_CREATIVES)
                        .setOperation(Operation.INSERT)
                        .setMapper { mapToCreative(it) }
                        .build()
                )

                addTableChange(
                    TableChange.Builder<Mysql2GrutReplicationObject>()
                        .setTable(PERF_CREATIVES)
                        .setOperation(Operation.UPDATE)
                        .setMapper { mapToCreative(it) }
                        .build()
                )

                addTableChange(
                    TableChange.Builder<Mysql2GrutReplicationObject>()
                        .setTable(PERF_CREATIVES)
                        .setOperation(Operation.DELETE)
                        .setMapper { mapToCreative(it) }
                        .build()
                )
            }
        }
        // Multipliers
        addMultipliersTableChanged(tableChangesHandler)

        // Page Blocks
        addRulesToHandler(
            tableChangesHandler,
            ADGROUP_PAGE_TARGETS,
            { ch -> mapAdGroupWithPrimaryKey(ch, ADGROUP_PAGE_TARGETS.PID) },
            listOf(ADGROUP_PAGE_TARGETS.PAGE_BLOCKS)
        )

        addBannersTableChanged(tableChangesHandler)

        addBiddableShowConditionsTablesChanged(tableChangesHandler)

        addRulesToHandler(
            tableChangesHandler,
            RETARGETING_CONDITIONS,
            { change ->
                Mysql2GrutReplicationObject(
                    retargetingConditionId = change.getLongPrimaryKey(RETARGETING_CONDITIONS.RET_COND_ID),
                    isDeleted = change.operation == Operation.DELETE,
                )
            },
            listOf(
                RETARGETING_CONDITIONS.CONDITION_NAME,
                RETARGETING_CONDITIONS.CONDITION_DESC,
                RETARGETING_CONDITIONS.CONDITION_JSON,
                RETARGETING_CONDITIONS.IS_DELETED,
            ),
        )

        addRulesToHandler(
            tableChangesHandler,
            RETARGETING_GOALS,
            { change ->
                Mysql2GrutReplicationObject(
                    retargetingConditionId = change.getLongPrimaryKey(RETARGETING_GOALS.RET_COND_ID),
                )
            },
            listOf(
                RETARGETING_GOALS.GOAL_ID,
                RETARGETING_GOALS.IS_ACCESSIBLE,
                RETARGETING_GOALS.GOAL_TYPE,
                RETARGETING_GOALS.GOAL_SOURCE,
            )
        )
    }

    private fun isBillingAggregateCampaign(p: ProceededChange): Boolean {
        val type: String = p.getBeforeOrAfter(CAMPAIGNS.TYPE)
        return CampaignsType.billing_aggregate.literal.equals(type)
    }

    private fun mapToCreative(change: ProceededChange): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(
            creativeId = change.getLongPrimaryKey(PERF_CREATIVES.CREATIVE_ID),
            isDeleted = change.operation == Operation.DELETE
        )
    }

    override fun mapBinlogEvent(binlogEvent: BinlogEvent): List<Mysql2GrutReplicationObject> {
        return tableChangesHandler.processChanges(binlogEvent)
    }

    private fun mapCampaignToClientObject(change: ProceededChange): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(clientId = change.getLongBeforeOrAfter(CAMPAIGNS.CLIENT_ID))
    }

    private fun mapToClientObject(change: ProceededChange): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(
            clientId = change.getLongPrimaryKey(CLIENTS.CLIENT_ID),
            isDeleted = change.operation == Operation.DELETE
        )
    }

    private fun mapToClientObjectWithPrimaryKey(
        change: ProceededChange,
        primaryKeyField: Field<Long>
    ): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(
            clientId = change.getLongPrimaryKey(primaryKeyField)
        )
    }

    private fun mapToCampaignObject(change: ProceededChange): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(
            campaignId = change.getLongPrimaryKey(CAMPAIGNS.CID),
            isDeleted = change.operation == Operation.DELETE
        )
    }

    private fun mapToCampaignObjectWithPrimary(
        change: ProceededChange,
        fieldName: Field<Long>
    ): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(campaignId = change.getLongPrimaryKey(fieldName))
    }

    private fun mapToAdGroupObject(change: ProceededChange): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(
            adGroupId = change.getLongPrimaryKey(PHRASES.PID),
            isDeleted = change.operation == Operation.DELETE
        )
    }

    private fun mapAdGroupWithPrimaryKey(
        change: ProceededChange,
        fieldName: Field<Long>
    ): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(adGroupId = change.getPrimaryKey(fieldName))
    }

    private fun mapToMinusWordsObject(
        change: ProceededChange,
        isDeleted: Boolean = false
    ): Mysql2GrutReplicationObject {
        val minusWordId: Long = change.getPrimaryKey(MINUS_WORDS.MW_ID)
        return Mysql2GrutReplicationObject(minusPhraseId = minusWordId, isDeleted = isDeleted)
    }

    private fun mapToMobileContentObject(
        change: ProceededChange,
        isDeleted: Boolean = false
    ): Mysql2GrutReplicationObject {
        val mobileContentId: Long = change.getPrimaryKey(MOBILE_CONTENT.MOBILE_CONTENT_ID)
        return Mysql2GrutReplicationObject(mobileContentId = mobileContentId, isDeleted = isDeleted)
    }

    private fun mapToVcard(change: ProceededChange): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(
            vcardId = change.getPrimaryKey(VCARDS.VCARD_ID),
            isDeleted = change.operation == Operation.DELETE
        )
    }

    private fun mapToMultiplier(
        change: ProceededChange,
        type: BidModifierTableType,
        keyField: Field<Long>
    ): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(
            bidModifierId = change.getLongPrimaryKey(keyField),
            bidModifierTableType = type,
            isDeleted = change.operation == Operation.DELETE,
        )
    }

    private fun mapToBiddableShowCondition(
        change: ProceededChange,
        primaryKey: Long,
        type: BiddableShowConditionChangeType
    ): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(
            biddableShowConditionId = primaryKey,
            biddableShowConditionType = type,
            isDeleted = change.operation == Operation.DELETE,
        )
    }

    val BIDS_BASE_TYPE_MAPPING = mapOf(
        BidsBaseBidType.relevance_match.literal to BiddableShowConditionChangeType.RELEVANCE_MATCH,
        BidsBaseBidType.offer_retargeting.literal to BiddableShowConditionChangeType.OFFER_RETARGETING
    )

    private fun mapBidsBaseToBiddableShowCondition(change: ProceededChange): Mysql2GrutReplicationObject {
        val bidTypeStr = change.getBeforeOrAfter<String, BidsBaseBidType>(BIDS_BASE.BID_TYPE)
        val type = BIDS_BASE_TYPE_MAPPING[bidTypeStr]
        return mapToBiddableShowCondition(change, change.getLongPrimaryKey(BIDS_BASE.BID_ID), type!!)
    }

    private fun mapToHrefParams(change: ProceededChange): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(
            biddableShowConditionId = change.getLongPrimaryKey(BIDS_HREF_PARAMS.ID),
            biddableShowConditionType = BiddableShowConditionChangeType.HREF_PARAM,
        )
    }

    /**
     * Добавляет правила отслеживания изменений в таблицах уточнений корректировок(multiplier_values)
     * для репликации связей группы-корректировки и кампании-корректировки;
     * отслеживается отдельно, чтобы уменьшить количество обрабатываемых событий в репликации групп и кампаний
     *
     * Для values-таблиц нас интересуют только insert-операции: update не приводят к изменению связей
     * delete отреплицируются автоматически в связи групп/кампаний при удалении корректировок из Грута
     */
    private fun mapBidModifierValueRelation(table: TableImpl<*>, hierarchicalMultiplierField: Field<Long>) {
        tableChangesHandler.addTableChange(
            TableChange.Builder<Mysql2GrutReplicationObject>()
                .setTable(table)
                .setOperation(Operation.INSERT)
                .setMapper { ch ->
                    Mysql2GrutReplicationObject(
                        bidModifierIdForRelation = ch.getLongBeforeOrAfter(hierarchicalMultiplierField),
                        bidModifierTableType = BidModifierTableType.PARENT,
                    )
                }
                .build()
        )
    }

    private fun addBiddableShowConditionsTablesChanged(tableChangesHandler: TableChangesHandler<Mysql2GrutReplicationObject>) {
        // Keywords
        addRulesToHandler(
            tableChangesHandler,
            BIDS,
            { ch ->
                mapToBiddableShowCondition(
                    ch,
                    ch.getLongPrimaryKey(BIDS.ID),
                    BiddableShowConditionChangeType.KEYWORD
                )
            },
            listOf(
                BIDS.PRICE,
                BIDS.PRICE_CONTEXT,
                BIDS.STATUS_MODERATE,
                BIDS.IS_SUSPENDED,
                BIDS.SHOWS_FORECAST,
                BIDS.PHRASE
            )
        )
        // href params used for biddable_show_condition keywords (bids table) and relevance match (bids_base table)
        addRulesToHandler(
            tableChangesHandler,
            BIDS_HREF_PARAMS,
            { ch -> mapToHrefParams(ch) },
            listOf(
                BIDS_HREF_PARAMS.PARAM1,
                BIDS_HREF_PARAMS.PARAM2,
            )
        )
        // adGroup -> keyword relation
        addRulesToHandler(
            tableChangesHandler,
            BIDS,
            { ch -> Mysql2GrutReplicationObject(adGroupId = ch.getLongBeforeOrAfter(BIDS.PID)) },
            operations = setOf(Operation.INSERT, Operation.DELETE)
        )

        //retargeting
        addRulesToHandler(
            tableChangesHandler,
            BIDS_RETARGETING,
            { ch ->
                mapToBiddableShowCondition(
                    ch,
                    ch.getPrimaryKey(BIDS_RETARGETING.RET_ID),
                    BiddableShowConditionChangeType.RETARGETING
                )
            },
            listOf(
                BIDS_RETARGETING.PRICE_CONTEXT,
                BIDS_RETARGETING.IS_SUSPENDED,
            )
        )
        // adGroup -> retargeting relation
        addRulesToHandler(
            tableChangesHandler,
            BIDS_RETARGETING,
            { ch -> Mysql2GrutReplicationObject(adGroupId = ch.getLongBeforeOrAfter(BIDS_RETARGETING.PID)) },
            operations = setOf(Operation.INSERT, Operation.DELETE)
        )

        // RELEVANCE_MATCH and OFFER_RETARGETING
        addRulesToHandler(
            tableChangesHandler,
            BIDS_BASE,
            { change -> mapBidsBaseToBiddableShowCondition(change) },
            listOf(
                BIDS_BASE.PRICE,
                BIDS_BASE.PRICE_CONTEXT,
                BIDS_BASE.OPTS,
                BIDS_BASE.RELEVANCE_MATCH_CATEGORIES
            ),
            { change: ProceededChange ->
                val bidsBaseType: String = change.getBeforeOrAfter(BIDS_BASE.BID_TYPE)
                BIDS_BASE_TYPE_MAPPING.contains(bidsBaseType)
            }
        )
        // adGroup -> relevance_match, adGroup -> offer_retargeting
        addRulesToHandler(
            tableChangesHandler,
            BIDS_BASE,
            { ch -> Mysql2GrutReplicationObject(adGroupId = ch.getLongBeforeOrAfter(BIDS_BASE.PID)) },
            valuesFilter = { change: ProceededChange ->
                val bidsBaseType: String = change.getBeforeOrAfter(BIDS_BASE.BID_TYPE)
                BIDS_BASE_TYPE_MAPPING.contains(bidsBaseType)
            },
            operations = setOf(Operation.INSERT, Operation.DELETE)
        )

        // dynamic bids
        addRulesToHandler(
            tableChangesHandler,
            BIDS_DYNAMIC,
            { ch ->
                mapToBiddableShowCondition(
                    ch,
                    ch.getLongPrimaryKey(BIDS_DYNAMIC.DYN_ID),
                    BiddableShowConditionChangeType.DYNAMIC
                )
            },
            listOf(
                BIDS_DYNAMIC.DYN_COND_ID,
                BIDS_DYNAMIC.PRICE,
                BIDS_DYNAMIC.PRICE_CONTEXT,
                BIDS_DYNAMIC.AUTOBUDGET_PRIORITY,
                BIDS_DYNAMIC.OPTS,
                BIDS_DYNAMIC.FROM_TAB,
            )
        )
        // adGroup -> bidsDynamic
        addRulesToHandler(
            tableChangesHandler,
            BIDS_DYNAMIC,
            { ch -> Mysql2GrutReplicationObject(adGroupId = ch.getLongBeforeOrAfter(BIDS_DYNAMIC.PID)) },
            operations = setOf(Operation.INSERT, Operation.DELETE)
        )

        addRulesToHandler(
            tableChangesHandler,
            DYNAMIC_CONDITIONS,
            { ch -> mapToDynamicCondition(ch) },
            listOf(
                DYNAMIC_CONDITIONS.CONDITION_NAME,
                DYNAMIC_CONDITIONS.CONDITION_JSON
            )
        )

        //performance
        addRulesToHandler(
            tableChangesHandler,
            BIDS_PERFORMANCE,
            { ch ->
                mapToBiddableShowCondition(
                    ch,
                    ch.getLongPrimaryKey(BIDS_PERFORMANCE.PERF_FILTER_ID),
                    BiddableShowConditionChangeType.PERFORMANCE
                )
            },
            listOf(
                BIDS_PERFORMANCE.PRICE_CPA,
                BIDS_PERFORMANCE.PRICE_CPC,
                BIDS_PERFORMANCE.TARGET_FUNNEL,
                BIDS_PERFORMANCE.IS_SUSPENDED,
                BIDS_PERFORMANCE.IS_DELETED,
                // для поля HrefParams
                BIDS_PERFORMANCE.NAME
            )
        )
        // adGroup -> bids_performance
        addRulesToHandler(
            tableChangesHandler,
            BIDS_PERFORMANCE,
            { ch -> Mysql2GrutReplicationObject(adGroupId = ch.getLongBeforeOrAfter(BIDS_PERFORMANCE.PID)) },
            operations = setOf(Operation.INSERT, Operation.DELETE)
        )
    }

    /**
     * Добавляет правила отслеживания изменений в таблицах корректировок для репликации корректировок в грут
     * (bid_modifiers)
     */
    private fun addMultipliersTableChanged(tableChangesHandler: TableChangesHandler<Mysql2GrutReplicationObject>) {
        addRulesToHandler(
            tableChangesHandler,
            HIERARCHICAL_MULTIPLIERS,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.PARENT,
                    HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID
                )
            },
            listOf(HIERARCHICAL_MULTIPLIERS.MULTIPLIER_PCT, HIERARCHICAL_MULTIPLIERS.IS_ENABLED)
        )

        // При событии на hierarhical_multipliers надо пересчитать связи из групп и кампаний
        addRulesToHandler(
            tableChangesHandler,
            HIERARCHICAL_MULTIPLIERS,
            { ch ->
                val pid = ch.getLongBeforeOrAfter(HIERARCHICAL_MULTIPLIERS.PID)
                when {
                    pid != null -> Mysql2GrutReplicationObject(adGroupId = pid)
                    else -> Mysql2GrutReplicationObject(
                        campaignId = ch.getLongBeforeOrAfter(HIERARCHICAL_MULTIPLIERS.CID)
                    )
                }
            },
            listOf(HIERARCHICAL_MULTIPLIERS.IS_ENABLED)
        )

        addRulesToHandler(
            tableChangesHandler,
            BANNER_TYPE_MULTIPLIER_VALUES,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.BANNER_TYPE,
                    BANNER_TYPE_MULTIPLIER_VALUES.BANNER_TYPE_MULTIPLIER_VALUE_ID
                )
            },
            listOf(BANNER_TYPE_MULTIPLIER_VALUES.MULTIPLIER_PCT)
        )
        mapBidModifierValueRelation(
            BANNER_TYPE_MULTIPLIER_VALUES,
            BANNER_TYPE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID
        )

        addRulesToHandler(
            tableChangesHandler,
            DEMOGRAPHY_MULTIPLIER_VALUES,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.DEMOGRAPHY,
                    DEMOGRAPHY_MULTIPLIER_VALUES.DEMOGRAPHY_MULTIPLIER_VALUE_ID
                )
            },
            listOf(
                DEMOGRAPHY_MULTIPLIER_VALUES.MULTIPLIER_PCT,
                DEMOGRAPHY_MULTIPLIER_VALUES.GENDER,
                DEMOGRAPHY_MULTIPLIER_VALUES.AGE,
            )
        )
        mapBidModifierValueRelation(
            DEMOGRAPHY_MULTIPLIER_VALUES,
            DEMOGRAPHY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID
        )

        addRulesToHandler(
            tableChangesHandler,
            EXPRESSION_MULTIPLIER_VALUES,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.EXPRESSION,
                    EXPRESSION_MULTIPLIER_VALUES.EXPRESSION_MULTIPLIER_VALUE_ID
                )
            },
            listOf(
                EXPRESSION_MULTIPLIER_VALUES.MULTIPLIER_PCT,
                EXPRESSION_MULTIPLIER_VALUES.CONDITION_JSON,
                EXPRESSION_MULTIPLIER_VALUES.MULTIPLIER_PCT,
            )
        )
        mapBidModifierValueRelation(
            EXPRESSION_MULTIPLIER_VALUES,
            EXPRESSION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID
        )

        addRulesToHandler(
            tableChangesHandler,
            INVENTORY_MULTIPLIER_VALUES,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.INVENTORY,
                    INVENTORY_MULTIPLIER_VALUES.INVENTORY_MULTIPLIER_VALUE_ID
                )
            },
            listOf(
                INVENTORY_MULTIPLIER_VALUES.MULTIPLIER_PCT,
                INVENTORY_MULTIPLIER_VALUES.INVENTORY_TYPE,
            )
        )
        mapBidModifierValueRelation(
            INVENTORY_MULTIPLIER_VALUES,
            INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID
        )


        addRulesToHandler(
            tableChangesHandler,
            MOBILE_MULTIPLIER_VALUES,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.MOBILE,
                    MOBILE_MULTIPLIER_VALUES.MOBILE_MULTIPLIER_VALUE_ID
                )
            },
            listOf(
                MOBILE_MULTIPLIER_VALUES.MULTIPLIER_PCT,
                MOBILE_MULTIPLIER_VALUES.OS_TYPE,
            )
        )
        mapBidModifierValueRelation(MOBILE_MULTIPLIER_VALUES, MOBILE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID)


        addRulesToHandler(
            tableChangesHandler,
            RETARGETING_MULTIPLIER_VALUES,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.RETARGETING,
                    RETARGETING_MULTIPLIER_VALUES.RETARGETING_MULTIPLIER_VALUE_ID
                )
            },
            listOf(
                RETARGETING_MULTIPLIER_VALUES.MULTIPLIER_PCT,
                RETARGETING_MULTIPLIER_VALUES.RET_COND_ID,
            )
        )
        mapBidModifierValueRelation(
            RETARGETING_MULTIPLIER_VALUES,
            RETARGETING_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID
        )

        addRulesToHandler(
            tableChangesHandler,
            AB_SEGMENT_MULTIPLIER_VALUES,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.AB_SEGMENT,
                    AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_MULTIPLIER_VALUE_ID,
                )
            },
            listOf(
                AB_SEGMENT_MULTIPLIER_VALUES.MULTIPLIER_PCT,
                AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_MULTIPLIER_VALUE_ID,
            )
        )
        mapBidModifierValueRelation(
            AB_SEGMENT_MULTIPLIER_VALUES,
            AB_SEGMENT_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID
        )

        addRulesToHandler(
            tableChangesHandler,
            TABLET_MULTIPLIER_VALUES,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.TABLET,
                    TABLET_MULTIPLIER_VALUES.TABLET_MULTIPLIER_VALUE_ID
                )
            },
            listOf(
                TABLET_MULTIPLIER_VALUES.MULTIPLIER_PCT,
                TABLET_MULTIPLIER_VALUES.OS_TYPE,
            )
        )
        mapBidModifierValueRelation(TABLET_MULTIPLIER_VALUES, TABLET_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID)


        addRulesToHandler(
            tableChangesHandler,
            TRAFARET_POSITION_MULTIPLIER_VALUES,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.TRAFARET_POSITION,
                    TRAFARET_POSITION_MULTIPLIER_VALUES.TRAFARET_POSITION_MULTIPLIER_VALUE_ID
                )
            },
            listOf(
                TRAFARET_POSITION_MULTIPLIER_VALUES.MULTIPLIER_PCT,
                TRAFARET_POSITION_MULTIPLIER_VALUES.TRAFARET_POSITION,
            )
        )
        mapBidModifierValueRelation(
            TRAFARET_POSITION_MULTIPLIER_VALUES,
            TRAFARET_POSITION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID
        )

        addRulesToHandler(
            tableChangesHandler,
            WEATHER_MULTIPLIER_VALUES,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.WEATHER,
                    WEATHER_MULTIPLIER_VALUES.WEATHER_MULTIPLIER_VALUE_ID
                )
            },
            listOf(
                WEATHER_MULTIPLIER_VALUES.MULTIPLIER_PCT,
                WEATHER_MULTIPLIER_VALUES.CONDITION_JSON,
            )
        )
        mapBidModifierValueRelation(WEATHER_MULTIPLIER_VALUES, WEATHER_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID)

        addRulesToHandler(
            tableChangesHandler,
            GEO_MULTIPLIER_VALUES,
            { ch ->
                mapToMultiplier(
                    ch,
                    BidModifierTableType.GEO,
                    GEO_MULTIPLIER_VALUES.GEO_MULTIPLIER_VALUE_ID
                )
            },
            listOf(
                GEO_MULTIPLIER_VALUES.MULTIPLIER_PCT,
                GEO_MULTIPLIER_VALUES.REGION_ID,
            )
        )
        mapBidModifierValueRelation(GEO_MULTIPLIER_VALUES, GEO_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID)
    }

    private fun mapToDynamicCondition(change: ProceededChange): Mysql2GrutReplicationObject {
        return Mysql2GrutReplicationObject(
            biddableShowConditionId = change.getLongPrimaryKey(DYNAMIC_CONDITIONS.DYN_COND_ID),
            biddableShowConditionType = BiddableShowConditionChangeType.DYNAMIC_CONDITION,
        )
    }

    private fun addBannersTableChanged(tableChangesHandler: TableChangesHandler<Mysql2GrutReplicationObject>) {
        tableChangesHandler.addTableChange(
            TableChange.Builder<Mysql2GrutReplicationObject>()
                .setTable(BANNERS)
                .setOperation(Operation.INSERT)
                .setMapper { Mysql2GrutReplicationObject(bannerId = it.getPrimaryKey(BANNERS.BID)) }
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<Mysql2GrutReplicationObject>()
                .setTable(BANNERS)
                .setOperation(Operation.UPDATE)
                .setColumns(
                    ColumnsChangeType.ANY_EXCEPT,
                    listOf(
                        BANNERS.STATUS_BS_SYNCED,
                        BANNERS.LAST_CHANGE,
                        BANNERS.BANNER_ID,
                        BANNERS.STATUS_POST_MODERATE
                    )
                )
                .setMapper { Mysql2GrutReplicationObject(bannerId = it.getPrimaryKey(BANNERS.BID)) }
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<Mysql2GrutReplicationObject>()
                .setTable(BANNERS)
                .setOperation(Operation.DELETE)
                .setMapper {
                    Mysql2GrutReplicationObject(
                        bannerId = it.getPrimaryKey(BANNERS.BID),
                        isDeleted = true
                    )
                }
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<Mysql2GrutReplicationObject>()
                .setTable(BANNER_IMAGES)
                .setOperation(Operation.INSERT)
                .setMapper { Mysql2GrutReplicationObject(bannerId = it.getBeforeOrAfter(BANNER_IMAGES.BID)) }
                .build()
        )
        tableChangesHandler.addTableChange(
            TableChange.Builder<Mysql2GrutReplicationObject>()
                .setTable(BANNER_IMAGES)
                .setOperation(Operation.UPDATE)
                .setMapper { Mysql2GrutReplicationObject(bannerId = it.getBeforeOrAfter(BANNER_IMAGES.BID)) }
                .build()
        )
    }
}
