package ru.yandex.direct.ess.router.rules.recomtracer.changehandlers;

import java.util.Objects;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.recommendation.RecommendationType;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecomTracerLogicObject;
import ru.yandex.direct.ess.router.rules.recomtracer.RecomTracerChangeHandler;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.binlog.model.Operation.DELETE;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createBannerChangeCanAffectedGroupLogicObject;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createCampaignChangeLogicObject;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createPhrasesChangeLogicObject;

@Component
public class ChangeAdGroupWithLowStatChangeHandler implements RecomTracerChangeHandler {
    @Override
    public RecommendationType supportedType() {
        return RecommendationType.changeAdGroupWithLowStat;
    }

    @Override
    public void initInterestingChanges(TableChangesHandler<RecomTracerLogicObject> tableChangesHandler) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS.ARCHIVED)
                        .setValuesFilter(ChangeHandlerHelper::isCampaignBecameArchived)
                        .setMapper(this::mapCampaigns)
                        .build());

        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(PHRASES)
                        .setOperation(UPDATE)
                        .setColumn(PHRASES.IS_BS_RARELY_LOADED)
                        .setValuesFilter(this::isPhrasesBsRarelyLoadedBeforeChanged)
                        .setMapper(this::mapPhrases)
                        .build());

        /* Отслеживаем архивность баннера - если все стали архивными - надо удалить рекомендацию */
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(BANNERS)
                        .setOperation(UPDATE)
                        .setColumn(BANNERS.STATUS_ARCH)
                        .setValuesFilter(ChangeHandlerHelper::isBannersStatusArchChanged)
                        .setMapper(this::mapBanners)
                        .build());

        /* Если при удалении баннера остались только ахивные - надо удалить рекомендацию */
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(BANNERS)
                        .setOperation(DELETE)
                        .setMapper(this::mapBanners)
                        .build());
    }

    private RecomTracerLogicObject mapCampaigns(ProceededChange change) {
        return createCampaignChangeLogicObject(supportedType().getId(), change);
    }

    private RecomTracerLogicObject mapBanners(ProceededChange change) {
        return createBannerChangeCanAffectedGroupLogicObject(supportedType().getId(), change);
    }

    private RecomTracerLogicObject mapPhrases(ProceededChange change) {
        return createPhrasesChangeLogicObject(supportedType().getId(), change);
    }

    // Если на группе был выключён флаг геноцида, то рекомендацию уже не нужно показывать
    private boolean isPhrasesBsRarelyLoadedBeforeChanged(ProceededChange change) {
        Long isBsRarelyLoadedBefore = change.getBefore(PHRASES.IS_BS_RARELY_LOADED);
        Long isBsRarelyLoadedAfter = change.getAfter(PHRASES.IS_BS_RARELY_LOADED);
        return (Objects.equals(isBsRarelyLoadedBefore, 1L) && Objects.equals(isBsRarelyLoadedAfter, 0L));
    }
}
