package ru.yandex.direct.ess.router.rules.recomtracer.changehandlers;

import java.util.Objects;
import java.util.function.Function;

import org.jooq.TableField;

import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.ess.common.utils.TablesEnum;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecomTracerLogicObject;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecommendationKeyIdentifier;
import ru.yandex.direct.ess.router.utils.ProceededChange;

import static ru.yandex.direct.binlog.model.Operation.INSERT;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;

public class ChangeHandlerHelper {

    private ChangeHandlerHelper() {
    }


    public static RecomTracerLogicObject createBannerChangeLogicObject(long typeId, ProceededChange change) {
        if (change.getOperation() == INSERT || change.getOperation() == UPDATE) {
            return createBannerChangeLogicObject(typeId, change, change::getAfter);
        } else {
            return createBannerChangeLogicObject(typeId, change, change::getBefore);
        }
    }

    public static RecomTracerLogicObject createSimpleBannerChangeLogicObject(long typeId, long bid) {
        return new RecomTracerLogicObject.Builder()
                .withRecommendationTypeId(typeId)
                .withTableToLoad(TablesEnum.BANNERS)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.BID, bid)
                .withPrimaryKey(bid)
                .build();
    }

    public static RecomTracerLogicObject createPhrasesChangeLogicObject(long typeId, ProceededChange change) {
        if (change.getOperation() == INSERT || change.getOperation() == UPDATE) {
            return createPhrasesChangeLogicObject(typeId, change, change::getAfter);
        } else {
            return createPhrasesChangeLogicObject(typeId, change, change::getBefore);
        }
    }


    public static RecomTracerLogicObject createCampaignChangeLogicObject(long typeId, ProceededChange change) {
        if (change.getOperation() == INSERT || change.getOperation() == UPDATE) {
            return createCampaignChangeLogicObject(typeId, change, change::getAfter);
        } else {
            return createCampaignChangeLogicObject(typeId, change, change::getBefore);
        }
    }


    /**
     * Если изменение в таблице BANNERS может повлиять на группу - дополнительные свойства из mysql должны быть взяты
     * из таблицы PHRASES
     */
    public static RecomTracerLogicObject createBannerChangeCanAffectedGroupLogicObject(long typeId,
                                                                                       ProceededChange change) {
        if (change.getOperation() == INSERT || change.getOperation() == UPDATE) {
            return createBannerChangeCanAffectedGroupLogicObject(typeId, change, change::getAfter);
        } else {
            return createBannerChangeCanAffectedGroupLogicObject(typeId, change, change::getBefore);
        }
    }

    public static boolean isCampaignBecameArchived(ProceededChange change) {
        if (change.beforeContains(CAMPAIGNS.ARCHIVED) && change.afterContains(CAMPAIGNS.ARCHIVED)) {
            String archivedBefore = change.getBefore(CAMPAIGNS.ARCHIVED);
            String archivedAfter = change.getAfter(CAMPAIGNS.ARCHIVED);
            return ("Yes".equals(archivedAfter) && "No".equals(archivedBefore));
        }
        return false;
    }

    // Если баннер был заархивирован, то возможно, вся группа стала архивной (это случается, если
    // все баннеры группы стали архивными). И нам нужно будет отменить все рекомендации для этой группы.
    // Но чтобы выяснить достоверно, все ли баннеры стали архивными, нужно выполнить отдельный запрос про группу
    public static boolean isBannersStatusArchChanged(ProceededChange change) {
        if (change.beforeContains(Tables.BANNERS.STATUS_ARCH) && change.afterContains(Tables.BANNERS.STATUS_ARCH)) {
            String statusArchBefore = change.getBefore(BANNERS.STATUS_ARCH);
            String statusArchAfter = change.getAfter(Tables.BANNERS.STATUS_ARCH);
            return "No".equals(statusArchBefore) && "Yes".equals(statusArchAfter);
        }
        return false;
    }

    public static boolean isBannerStatusModerateChanged(ProceededChange change) {
        if (change.beforeContains(BANNERS.STATUS_MODERATE) && change.afterContains(BANNERS.STATUS_MODERATE)) {
            String statusModerateBefore = change.getBefore(BANNERS.STATUS_MODERATE);
            String statusModerateAfter = change.getAfter(BANNERS.STATUS_MODERATE);
            return !Objects.equals(statusModerateBefore, statusModerateAfter);
        }
        return false;
    }

    private static RecomTracerLogicObject createBannerChangeLogicObject(long typeId, ProceededChange change,
                                                                        Function<TableField<?, Long>, Long> extractor) {
        Long bid = change.getPrimaryKey(BANNERS.BID);
        Long pid = extractor.apply(BANNERS.PID);
        Long cid = extractor.apply(BANNERS.CID);
        return new RecomTracerLogicObject.Builder()
                .withRecommendationTypeId(typeId)
                .withTableToLoad(TablesEnum.CAMPAIGNS)
                .withPrimaryKey(cid)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.BID, bid)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.PID, pid)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.CID, cid)
                .build();
    }

    private static RecomTracerLogicObject createCampaignChangeLogicObject(long typeId, ProceededChange change,
                                                                          Function<TableField<?, Long>, Long> extractor) {
        Long cid = change.getPrimaryKey(CAMPAIGNS.CID);
        Long clientId = extractor.apply(CAMPAIGNS.CLIENT_ID);
        return new RecomTracerLogicObject.Builder()
                .withRecommendationTypeId(typeId)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.CID, cid)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.CLIENT_ID, clientId)
                .build();
    }

    private static RecomTracerLogicObject createPhrasesChangeLogicObject(long typeId, ProceededChange change,
                                                                         Function<TableField<?, Long>, Long> extractor) {
        Long pid = change.getPrimaryKey(PHRASES.PID);
        Long cid = extractor.apply(PHRASES.CID);
        return new RecomTracerLogicObject.Builder()
                .withRecommendationTypeId(typeId)
                .withTableToLoad(TablesEnum.CAMPAIGNS)
                .withPrimaryKey(cid)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.PID, pid)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.CID, cid)
                .build();
    }

    private static RecomTracerLogicObject createBannerChangeCanAffectedGroupLogicObject(long typeId,
                                                                                        ProceededChange change,
                                                                                        Function<TableField<?, Long>,
                                                                                                Long> extractor) {
        Long bid = change.getPrimaryKey(BANNERS.BID);
        Long pid = extractor.apply(BANNERS.PID);
        return new RecomTracerLogicObject.Builder()
                .withRecommendationTypeId(typeId)
                .withTableToLoad(TablesEnum.PHRASES)
                .withPrimaryKey(pid)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.PID, pid)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.BID, bid)
                .build();
    }
}
