package ru.yandex.direct.ess.router.rules.recomtracer.changehandlers;

import java.math.BigDecimal;
import java.util.Objects;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.recommendation.RecommendationType;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecomTracerLogicObject;
import ru.yandex.direct.ess.router.rules.recomtracer.RecomTracerChangeHandler;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createCampaignChangeLogicObject;

@Component
public class DailyBudgetTypeChangeHandler implements RecomTracerChangeHandler {


    @Override
    public RecommendationType supportedType() {
        return RecommendationType.dailyBudget;
    }

    @Override
    public void initInterestingChanges(TableChangesHandler<RecomTracerLogicObject> tableChangesHandler) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS.DAY_BUDGET)
                        .setValuesFilter(this::isCampaignDayBudgetChanged)
                        .setMapper(this::mapCampaigns)
                        .build());

        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS.STRATEGY_NAME)
                        .setValuesFilter(this::isCampaignStrategyChanged)
                        .setMapper(this::mapCampaigns)
                        .build());

        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS.ARCHIVED)
                        .setValuesFilter(ChangeHandlerHelper::isCampaignBecameArchived)
                        .setMapper(this::mapCampaigns)
                        .build());
    }

    private RecomTracerLogicObject mapCampaigns(ProceededChange change) {
        return createCampaignChangeLogicObject(supportedType().getId(), change);
    }


    private boolean isCampaignDayBudgetChanged(ProceededChange change) {
        if (change.beforeContains(CAMPAIGNS.DAY_BUDGET) && change.afterContains(CAMPAIGNS.DAY_BUDGET)) {
            BigDecimal dayBudgetBefore = change.getBefore(CAMPAIGNS.DAY_BUDGET);
            BigDecimal dayBudgetAfter = change.getAfter(CAMPAIGNS.DAY_BUDGET);
            return !Objects.equals(dayBudgetBefore, dayBudgetAfter);
        }
        return false;
    }

    private boolean isCampaignStrategyChanged(ProceededChange change) {
        if (change.beforeContains(CAMPAIGNS.STRATEGY_NAME) && change.afterContains(CAMPAIGNS.STRATEGY_NAME)) {
            String strategyNameBefore = change.getBefore(CAMPAIGNS.STRATEGY_NAME);
            String strategyNameAfter = change.getAfter(CAMPAIGNS.STRATEGY_NAME);
            return !Objects.equals(strategyNameBefore, strategyNameAfter);
        }
        return false;
    }
}
