package ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.autotargeting;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.recommendation.RecommendationType;
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType;
import ru.yandex.direct.ess.common.utils.TablesEnum;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecomTracerLogicObject;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecommendationKeyIdentifier;
import ru.yandex.direct.ess.router.rules.recomtracer.RecomTracerChangeHandler;
import ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.binlog.model.Operation.DELETE;
import static ru.yandex.direct.binlog.model.Operation.INSERT;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_BASE;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createBannerChangeCanAffectedGroupLogicObject;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createCampaignChangeLogicObject;

/**
 * <pre>
 * Для рекомендаций "Уберите площадку из запрещённых" отслеживаются следующие события:
 * 1) Архивация кампании — здесь как обычно убираем все рекомендации по префиксу (client_id, type, cid)
 * 2) Архивация баннера — если вся группа стала архивной — и на группу целиком по префиксу (client_id, type, cid, pid)
 * 3) Включение автотаргетинга из интерфейса. Обычно это выражается в том, что прилетает
 *    INSERT записи в таблицу bids_base, причём bids_type=relevance_match.
 *    В этом случае мы тоже убираем рекомендацию на всю группу (по префиксу).
 * </pre>
 */
@Component
public class SwitchOnAutotargetingChangeHandler implements RecomTracerChangeHandler {
    @Override
    public RecommendationType supportedType() {
        return RecommendationType.switchOnAutotargeting;
    }

    @Override
    public void initInterestingChanges(TableChangesHandler<RecomTracerLogicObject> tableChangesHandler) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS.ARCHIVED)
                        .setValuesFilter(ChangeHandlerHelper::isCampaignBecameArchived)
                        .setMapper(this::mapCampaigns)
                        .build());

        /* Отслеживаем архивность баннера - если все стали архивными - надо удалить рекомендацию */
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(BANNERS)
                        .setOperation(UPDATE)
                        .setColumn(BANNERS.STATUS_ARCH)
                        .setValuesFilter(ChangeHandlerHelper::isBannersStatusArchChanged)
                        .setMapper(this::mapBanners)
                        .build());

        /* Если при удалении баннера остались только ахивные - надо удалить рекомендацию */
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(BANNERS)
                        .setOperation(DELETE)
                        .setMapper(this::mapBanners)
                        .build());

        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(BIDS_BASE)
                        .setOperation(INSERT)
                        .setValuesFilter(this::isBidBaseWithRelevanceMatchType)
                        .setMapper(this::mapBidsBase)
                        .build());
    }


    private RecomTracerLogicObject mapCampaigns(ProceededChange change) {
        return createCampaignChangeLogicObject(supportedType().getId(), change);
    }

    private RecomTracerLogicObject mapBanners(ProceededChange change) {
        return createBannerChangeCanAffectedGroupLogicObject(supportedType().getId(), change);
    }

    private RecomTracerLogicObject mapBidsBase(ProceededChange change) {
        Long cid = change.getAfter(BIDS_BASE.CID);
        Long pid = change.getAfter(BIDS_BASE.PID);
        return new RecomTracerLogicObject.Builder()
                .withRecommendationTypeId(supportedType().getId())
                .withTableToLoad(TablesEnum.CAMPAIGNS)
                .withPrimaryKey(cid)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.PID, pid)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.CID, cid)
                .build();
    }

    private boolean isBidBaseWithRelevanceMatchType(ProceededChange change) {
        String bidType = change.getAfter(BIDS_BASE.BID_TYPE);
        return BidsBaseBidType.relevance_match.getLiteral().equals(bidType);
    }
}
