package ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.geolicense;

import java.util.Objects;

import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecomTracerLogicObject;
import ru.yandex.direct.ess.router.rules.recomtracer.RecomTracerChangeHandler;
import ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static ru.yandex.direct.binlog.model.Operation.DELETE;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_MINUS_GEO;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createBannerChangeCanAffectedGroupLogicObject;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createBannerChangeLogicObject;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createCampaignChangeLogicObject;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createPhrasesChangeLogicObject;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createSimpleBannerChangeLogicObject;

/**
 * <pre>
 * Геолицензирование. Эти рекомендаций привязаны к баннерам.
 * 1) При архивтировании кампании скрываем все рекомендации, привязанные к дочерним объектам этой кампании
 * 2) Аналогично при изменении statusShow ("Yes" → "No") и geo (любое изменение) на кампании
 * 3) На группе изменение статуса модерации (любое) и geo (любое) — аналогично, скрывает дочерние по отношению к этой
 * группе рекомендации
 * 4) Изменения полей на баннере. Одновременно с этим проверяется статус архивности группы. Если
 *    группа стала архивной, то будут скрыты все рекомендации на этой группе. В противном случае будет скрыта только
 *    отдельная рекомендация на изменённом баннере.
 * </pre>
 */
public abstract class BaseLicensesChangeHandler implements RecomTracerChangeHandler {

    @Override
    public void initInterestingChanges(TableChangesHandler<RecomTracerLogicObject> tableChangesHandler) {
        initCampaignsChanges(tableChangesHandler);
        initPhrasesChanges(tableChangesHandler);
        initBannersChanges(tableChangesHandler);
        initBannersMinusGeoChanges(tableChangesHandler);
    }

    private void initCampaignsChanges(TableChangesHandler<RecomTracerLogicObject> tableChangesHandler) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS.ARCHIVED)
                        .setValuesFilter(ChangeHandlerHelper::isCampaignBecameArchived)
                        .setMapper(this::mapCampaigns)
                        .build());

        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS.STATUS_SHOW)
                        .setValuesFilter(this::isCampaignStatusShowChanged)
                        .setMapper(this::mapCampaigns)
                        .build());

        // geo поле текстовое, а бинлоги у нас включены в режиме NOBLOB, поэтому значение before не придёт
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS.GEO)
                        .setMapper(this::mapCampaigns)
                        .build());
    }

    private void initPhrasesChanges(TableChangesHandler<RecomTracerLogicObject> tableChangesHandler) {
        // При удалении группы мы должны загрузить ClientID этой кампании и
        // удалить все рекомендации по префиксу (client_id, type, cid, pid)
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(Tables.PHRASES)
                        .setOperation(DELETE)
                        .setMapper(this::mapPhrases)
                        .build());

        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(Tables.PHRASES)
                        .setOperation(UPDATE)
                        .setColumn(PHRASES.STATUS_MODERATE)
                        .setValuesFilter(this::isPhrasesStatusModerateChanged)
                        .setMapper(this::mapPhrases)
                        .build());

        // geo - колонка с типом text, а бинлоги у нас включены в режиме NOBLOB, поэтому значение before не
        // придёт
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(Tables.PHRASES)
                        .setOperation(UPDATE)
                        .setColumn(PHRASES.GEO)
                        .setMapper(this::mapPhrases)
                        .build());
    }

    private void initBannersChanges(TableChangesHandler<RecomTracerLogicObject> tableChangesHandler) {
        /*
         * Если баннер удалился, в logicProcessor'е делается дополнительная проверка, что не все оставшиеся баннеры
         * на группе являются архивными
         * Такое поведение актуально только для рекомендаций на группе, но по сколько для всех удаленных баннеров
         * будет делаться запрос в mysql для проверки (из-за SwitchOnAutotargetingChangeHandler), то не будем делать
         * еще один запрос для таких событый, только без проверки архивности группы
         */
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(BANNERS)
                        .setOperation(DELETE)
                        .setMapper(this::mapBannersDeleteOrBecameArchive)
                        .build());

        /*
         * Аналогично удаленным баннерам
         */
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(BANNERS)
                        .setOperation(UPDATE)
                        .setColumn(BANNERS.STATUS_ARCH)
                        .setValuesFilter(ChangeHandlerHelper::isBannersStatusArchChanged)
                        .setMapper(this::mapBannersDeleteOrBecameArchive)
                        .build());

        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(BANNERS)
                        .setOperation(UPDATE)
                        .setColumn(BANNERS.STATUS_MODERATE)
                        .setValuesFilter(ChangeHandlerHelper::isBannerStatusModerateChanged)
                        .setMapper(this::mapBanners)
                        .build());

        // flags - колонка с типом text, а бинлоги у нас включены в режиме NOBLOB, поэтому значение
        // before не придёт
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(BANNERS)
                        .setOperation(UPDATE)
                        .setColumn(BANNERS.FLAGS)
                        .setMapper(this::mapBanners)
                        .build());
    }

    private void initBannersMinusGeoChanges(TableChangesHandler<RecomTracerLogicObject> tableChangesHandler) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(BANNERS_MINUS_GEO)
                        .setOperation(UPDATE)
                        .setColumn(BANNERS_MINUS_GEO.MINUS_GEO)
                        .setMapper(this::mapBannersMinusGeo)
                        .build());
    }

    private RecomTracerLogicObject mapCampaigns(ProceededChange change) {
        return createCampaignChangeLogicObject(supportedType().getId(), change);
    }

    private RecomTracerLogicObject mapBanners(ProceededChange change) {
        return createBannerChangeLogicObject(supportedType().getId(), change);
    }

    private RecomTracerLogicObject mapPhrases(ProceededChange change) {
        return createPhrasesChangeLogicObject(supportedType().getId(), change);
    }

    private RecomTracerLogicObject mapBannersMinusGeo(ProceededChange change) {
        Long bid = change.getPrimaryKey(BANNERS_MINUS_GEO.BID);
        return createSimpleBannerChangeLogicObject(supportedType().getId(), bid);
    }

    private RecomTracerLogicObject mapBannersDeleteOrBecameArchive(ProceededChange change) {
        return createBannerChangeCanAffectedGroupLogicObject(supportedType().getId(), change);
    }

    private boolean isPhrasesStatusModerateChanged(ProceededChange change) {
        if (change.beforeContains(PHRASES.STATUS_MODERATE) && change.afterContains(PHRASES.STATUS_MODERATE)) {
            String statusModerateBefore = change.getBefore(PHRASES.STATUS_MODERATE);
            String statusModerateAfter = change.getAfter(PHRASES.STATUS_MODERATE);
            return !Objects.equals(statusModerateBefore, statusModerateAfter);
        }
        return false;
    }

    private boolean isCampaignStatusShowChanged(ProceededChange change) {
        if (change.afterContains(CAMPAIGNS.STATUS_SHOW) && change.beforeContains(CAMPAIGNS.STATUS_SHOW)) {
            String statusShowBefore = change.getBefore(CAMPAIGNS.STATUS_SHOW);
            String statusShowAfter = change.getAfter(CAMPAIGNS.STATUS_SHOW);
            return !Objects.equals(statusShowBefore, statusShowAfter);
        }
        return false;
    }
}
