package ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.pages;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.recommendation.RecommendationType;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecomTracerLogicObject;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecommendationKeyIdentifier;
import ru.yandex.direct.ess.router.rules.recomtracer.RecomTracerChangeHandler;
import ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper;
import ru.yandex.direct.ess.router.utils.ProceededChange;
import ru.yandex.direct.ess.router.utils.TableChange;
import ru.yandex.direct.ess.router.utils.TableChangesHandler;

import static com.google.common.base.MoreObjects.firstNonNull;
import static ru.yandex.direct.binlog.model.Operation.UPDATE;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.ess.router.rules.recomtracer.changehandlers.ChangeHandlerHelper.createCampaignChangeLogicObject;

/**
 * <pre>
 * Для рекомендаций "Уберите площадку из запрещённых" отслеживаются следующие события:
 * 1) Архивация кампании — здесь как обычно убираем все рекомендации по префиксу (client_id, type, cid)
 * 2) Изменение campaigns.DontShow — убираем все рекомендации по префиксу (client_id, type, cid). Если бы
 *    поле не было текстовым (тип text), то можно было бы определить, какие именно площадки были разбанены,
 *    и скрыть рекомендации более селективно. К сожалению, тип text с моделью репликации NOBLOB не позволяет
 *    этого сделать.
 * </pre>
 */
@Component
public class RemovePagesFromBlackListChangeHandler implements RecomTracerChangeHandler {
    @Override
    public RecommendationType supportedType() {
        return RecommendationType.removePagesFromBlackListOfACampaign;
    }

    @Override
    public void initInterestingChanges(TableChangesHandler<RecomTracerLogicObject> tableChangesHandler) {
        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS.ARCHIVED)
                        .setValuesFilter(ChangeHandlerHelper::isCampaignBecameArchived)
                        .setMapper(this::mapCampaigns)
                        .build());

        tableChangesHandler.addTableChange(
                new TableChange.Builder<RecomTracerLogicObject>()
                        .setTable(CAMPAIGNS)
                        .setOperation(UPDATE)
                        .setColumn(CAMPAIGNS.DONT_SHOW)
                        .setMapper(this::mapCampaignsDontShow)
                        .build());
    }

    private RecomTracerLogicObject mapCampaigns(ProceededChange change) {
        return createCampaignChangeLogicObject(supportedType().getId(), change);
    }

    private RecomTracerLogicObject mapCampaignsDontShow(ProceededChange change) {
        String dontShowAfter = firstNonNull(change.getAfter(CAMPAIGNS.DONT_SHOW), "");
        Long cid = change.getPrimaryKey(CAMPAIGNS.CID);
        Long clientId = change.getBefore(CAMPAIGNS.CLIENT_ID);
        return new RecomTracerLogicObject.Builder()
                .withRecommendationTypeId(supportedType().getId())
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.CID, cid)
                .addRecommendationKeyIdentifier(RecommendationKeyIdentifier.CLIENT_ID, clientId)
                .addAdditionalColumn(CAMPAIGNS.DONT_SHOW, dontShowAfter)
                .build();
    }
}
