package ru.yandex.direct.ess.router.rules.uac.updatestatuses

import java.math.BigInteger
import ru.yandex.direct.binlog.model.BinlogEvent
import ru.yandex.direct.binlog.model.Operation
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS
import ru.yandex.direct.dbschema.ppc.enums.ModReasonsType
import ru.yandex.direct.ess.config.uac.updatestatuses.UacUpdateStatusesConfig
import ru.yandex.direct.ess.logicobjects.uac.updatestatuses.UacUpdateStatusesObject
import ru.yandex.direct.ess.router.models.TEssEvent
import ru.yandex.direct.ess.router.models.rule.AbstractRule
import ru.yandex.direct.ess.router.models.rule.EssRule
import ru.yandex.direct.ess.router.utils.ColumnsChangeType
import ru.yandex.direct.ess.router.utils.GrutTableChange
import ru.yandex.direct.ess.router.utils.ProceededChange
import ru.yandex.direct.ess.router.utils.TableChange
import ru.yandex.direct.ess.router.utils.TableChangesHandler
import ru.yandex.grut.objects.proto.client.Schema.EObjectType

@EssRule(UacUpdateStatusesConfig::class)
class UacUpdateStatusesRule : AbstractRule<UacUpdateStatusesObject>() {

    companion object {
        private val MOD_REASONS_TYPES: Set<String> = setOf(ModReasonsType.banner.literal, ModReasonsType.image.literal, ModReasonsType.video_addition.literal)
    }

    private val tableChangesHandler = TableChangesHandler<UacUpdateStatusesObject>()

    init {
        tableChangesHandler.apply {
            addGrutTableChange(
                GrutTableChange.Builder<UacUpdateStatusesObject>()
                    .setObjectType(EObjectType.OT_BANNER_CANDIDATE)
                    .setWatchlogMapper { UacUpdateStatusesObject(campaignId = it.directCampaignId, fromGrut = true) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<UacUpdateStatusesObject>()
                    .setTable(BANNERS)
                    .setOperation(Operation.INSERT)
                    .setMapper { UacUpdateStatusesObject(bannerId = it.getPrimaryKey(BANNERS.BID)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<UacUpdateStatusesObject>()
                    .setTable(BANNERS)
                    .setOperation(Operation.DELETE)
                    .setMapper { UacUpdateStatusesObject(campaignId = it.getBefore(BANNERS.CID)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<UacUpdateStatusesObject>()
                    .setTable(BANNERS)
                    .setColumns(ColumnsChangeType.ANY, listOf(
                        BANNERS.STATUS_MODERATE,
                        BANNERS.FLAGS,
                        BANNERS.STATUS_ARCH,
                        BANNERS.STATUS_SHOW,
                        BANNERS.STATUS_ACTIVE,
                        BANNERS.STATUS_SITELINKS_MODERATE,
                    ))
                    .setOperation(Operation.UPDATE)
                    .setMapper { UacUpdateStatusesObject(bannerId = it.getPrimaryKey(BANNERS.BID)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<UacUpdateStatusesObject>()
                    .setTable(Tables.BANNER_IMAGES)
                    .setOperation(Operation.INSERT)
                    .setMapper { UacUpdateStatusesObject(bannerId = it.getBeforeOrAfter(Tables.BANNER_IMAGES.BID)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<UacUpdateStatusesObject>()
                    .setTable(Tables.BANNER_IMAGES)
                    .setColumn(Tables.BANNER_IMAGES.STATUS_MODERATE)
                    .setOperation(Operation.UPDATE)
                    .setMapper { UacUpdateStatusesObject(bannerId = it.getBeforeOrAfter(Tables.BANNER_IMAGES.BID)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<UacUpdateStatusesObject>()
                    .setTable(Tables.BANNERS_PERFORMANCE)
                    .setColumn(Tables.BANNERS_PERFORMANCE.STATUS_MODERATE)
                    .setOperation(Operation.UPDATE)
                    .setMapper { UacUpdateStatusesObject(bannerId = it.getBeforeOrAfter(Tables.BANNERS_PERFORMANCE.BID)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<UacUpdateStatusesObject>()
                    .setTable(Tables.MOD_REASONS)
                    .setColumn(Tables.MOD_REASONS.REASON)
                    .setOperation(Operation.UPDATE)
                    .setValuesFilter { MOD_REASONS_TYPES.contains(it.getAfter(Tables.MOD_REASONS.TYPE)) }
                    .setMapper { UacUpdateStatusesObject(bannerId = getObjectIdFromModReason(it)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<UacUpdateStatusesObject>()
                    .setTable(Tables.MOD_REASONS)
                    .setOperation(Operation.INSERT)
                    .setValuesFilter { MOD_REASONS_TYPES.contains(it.getAfter(Tables.MOD_REASONS.TYPE)) }
                    .setMapper { UacUpdateStatusesObject(bannerId = getObjectIdFromModReason(it)) }
                    .build()
            )

            addTableChange(
                TableChange.Builder<UacUpdateStatusesObject>()
                    .setTable(Tables.MOD_REASONS)
                    .setOperation(Operation.DELETE)
                    .setValuesFilter { MOD_REASONS_TYPES.contains(it.getBefore(Tables.MOD_REASONS.TYPE)) }
                    .setMapper { UacUpdateStatusesObject(bannerId = getObjectIdFromModReason(it)) }
                    .build()
            )
        }

    }

    override fun mapBinlogEvent(binlogEvent: BinlogEvent): List<UacUpdateStatusesObject> {
        return tableChangesHandler.processChanges(binlogEvent)
    }

    override fun mapWatchlogEvent(watchlogEvent: TEssEvent): List<UacUpdateStatusesObject> {
        return tableChangesHandler.processGrutChanges(watchlogEvent)
    }

    private fun getObjectIdFromModReason(change: ProceededChange): Long {
        val id: BigInteger = change.getBeforeOrAfter(Tables.MOD_REASONS.ID)
        return id.toLong()
    }
}
