package ru.yandex.direct.ess.router.utils;

import java.math.BigInteger;
import java.time.LocalDateTime;
import java.util.Map;
import java.util.Objects;

import org.jooq.Field;
import org.jooq.Table;

import ru.yandex.direct.binlog.model.Operation;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.binlog.model.Operation.DELETE;

public class ProceededChange {
    private final String tableName;
    private final Operation operation;
    private final LocalDateTime binlogTimestamp;
    private final String essTag;
    private final Long reqId;
    private final String service;
    private final String method;
    private final Map<String, Object> primaryKeys;
    private final Map<String, Object> before;
    private final Map<String, Object> after;
    private final boolean isResharding;

    private ProceededChange(String tableName, Operation operation, LocalDateTime binlogTimestamp, String essTag,
                            Long reqId, String service, String method,
                            Map<String, Object> primaryKeys, Map<String, Object> before, Map<String, Object> after,
                            boolean isResharding) {
        this.tableName = tableName;
        this.operation = operation;
        this.binlogTimestamp = binlogTimestamp;
        this.essTag = essTag;
        this.reqId = reqId;
        this.service = service;
        this.method = method;
        this.primaryKeys = primaryKeys;
        this.before = before;
        this.after = after;
        this.isResharding = isResharding;
    }

    public boolean tableIs(Table<?> table) {
        return this.tableName.equals(table.getName());
    }

    public Operation getOperation() {
        return operation;
    }

    public LocalDateTime getBinlogTimestamp() {
        return binlogTimestamp;
    }

    public String getEssTag() {
        return essTag;
    }

    public Long getReqId() {
        return reqId;
    }

    public String getService() {
        return service;
    }

    public String getMethod() {
        return method;
    }

    public boolean afterContains(Field<?> fieldName) {
        return after.containsKey(fieldName.getName());
    }

    @SuppressWarnings("unchecked")
    public <R, T> R getAfter(Field<T> fieldName) {
        return (R) after.get(fieldName.getName());
    }

    public boolean beforeContains(Field<?> fieldName) {
        return before.containsKey(fieldName.getName());
    }

    @SuppressWarnings("unchecked")
    public <R, T> R getBefore(Field<T> fieldName) {
        return (R) before.get(fieldName.getName());
    }

    public boolean primaryKeyContains(Field<?> fieldName) {
        return primaryKeys.containsKey(fieldName.getName());
    }

    @SuppressWarnings("unchecked")
    public <R, T> R getPrimaryKey(Field<T> fieldName) {
        return (R) primaryKeys.get(fieldName.getName());
    }

    public boolean getIsResharding() {
        return isResharding;
    }

    /**
     * Получение значения ключевого поля типа Long.
     * Если в MySQL это поле bigint unsigned, и станет BigInteger - оно конвертируется в Long.
     */
    public Long getLongPrimaryKey(Field<Long> field) {
        Object pkey = getPrimaryKey(field);
        Long pkeyId;
        if (pkey instanceof BigInteger) {
            pkeyId = ((BigInteger) pkey).longValue();
        } else {
            pkeyId = (Long) pkey;
        }
        return pkeyId;
    }

    @SuppressWarnings("unchecked")
    public <R, T> R getBeforeOrAfter(Field<T> fieldName) {
        return (R) ((operation.equals(DELETE) ? before : after).get(fieldName.getName()));

    }

    /**
     * Получение значения поля типа Long.
     * В случае операции Delete значение берется из состояния до операции, иначе из состояния после.
     * Если в MySQL это поле bigint unsigned, и станет BigInteger - оно конвертируется в Long.
     */
    public Long getLongBeforeOrAfter(Field<Long> field) {
        Object pkey = (operation.equals(DELETE) ? before : after).get(field.getName());
        Long pkeyId;
        if (pkey instanceof BigInteger) {
            // Первичный ключ в MySQL может быть bigint unsigned, его нужно явно приводить к long
            pkeyId = ((BigInteger) pkey).longValue();
        } else {
            pkeyId = (Long) pkey;
        }
        return pkeyId;
    }

    public boolean beforeOrAfterContains(Field<?> field) {
        return beforeOrAfterContains(field.getName());
    }

    public boolean beforeOrAfterContains(String fieldName) {
        return (operation.equals(DELETE) ? before : after).containsKey(fieldName);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ProceededChange that = (ProceededChange) o;
        return Objects.equals(tableName, that.tableName) &&
                operation == that.operation &&
                Objects.equals(binlogTimestamp, that.binlogTimestamp) &&
                Objects.equals(essTag, that.essTag) &&
                Objects.equals(reqId, that.reqId) &&
                Objects.equals(service, that.service) &&
                Objects.equals(method, that.method) &&
                Objects.equals(primaryKeys, that.primaryKeys) &&
                Objects.equals(before, that.before) &&
                Objects.equals(after, that.after);
    }

    @Override
    public String toString() {
        return "ProceededChange{" +
                "tableName='" + tableName + '\'' +
                ", operation=" + operation +
                ", binlogTimestamp=" + binlogTimestamp +
                ", essTag='" + essTag + '\'' +
                ", reqId=" + reqId +
                ", service='" + service + '\'' +
                ", method='" + method + '\'' +
                ", primaryKeys=" + primaryKeys +
                ", before=" + before +
                ", after=" + after +
                '}';
    }

    @Override
    public int hashCode() {
        return Objects.hash(tableName, operation, binlogTimestamp, essTag, reqId, service, method, primaryKeys,
                before, after);
    }

    public static class Builder {
        private String tableName;
        private Operation operation;
        private LocalDateTime binlogTimestamp;
        private String essTag = "";
        private Long reqId = 0L;
        private String service = "";
        private String method = "";
        private Map<String, Object> primaryKeys;
        private Map<String, Object> before;
        private Map<String, Object> after;
        private boolean isResharding = false;

        Builder setTableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        Builder setOperation(Operation operation) {
            this.operation = operation;
            return this;
        }

        Builder setBinlogTimestamp(LocalDateTime binlogTimestamp) {
            this.binlogTimestamp = binlogTimestamp;
            return this;
        }

        Builder setEssTag(String essTag) {
            this.essTag = essTag;
            return this;
        }

        Builder setPrimaryKeys(Map<String, Object> primaryKeys) {
            this.primaryKeys = primaryKeys;
            return this;
        }

        public Builder setBefore(Map<String, Object> before) {
            this.before = before;
            return this;
        }

        public Builder setAfter(Map<String, Object> after) {
            this.after = after;
            return this;
        }

        public Builder setReqId(Long reqId) {
            this.reqId = reqId;
            return this;
        }

        public Builder setService(String service) {
            this.service = service;
            return this;
        }

        public Builder setMethod(String method) {
            this.method = method;
            return this;
        }

        public Builder setIsResharding(boolean isResharding) {
            this.isResharding = isResharding;
            return this;
        }

        public ProceededChange build() {
            checkState(tableName != null);
            checkState(operation != null);
            checkState(primaryKeys != null);
            checkState(before != null);
            checkState(after != null);
            return new ProceededChange(tableName, operation, binlogTimestamp, essTag, reqId, service, method,
                    primaryKeys, before, after, isResharding);
        }

    }
}

