package ru.yandex.direct.ess.router.utils;

import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.jooq.Named;
import org.jooq.Table;

import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.ess.common.models.BaseLogicObject;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.ess.router.utils.ColumnsChangeType.ALL;
import static ru.yandex.direct.ess.router.utils.ColumnsChangeType.ANY;

public class TableChange<T> {
    private static final List<String> ALL_FIELDS = emptyList();
    private static final Predicate<ProceededChange> ALL_VALUES = proceededChange -> true;

    private final String tableName;
    private final Operation operation;
    private final ColumnsWithChangeType columnsWithChangeType;
    private final Predicate<ProceededChange> valuesFilter;
    private final Function<ProceededChange, T> binlogMapper;

    private TableChange(String tableName, Operation operation, ColumnsWithChangeType columnsWithChangeType,
                        Predicate<ProceededChange> valuesFilter, Function<ProceededChange, T> binlogMapper) {
        this.tableName = tableName;
        this.operation = operation;
        this.columnsWithChangeType = columnsWithChangeType;
        this.valuesFilter = valuesFilter;
        this.binlogMapper = binlogMapper;
    }

    String getTableName() {
        return tableName;
    }

    Operation getOperation() {
        return operation;
    }

    public ColumnsWithChangeType getColumnsWithChangeType() {
        return columnsWithChangeType;
    }

    Predicate<ProceededChange> getValuesFilter() {
        return valuesFilter;
    }

    Function<ProceededChange, T> getBinlogMapper() {
        return binlogMapper;
    }

    public static class Builder<T extends BaseLogicObject> {
        private String tableName;
        private Operation operation;
        private ColumnsWithChangeType columnsWithChangeType = new ColumnsWithChangeType(ALL, ALL_FIELDS);
        private Predicate<ProceededChange> valuesFilter = ALL_VALUES;
        private Function<ProceededChange, T> binlogMapper;

        /**
         * @param table - таблица, измененение которой интересно
         */
        public Builder<T> setTable(Table<?> table) {
            this.tableName = table.getName();
            return this;
        }

        /**
         * @param operation - опреация над таблицей
         */
        public Builder<T> setOperation(Operation operation) {
            this.operation = operation;
            return this;
        }

        /**
         * По умолчанию берутся изменения, в которых поменялся хотя бы один из столбцов
         * Следовательно, для INSERT, DELETE или UPDATE любой колонки выставлять список столбцов не нужно
         * Если указать тип {@link ColumnsChangeType#ANY} и пустой список колонок - условие никогда не выполнится
         *
         * @param columnsChangeType тип изменения списка столбцов {@link ColumnsChangeType}
         * @param columns           изменение каких стобцов надо отслеживать.
         */
        public Builder<T> setColumns(ColumnsChangeType columnsChangeType, List<Named> columns) {
            var namedColumns = columns.stream()
                    .map(Named::getName)
                    .sorted()
                    .collect(Collectors.toList());
            this.columnsWithChangeType = new ColumnsWithChangeType(columnsChangeType, namedColumns);
            return this;
        }

        /**
         * Упрощенный вариант метода {@link #setColumns(ColumnsChangeType, List)}, если надо отслеживать только одну
         * колонку
         */
        public Builder<T> setColumn(Named column) {
            return setColumns(ANY, List.of(column));
        }

        /**
         * @param valuesFilter - предикат проверки значений, если нужно отслеживать любое изменение поля - передавать
         *                     не нужно
         */
        public Builder<T> setValuesFilter(Predicate<ProceededChange> valuesFilter) {
            this.valuesFilter = valuesFilter;
            return this;
        }

        /**
         * @param binlogMapper - преобразование {@link ProceededChange} в свой класс, наследуемый от
         *                     {@link BaseLogicObject}
         */
        public Builder<T> setMapper(Function<ProceededChange, T> binlogMapper) {
            this.binlogMapper = binlogMapper;
            return this;
        }

        public TableChange<T> build() {
            checkArgument(tableName != null, "Null table name!");
            checkArgument(operation != null, "Null operation!");
            checkArgument(binlogMapper != null, "Null binlogMapper!");
            return new TableChange<>(tableName, operation, columnsWithChangeType, valuesFilter,
                    binlogMapper);
        }
    }

    static class ColumnsWithChangeType {
        private ColumnsChangeType columnsChangeType;
        private List<String> columns;

        public ColumnsWithChangeType(ColumnsChangeType columnsChangeType, List<String> columns) {
            this.columnsChangeType = columnsChangeType;
            this.columns = columns;
        }

        public ColumnsChangeType getColumnsChangeType() {
            return columnsChangeType;
        }

        public List<String> getColumns() {
            return columns;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ColumnsWithChangeType that = (ColumnsWithChangeType) o;
            return columnsChangeType == that.columnsChangeType &&
                    Objects.equals(columns, that.columns);
        }

        @Override
        public int hashCode() {
            return Objects.hash(columnsChangeType, columns);
        }
    }
}
