package ru.yandex.direct.ess.router.utils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import javax.annotation.Nonnull;

import one.util.streamex.StreamEx;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.ess.router.models.TEssEvent;
import ru.yandex.grut.objects.proto.client.Schema.EObjectType;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

public class TableChangesHandler<T> {

    private final Map<String, Map<Operation, Map<TableChange.ColumnsWithChangeType, List<CustomRowFilterAndMapper>>>> changes =
            new HashMap<>();
    private final Map<EObjectType, Function<TEssEvent, T>> grutChanges = new HashMap<>();

    public void addGrutTableChange(GrutTableChange<T> grutTableChange) {
        grutChanges.put(grutTableChange.getObjectType(), grutTableChange.getWatchlogMapper());
    }

    public List<T> processGrutChanges(TEssEvent event) {
        var watchlogMapper = grutChanges.get(getObjectType(event));
        if (watchlogMapper == null) {
            return emptyList();
        }

        return List.of(watchlogMapper.apply(event));
    }

    private EObjectType getObjectType(TEssEvent event) {
        if (event.hasObjectType()) {
            return event.getObjectType();
        } else {
            return EObjectType.OT_BANNER_CANDIDATE;
        }
    }

    public void addTableChange(TableChange<T> tableChange) {
        changes.computeIfAbsent(tableChange.getTableName(), tableName -> new EnumMap<>(Operation.class))
                .computeIfAbsent(tableChange.getOperation(), operation -> new HashMap<>())
                .computeIfAbsent(tableChange.getColumnsWithChangeType(), columns -> new ArrayList<>())
                .add(new CustomRowFilterAndMapper(tableChange.getValuesFilter(), tableChange.getBinlogMapper()));
    }

    public List<T> processChanges(BinlogEvent binlogEvent) {
        if (!containsTableOperation(binlogEvent.getTable(), binlogEvent.getOperation())) {
            return emptyList();
        }

        Map<TableChange.ColumnsWithChangeType, List<CustomRowFilterAndMapper>> columnsToFiltersAndMappers =
                changes.get(binlogEvent.getTable()).get(binlogEvent.getOperation());
        List<BinlogEvent.Row> rows = binlogEvent.getRows();

        return columnsToFiltersAndMappers.entrySet().stream()
                // Для каждого интересного изменения столбцов, находим индексы строк, в которых эти столбцы
                // изменились. С индексом возвращается еще пользовательский обработчики
                .flatMap(e -> allSuitableIndexesWithFiltersAndMappers(rows, e.getKey(), e.getValue()
                ))
                // Если строка подходит под несколько отслеживаний, объединяем обработчики
                .collect(toMap(RowIndexToFiltersAndMappers::getIndex, RowIndexToFiltersAndMappers::getFiltersAndMappers,
                        this::mergeLists))
                .entrySet().stream()
                // Применяются пользовательские фильтры, полученный результат преобразуется в LogicObject
                .flatMap(rowIndexToFilterAndMapper -> {
                    Integer rowIndex = rowIndexToFilterAndMapper.getKey();
                    List<CustomRowFilterAndMapper> rowFiltersAndMappers = rowIndexToFilterAndMapper.getValue();
                    return customFilterAndMap(rowFiltersAndMappers, binlogEvent,
                            rows.get(rowIndex).getPrimaryKey(), rows.get(rowIndex).getBefore(),
                            rows.get(rowIndex).getAfter());
                })
                .collect(toList());
    }

    private Stream<RowIndexToFiltersAndMappers> allSuitableIndexesWithFiltersAndMappers(
            List<BinlogEvent.Row> rows,
            TableChange.ColumnsWithChangeType columnsToChangeWithChangeType,
            List<CustomRowFilterAndMapper> rowFiltersAndMappers) {
        return IntStream.range(0, rows.size())
                // Если операция INSERT или DELETE, то
                // columnsToChange - пустой список -> все индексы подойдут
                .filter(rowIndex -> filterColumnsByChangeType(columnsToChangeWithChangeType.getColumnsChangeType(),
                        columnsToChangeWithChangeType.getColumns(), rows.get(rowIndex)))
                .boxed()
                .map(rowIndex -> new RowIndexToFiltersAndMappers(rowIndex, rowFiltersAndMappers));
    }

    private boolean filterColumnsByChangeType(@Nonnull ColumnsChangeType columnsChangeType,
                                              Collection<String> columnsToChange,
                                              BinlogEvent.Row row) {
        Predicate<String> condition = columnToChange -> isColumnValueChanged(columnToChange, row.getAfter(),
                row.getBefore());
        var startStream = columnsToChange.stream();
        if (columnsChangeType == ColumnsChangeType.ANY) {
            return startStream.anyMatch(condition);
        } else if (columnsChangeType == ColumnsChangeType.ANY_EXCEPT) {
            var otherColumns = StreamEx.of(row.getAfter().keySet())
                    .filter(r -> !columnsToChange.contains(r))
                    .filter(columnToChange -> isColumnValueChanged(columnToChange, row.getAfter(), row.getBefore()))
                    .toList();
            return !otherColumns.isEmpty();
        } else {
            return startStream.allMatch(condition);
        }
    }

    private boolean isColumnValueChanged(String columnName, Map<String, Object> rowAfter,
                                         Map<String, Object> rowBefore) {
        Object valueBefore = rowBefore.get(columnName);
        Object valueAfter = rowAfter.get(columnName);

        return !Objects.equals(valueBefore, valueAfter)
                // blob-поле, предыдущего значения которого нет в бинлоге в силу NOBLOB, превратилось в null
                || !rowBefore.containsKey(columnName) && rowAfter.containsKey(columnName)
                ;
    }

    private Stream<T> customFilterAndMap(List<CustomRowFilterAndMapper> customRowFilterAndMappers,
                                         BinlogEvent binlogEvent,
                                         Map<String, Object> rowPrimaryKeys,
                                         Map<String, Object> rowBefore,
                                         Map<String, Object> rowAfter) {
        ProceededChange proceededChange = new ProceededChange.Builder()
                .setTableName(binlogEvent.getTable())
                .setOperation(binlogEvent.getOperation())
                .setBinlogTimestamp(binlogEvent.getUtcTimestamp())
                .setEssTag(binlogEvent.getEssTag())
                .setReqId(binlogEvent.getTraceInfoReqId())
                .setService(binlogEvent.getTraceInfoService())
                .setMethod(binlogEvent.getTraceInfoMethod())
                .setPrimaryKeys(rowPrimaryKeys)
                .setBefore(rowBefore)
                .setAfter(rowAfter)
                .setIsResharding(binlogEvent.isResharding())
                .build();
        return customRowFilterAndMappers.stream()
                .filter(changesHandler -> changesHandler.valuesFilter.test(proceededChange))
                .map(changeHandler -> changeHandler.binlogMapper.apply(proceededChange));
    }

    private boolean containsTableOperation(String table, Operation operation) {
        return changes.containsKey(table) && changes.get(table).containsKey(operation);
    }

    private List<CustomRowFilterAndMapper> mergeLists(List<CustomRowFilterAndMapper> l1,
                                                      List<CustomRowFilterAndMapper> l2) {
        List<CustomRowFilterAndMapper> resultList = new ArrayList<>();
        resultList.addAll(l1);
        resultList.addAll(l2);
        return resultList;
    }

    private class CustomRowFilterAndMapper {
        Predicate<ProceededChange> valuesFilter;
        Function<ProceededChange, T> binlogMapper;

        CustomRowFilterAndMapper(Predicate<ProceededChange> valuesFilter, Function<ProceededChange, T> binlogMapper) {
            this.valuesFilter = valuesFilter;
            this.binlogMapper = binlogMapper;
        }
    }

    private class RowIndexToFiltersAndMappers {
        int index;
        List<CustomRowFilterAndMapper> filtersAndMappers;

        RowIndexToFiltersAndMappers(int index, List<CustomRowFilterAndMapper> filtersAndMappers) {
            this.index = index;
            this.filtersAndMappers = filtersAndMappers;
        }

        int getIndex() {
            return index;
        }

        List<CustomRowFilterAndMapper> getFiltersAndMappers() {
            return filtersAndMappers;
        }
    }
}
