#include <kikimr/persqueue/sdk/deprecated/cpp/v2/persqueue.h>
#include <library/cpp/getopt/opt.h>
#include <library/cpp/getopt/small/modchooser.h>
#include <util/stream/file.h>
#include <util/string/vector.h>
#include <util/system/fs.h>

using namespace NLastGetopt;
using namespace NThreading;
using namespace NPersQueue;

bool NeedStop = false;

void SigHandler(int) {
    NeedStop = true;
}

template<class T>
bool WaitFuture(T& future)
{
    while (true) {
        future.Wait(TDuration::MilliSeconds(200));
        if (NeedStop) {
            return false;
        }
        if (future.HasValue())
            return true;
    }
}

int Consumer(int argc, const char* argv[]) {
    TString topic, server, client, group, tvmSecretPath;
    ui16 port;
    ui32 maxCount;
    ui32 tvmId;
    ui32 dstClientId;
    int logLevel;
    bool onlyOriginal = false;
    bool skipCommit = false;
    bool skipBrokenChunks = false;
    TVector<TString> topics, groups;

    {
        TOpts opts;
        opts.AddHelpOption('h');
        opts.AddLongOption('s', "server", "server addr")
            .StoreResult(&server)
            .Required();
        opts.AddLongOption('p', "port", "server port")
            .StoreResult(&port)
            .DefaultValue(2135);
        opts.AddLongOption('c', "consumer", "consumer id")
            .StoreResult(&client)
            .DefaultValue("direct-debug-reader");
        opts.AddLongOption('t', "topics", "topics to read from, comma-separated")
            .Required()
            .SplitHandler(&topics, ',');
        opts.AddLongOption('o', "only-original", "read only original topics")
            .NoArgument()
            .SetFlag(&onlyOriginal)
            .DefaultValue("false");
        opts.AddLongOption('n', "no-commit", "don't commit offset to LogBroker after read")
            .NoArgument()
            .SetFlag(&skipCommit)
            .DefaultValue("false");
        opts.AddLongOption('b', "skip-broken", "skip broken chunks")
            .NoArgument()
            .SetFlag(&skipBrokenChunks)
            .DefaultValue("false");
        opts.AddLongOption('l', "limit", "maxium messages count, 0 - unlimited")
            .StoreResult(&maxCount)
            .DefaultValue(0);
        opts.AddLongOption('g', "groups", "groups to read from, comma-separated")
            .SplitHandler(&groups, ',');
        opts.AddLongOption("tvm-client-id", "consumer TVM client id")
            .StoreResult(&tvmId)
            .DefaultValue(2001804);
        opts.AddLongOption("tvm-secret-file", "path to file with TVM secret")
            .StoreResult(&tvmSecretPath)
            .DefaultValue("/etc/direct-tokens/tvm2_logbroker-prestable-consumer-direct-debug-reader");
        opts.AddLongOption("tvm-server-id", "LogBroker TVM server id, see https://abc.yandex-team.ru/services/Logbroker/resources/?tag=2&tag=29&supplier=14&type=47&state=granted&view=consuming")
            .StoreResult(&dstClientId)
            .DefaultValue(2001147);
        opts.AddLongOption('L', "log-level", "log level (0-7)")
            .StoreResult(&logLevel)
            .DefaultValue(6);

        TOptsParseResult res(&opts, argc, argv);
    }

    TPQLibSettings pqLibSettings;
    pqLibSettings.ThreadsCount = 1;
    TPQLib pq(pqLibSettings);
    TConsumerSettings settings;
    settings.MaxCount = maxCount;
    settings.MaxUncommittedCount = maxCount;
    settings.MaxInflyRequests = 1;
    settings.Server = TServerSetting{server, port};
    settings.ClientId = client;
    settings.Topics = topics;
    settings.ReadMirroredPartitions = !onlyOriginal;
    for (const auto& g : groups) {
        settings.PartitionGroups.push_back(FromString<ui32>(g));
    }
    settings.SkipBrokenChunks = skipBrokenChunks;

    TIntrusivePtr<TCerrLogger> logger(new TCerrLogger(logLevel));

    if (!NFs::Exists(tvmSecretPath)) {
        Cerr << "File '" << tvmSecretPath << "' does not exist";
        return 1;
    }

    TFileInput input(tvmSecretPath);
    TString tvmSecret = input.ReadLine();
    settings.CredentialsProvider = CreateTVMCredentialsProvider(tvmSecret, tvmId, dstClientId, logger);

    auto consumer = pq.CreateConsumer(settings, logger, true);
    auto initFuture = consumer->Start();
    bool status = WaitFuture(initFuture);
    if (!status) {
        Cerr << "Terminated\n";
        return 1;
    }

    if (initFuture.GetValue().Response.HasError()) {
        TStringBuilder message;
        message << "Consumer initialization failed " << initFuture.GetValue().Response.GetError() << "\n";
        Cerr << message;
        return 1;
    }

    auto messageFuture = consumer->GetNextMessage();

    ui32 messagesCount = 0;
    while (true) {
        messageFuture.Wait(TDuration::MilliSeconds(200));
        if (NeedStop) {
            Cerr << "Terminated\n";
            return 1;
        }

        if (messageFuture.HasValue()) {
            auto msg = messageFuture.GetValue();
            if (msg.Type == EMT_ERROR) {
                TStringBuilder message;
                message << "Request failed " << msg.Response.GetError() << "\n";
                Cerr << message;
                return 1;
            } else if (msg.Type == EMT_DATA) {
                for (const auto& batch: msg.Response.GetData().GetMessageBatch()) {
                    for (const auto& message: batch.GetMessage()) {
                        Cout << message.GetData();
                        messagesCount++;
                    }
                }
                if (!skipCommit) {
                    auto cookie = msg.Response.GetData().GetCookie();
                    consumer->Commit({cookie});
                }
                if (maxCount > 0 && messagesCount >= maxCount) {
                    Cerr << "Reached messages limit, stop\n";
                    NeedStop = true;
                }
            } else if (msg.Type == EMT_COMMIT) {
            } else {
                TStringBuilder message;
                message << "Unknown response " << msg.Response << "\n";
                Cerr << message;
                return 1;
            }
            messageFuture = consumer->GetNextMessage();
        }
    }
    return 0;
}



int main(int argc, const char* argv[]) {
    signal(SIGTERM, &SigHandler);
    signal(SIGINT, &SigHandler);

    TModChooser modChooser;
    modChooser.AddMode(
        "read",
        Consumer,
        "--  read from LB"
    );

    modChooser.Run(argc, argv);
    return 0;
}
