package ru.yandex.direct.reports;

import java.io.IOException;
import java.nio.file.Path;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.MapType;
import com.google.common.base.Strings;
import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.joda.time.DateTime;

import ru.yandex.direct.logging.LoggingInitializer;
import ru.yandex.direct.logging.LoggingInitializerParams;
import ru.yandex.direct.reports.issues.Comment;
import ru.yandex.direct.reports.issues.GoalGroup;
import ru.yandex.direct.reports.issues.Group;
import ru.yandex.direct.reports.issues.IssueGroup;
import ru.yandex.direct.reports.issues.IssueInfo;
import ru.yandex.direct.staff.client.StaffClient;
import ru.yandex.direct.staff.client.model.GapType;
import ru.yandex.direct.staff.client.model.json.Gap;
import ru.yandex.direct.staff.client.model.json.Head;
import ru.yandex.direct.staff.client.model.json.Name;
import ru.yandex.direct.staff.client.model.json.Person;
import ru.yandex.direct.staff.client.model.json.PersonInfo;
import ru.yandex.direct.tools.IntranetTools;
import ru.yandex.direct.tools.goals.Goal;
import ru.yandex.direct.tools.goals.GoalsClient;
import ru.yandex.direct.tools.wiki.WikiClient;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.io.FileUtils;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.error.ForbiddenException;
import ru.yandex.startrek.client.model.ComponentRef;
import ru.yandex.startrek.client.model.Field;
import ru.yandex.startrek.client.model.GoalRef;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueRef;
import ru.yandex.startrek.client.model.Language;

import static com.google.common.base.Preconditions.checkState;
import static java.time.DayOfWeek.MONDAY;
import static java.time.DayOfWeek.SUNDAY;
import static java.time.temporal.TemporalAdjusters.nextOrSame;
import static java.time.temporal.TemporalAdjusters.previousOrSame;
import static ru.yandex.direct.tools.Constants.SERVICE;
import static ru.yandex.direct.tools.StartrekTools.createClient;

/**
 * Запусать с помощью direct/bin/st_reporter.sh
 */
public class TaskResultsReport {
    private static final String APP = "debug-app";

    public static final String STAFF_HREF = "https://staff.yandex-team.ru/";

    public static final Map<String, Integer> STATUS_SORT_ORDER = Map.of(
            "new", 0,
            "open", 1,
            "needInfo", 2,
            "inProgress", 3,
            "codeReview", 4,
            "readyForTest", 5,
            "testing", 6,
            "betaTested", 7,
            "closed", 8
    );

    Map<String, Issue> mainEpics = new HashMap<>();

    public static final Map<GapType, String> GAP_HEADERS = Map.of(
            GapType.ILLNESS, "Болезнь",
            GapType.VACATION, "Отпуск",
            GapType.PAID_DAY_OFF, "Отгул"
    );

    public static final Set<String> BUG_TYPES = Set.of("bug", "subBug", "bugToVerify");
    public static final String BUGS = "Баги";
    public static final String TASKS = "Задачи";

    private static final String PENDING_REPLY_FROM = "pendingReplyFrom";
    private static final String WEIGHT = "weight";
    private static final String GOALS = "goals";
    private static final String GOAL_IMPORTANCE = "goalImportance";

    public static final Set<Long> TOP_LEVEL_GOALS = Set.of(
            92050L //VS Реклама Q4'20-Q1'21
    );

    private static final Map<String, Field.Schema> CUSTOM_FIELDS = Map.of(
            PENDING_REPLY_FROM, Field.Schema.array(Field.Schema.Type.USER),
            GOALS, Field.Schema.array(Field.Schema.Type.GOAL),
            GOAL_IMPORTANCE, Field.Schema.scalar(Field.Schema.Type.TRANSLATION, false),
            WEIGHT, Field.Schema.scalar(Field.Schema.Type.FLOAT, false));

    //алгоритм сортировки статусов тикетов по каждому человеку
    private static final Comparator<String> STATUS_COMPARATOR =
            Comparator.comparing(s -> STATUS_SORT_ORDER.getOrDefault(s, 500));

    public static final Goal EMPTY_GOAL =
            new Goal().withId(0L).withTitle("Цели без привязки к зонтику").withImportance(1000L);

    public static final GoalGroup EMPTY_GOAL_GROUP = new GoalGroup(EMPTY_GOAL);

    public static final DateTimeFormatter DAY_FORMATTER = DateTimeFormatter.ofPattern("dd/MM");
    public static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    public static final DateTimeFormatter YEAR_FORMATTER = DateTimeFormatter.ofPattern("yyyy");
    public static final DateTimeFormatter WEEK_FORMATTER = DateTimeFormatter.ofPattern("w");

    private final StaffClient staffClient;
    private final GoalsClient goalsClient;
    private final WikiClient wikiClient;
    private final Session stClient;

    @Parameter(
            names = {"-l", "--heads"},
            description = "Список логинов через запятую. Статистика будет выгружена для всех подразделений этих логинов"
    )
    private Set<String> heads = Set.of("mariabye");

    @Parameter(
            names = {"-p", "--wiki-prefix"},
            description = "Адрес страницы для сохранения"
    )
    private String wikiPrefix;

    @Parameter(
            names = {"-d", "--date"},
            description = "Дата для формирования запроса"
    )
    private String epxDate;

    public TaskResultsReport(AsyncHttpClient httpClient) {
        checkState(httpClient != null, "httpClient cannot be null");
        stClient = createClient(CUSTOM_FIELDS, Language.RUSSIAN);
        staffClient = IntranetTools.createStaffClient(httpClient);
        goalsClient = IntranetTools.createGoalsClient(stClient);
        wikiClient = IntranetTools.createWikiClient(httpClient);
    }

    public static void main(String[] args) throws IOException {
        try (AsyncHttpClient asyncHttpClient = new DefaultAsyncHttpClient()) {
            TaskResultsReport tool = new TaskResultsReport(asyncHttpClient);

            JCommander commander = new JCommander();
            commander.addObject(tool);
            commander.parse(args);

            LocalDate today = tool.epxDate == null ? LocalDate.now() : LocalDate.parse(tool.epxDate);
            LocalDate startDate = today.with(previousOrSame(MONDAY));
            LocalDate endDate = today.with(nextOrSame(SUNDAY));

            LoggingInitializer.initialize(new LoggingInitializerParams(), SERVICE, APP);
            tool.calcState(startDate, endDate);
        } catch (IOException e) {
            throw e;
        }
    }

    /**
     * Выполнить выгрузку тикетов на вики
     *
     * @param startDate дата начала выгрузки
     * @param endDate   дата окончания выгрузки
     */
    @SuppressWarnings("MethodLength")
    public void calcState(LocalDate startDate, LocalDate endDate)
            throws IOException {
        String emptyTicket = "DIRECT-119823";
        String dutyTicket = "DIRECT-119886";
        String zbpTicket = "DIRECT-119887";
        String dnaFeedbackTicket = "DIRECT-119889";

        Goal commanderGoal = goalsClient.getGoal(83114L);
        Goal lpcGoal = goalsClient.getGoal(83115L);
        //todo сделать отдельные цели, добавить отдельно коммандер и конструктор турбо

        String commentsDate = startDate.format(DATE_FORMATTER);
        DateTime jodaCommentsDate = DateTime.parse(commentsDate);

        String dateCode = startDate.format(YEAR_FORMATTER) + "_w" + startDate.format(WEEK_FORMATTER);

//        String wikiPath = "users/kuhtich/test4";
//        Set<String> staffGroups = Set.of("yandex_monetize_search_direct_interface_9733_dep27489");
//        Set<String> staffGroups = Set.of(
//                "yandex_monetize_search_direct_interface_9733",
//                "yandex_monetize_interface_dir");
        Path savedGoalsPath = FileUtils.expandHome("~/tmp/goals_to_umbrellas");
        Map<Long, Goal> goalsToUmbrellas = new HashMap<>();

        ObjectMapper mapper = JsonUtils.getObjectMapper();

        if (!savedGoalsPath.toFile().exists()) {

            Map<Long, Goal> umbrellaGoals = new HashMap<>();

            for (Long topLevelGoal : TOP_LEVEL_GOALS) {
                umbrellaGoals.putAll(
                        StreamEx.of(goalsClient.getRelatedGoals(topLevelGoal)).toMap(Goal::getId, Function.identity())
                );
            }

            for (Goal umbrella : umbrellaGoals.values()) {
                for (Goal descendant : getAllDescendantGoals(goalsClient, umbrella, new HashSet<>())) {
                    goalsToUmbrellas.put(descendant.getId(), umbrella);
                }
                goalsToUmbrellas.put(umbrella.getId(), umbrella);
            }

            mapper.writeValue(savedGoalsPath.toFile(), goalsToUmbrellas);
        } else {
            MapType type = JsonUtils.getTypeFactory().constructMapType(HashMap.class, Long.class, Goal.class);
            goalsToUmbrellas = mapper.readValue(savedGoalsPath.toFile(), type);
        }

        //алгоритм сортировки головных тикетов
        Comparator<Group> comparator = Comparator.comparing(Group::order);
        comparator = comparator.thenComparing(Group::text);

        Issue empty = stClient.issues().get(emptyTicket).load();
        Issue duty = stClient.issues().get(dutyTicket).load();
        Issue zbp = stClient.issues().get(zbpTicket).load();
        Issue dnaFeedback = stClient.issues().get(dnaFeedbackTicket).load();

        StringBuilder wiki = new StringBuilder();

        //найти группы, в которых числятся переданные логины
        Set<String> staffGroups = StreamEx.of(staffClient.getStaffUsers(buildStaffHeadsRequest(heads)))
                .map(t -> t.getDepartmentGroup().getUrl())
                .collect(Collectors.toSet());

        //все логины
        Map<String, PersonInfo> staffUsers = StreamEx.of(staffClient.getStaffUsers(buildStaffGroupRequest(staffGroups)))
                .toMap(PersonInfo::getLogin, Function.identity());

        //т.к. только у Ваниной группы нет руководителя, он будет по умолчанию считаться руководителем у группы,
        // где нет руководителя
        Head defaultHead = new Head().withPerson(new Person().withLogin(staffUsers.getOrDefault("cherevko",
                new PersonInfo().withLogin(heads.stream().findFirst().get())).getLogin()));

        String absences = absences(staffUsers.keySet(), startDate, endDate);
        if (!absences.isEmpty()) {
            wiki.append(collapse("Отсутствия", absences));
        }

        //основной костяк запроса
        String queryMain =
                stGroupQuery(staffGroups) +
                        "Resolution: !\"Won't fix\" " +
                        "Resolution: !\"Will Not Fix\" " +
                        "Resolution: !\"Can't reproduce\" " +
                        "Resolution: !Duplicate " +
                        "Resolution: !Invalid " +
                        "Resolution: !Later ";

        String querySortBy = "\"Sort By\": updated ASC";

        //все тикеты, в которых менялся статус за период
        String query1 = queryMain +
                "(status:changed(date: \"" + startDate + "\"..\"" + endDate + "\")) " +
                querySortBy;

        //незакрытые тикеты, в которых были апдейты
        String query2 = queryMain +
                "Status: !closed " +
                "Status: !Beta-tested " +
                "Updated: >= \"" + commentsDate + "\" " +
                querySortBy;

        var issues = stClient.issues().find(query1);

        Map<String, Issue> issueMap = StreamEx.of(issues)
                .distinct(Issue::getKey) //wtf?
                .toMap(Issue::getKey, Function.identity());

        //добавить в мапу незакрытые тикеты, которые не попали в первый фильтр, но в них есть новые комменты
        var withUpdatedComments = stClient.issues().find(query2);
        for (var it = withUpdatedComments.iterator(); it.hasNext(); ) {
            Issue issue = it.next();
            if (issueMap.containsKey(issue.getKey())) {
                continue;
            }
            boolean hasChangedComments = issue.getComments()
                    .stream()
                    .anyMatch(t -> t.getCreatedAt().toDateTime().isAfter(jodaCommentsDate));
            if (hasChangedComments) {
                issueMap.put(issue.getKey(), issue);
            }
        }

        Map<Long, Goal> goals = new HashMap<>();

        List<IssueInfo> issueInfoList = new ArrayList<>();

        for (Issue issue : issueMap.values()) {
            IssueInfo issueInfo = new IssueInfo(issue);
            Issue rootIssue = findRootIssue(issue, empty, duty, zbp, dnaFeedback);
            Goal goal;
            Goal umbrella = null;
            if (issue.getKey().startsWith("LPC") || issue.getKey().startsWith("UCSUPPORT")) {
                goal = lpcGoal;
                umbrella = lpcGoal;
            } else if (issue.getKey().startsWith("DIRECTCLIENT") || issue.getKey().startsWith("DCSUP")) {
                goal = commanderGoal;
                umbrella = commanderGoal;
            } else {
                goal = getGoal(issue, goals);
            }
            if (goal == null) {
                goal = getGoal(rootIssue, goals);
            }
            if (goal != null) {
                issueInfo.withGoal(goal);
                if (umbrella == null) {
                    umbrella = goalsToUmbrellas.get(goal.getId());
                }
                if (umbrella != null) {
                    goals.putIfAbsent(umbrella.getId(), umbrella);
                    issueInfo.withGroup1(new GoalGroup(umbrella));
                    issueInfo.withUmbrella(umbrella);
                } else {
                    goals.putIfAbsent(goal.getId(), goal);
                    issueInfo.withGroup1(EMPTY_GOAL_GROUP);
                }
                issueInfo.withGroup2(new GoalGroup(goal));
            } else {
                issueInfo.withGroup1(new IssueGroup(empty));
                issueInfo.withGroup2(new IssueGroup(rootIssue));
                issueInfo.withRoot(rootIssue);
                issueInfo.withGoal(EMPTY_GOAL);
                issueInfo.withUmbrella(EMPTY_GOAL);
            }
            issueInfoList.add(issueInfo);
        }

        //итерируемся по упорядоченому сету ключей
        SortedSet<Group> group1 = new TreeSet<>(comparator);

        Map<Group, List<IssueInfo>> byGroup1 = StreamEx.of(issueInfoList)
                .groupingBy(IssueInfo::getGroup1);

        group1.addAll(byGroup1.keySet());
        for (Group g1 : group1) {
            StringBuilder b1 = new StringBuilder();
            List<IssueInfo> level1 = byGroup1.get(g1);
            long headCount1 = headCount(level1);
            wiki.append(header1(link(g1.link(), g1.text()))).append(String.format(" (%s)", headCount1)).append("\n");
            Map<Group, List<IssueInfo>> byGroup2 = StreamEx.of(level1)
                    .groupingBy(IssueInfo::getGroup2);
            SortedSet<Group> group2 = new TreeSet<>(comparator);
            group2.addAll(byGroup2.keySet());
            for (Group g2 : group2) {
                List<IssueInfo> list = byGroup2.get(g2);
                long headCount2 = headCount(list);
                b1.append(header2(link(g2.link(), g2.text())))
                        .append(String.format(" (%s)", headCount2))
                        .append("\n");
                Comment comment = g2.lastComment();
                if (comment != null) {
                    String date = comment.getUpdatedAt().format(DAY_FORMATTER);
                    b1.append(collapse("комментарий " + date + " " + staff(comment.getLogin()),
                            comment.getText()))
                            .append("\n");
                }
                b1.append(formatIssues(g2, list)).append("\n");
            }
            wiki.append(collapse("", b1.toString()));
        }

        System.out.println(wikiPrefix);
        wikiClient.createOrUpdatePage("Отчёт " + dateCode, wiki.toString(), wikiPrefix + dateCode);
        wikiClient.createOrUpdatePage("Отчёт " + dateCode, wiki.toString(), wikiPrefix + "last");

        Map<String, List<IssueInfo>> byHeads = StreamEx.of(issueInfoList)
                .groupingBy(i -> staffUsers.get(i.getAssigneeLogin()).getDepartmentGroup().getDepartment()
                        .getHeads().stream().findFirst()
                        .orElse(defaultHead).getPerson().getLogin()); //в группе нет руководителя

        StringBuilder loginsWiki = new StringBuilder();

        for (String head : byHeads.keySet()) {
            loginsWiki.append(header1(nameRu(staffUsers, head) + " " + link(STAFF_HREF + head, "§"))).append("\n");
            List<IssueInfo> list = byHeads.get(head);
            Map<String, List<IssueInfo>> byLogin = StreamEx.of(list).groupingBy(IssueInfo::getAssigneeLogin);
            for (String login : byLogin.keySet()) {
                List<IssueInfo> loginList = byLogin.get(login);
                loginsWiki.append(header2(nameRu(staffUsers, login) + " " + link(STAFF_HREF + login, "§")))
                        .append("\n");
                Map<Group, List<IssueInfo>> byGoal = StreamEx.of(loginList)
                        .groupingBy(IssueInfo::getGroup1);
                SortedSet<Group> groups = new TreeSet<>(comparator);
                groups.addAll(byGoal.keySet());
                for (Group goal : groups) {
                    loginsWiki.append(li(0, link(goal.link(), goal.text()))).append("\n");

                    Map<String, List<IssueInfo>> byType = groupByType(byGoal.get(goal));
                    for (String type : byType.keySet()) {
                        Set<IssueInfo> issueSet = StreamEx.of(byType.get(type))
                                .sorted(Comparator.comparing(IssueInfo::getStatus, STATUS_COMPARATOR))
                                .collect(Collectors.toCollection(LinkedHashSet::new));
                        loginsWiki.append(li(1, type)).append("\n");
                        for (IssueInfo issueInfo : issueSet) {
                            loginsWiki.append(li(2, issueInfo.getKey())).append("\n");
                        }
                    }

                }
            }
        }
        wikiClient.createOrUpdatePage("По людям " + dateCode, loginsWiki.toString(), wikiPrefix + dateCode + "/logins");
        wikiClient.createOrUpdatePage("По людям " + dateCode, loginsWiki.toString(), wikiPrefix + "last" + "/logins");
    }

    private Map<String, List<IssueInfo>> groupByType(List<IssueInfo> list) {
        Map<String, List<IssueInfo>> res = new TreeMap<>(Comparator.reverseOrder());
        for (IssueInfo i : list) {
            String key;
            if (BUG_TYPES.contains(i.getType())) {
                key = BUGS;
            } else {
                key = TASKS;
            }
            var group = res.getOrDefault(key, new ArrayList<>());
            group.add(i);
            res.put(key, group);
        }
        return res;
    }

    private String nameRu(Map<String, PersonInfo> users, String login) {
        var person = users.get(login);
        if (person == null) {
            person = users.get(heads.stream().findFirst().get());
        }
        Name name = person.getName();
        return name.getFirst().getRu() + " " + name.getLast().getRu();
    }

    private long headCount(List<IssueInfo> list) {
        return StreamEx.of(list)
                .map(IssueInfo::getAssigneeLogin)
                .distinct()
                .count();
    }

    private Issue findRootIssue(Issue issue, Issue empty, Issue duty, Issue zbp, Issue dnaFeedback) {
        Issue rootIssue;
        if (isDutyIssue(issue)) {
            rootIssue = duty;
        } else if (isZBPIssue(issue)) {
            rootIssue = zbp;
        } else if (idDnaFeedback(issue)) {
            rootIssue = dnaFeedback;
        } else if (issue.getType().getKey().equals("epic")) {
            rootIssue = mainEpic(issue);
        } else if (issue.getEpic().isPresent()) {
            rootIssue = mainEpic(issue.getEpic().get().load());
        } else if (issue.getParent().isPresent()) {
            Issue root = getRoot(issue);
            if (root.getEpic().isPresent()) {
                rootIssue = mainEpic(root.getEpic().get().load());
            } else {
                rootIssue = root;
            }
        } else {
            rootIssue = empty;
        }
        return rootIssue;
    }

    private Set<Goal> getAllDescendantGoals(GoalsClient goalsClient, Goal goal, Set<Goal> result) {
        Set<Goal> children = goalsClient.getRelatedGoals(goal.getId());

        for (Goal child : children) {
            Set<Goal> subChildren = goalsClient.getRelatedGoals(child.getId());
            for (Goal subChild : subChildren) {
                if (!result.contains(subChild)) {
                    result.add(subChild);
                    result.addAll(getAllDescendantGoals(goalsClient, subChild, result));
                }
            }
        }
        result.addAll(children);
        return result;
    }

    private String formatIssues(Group group, List<IssueInfo> issues) {
        StringBuilder wiki = new StringBuilder();
        System.out.println(group.link() + " " + group.text());
        Map<String, List<IssueInfo>> byAssignee = StreamEx.of(issues)
                .sorted(Comparator.comparing(IssueInfo::getAssigneeLogin))
                .groupingBy(IssueInfo::getAssigneeLogin);
        for (String assignee : byAssignee.keySet()) {
            System.out.println("\t" + assignee + ":");
            wiki.append(li(0, staff(assignee))).append("\n");
            List<IssueInfo> issuesByAssignee = byAssignee.get(assignee);
            Map<String, List<IssueInfo>> issuesByStatus = StreamEx.of(issuesByAssignee)
                    .groupingBy(IssueInfo::getStatus);
            SortedSet<String> statuses = new TreeSet<>(STATUS_COMPARATOR);
            statuses.addAll(issuesByStatus.keySet());
            for (String status : statuses) {
                System.out.println("\t\t" + status + ":");
                wiki.append(li(1, status)).append("\n");
                for (IssueInfo issue : issuesByStatus.get(status)) {
                    System.out.println("\t\t\t" + issue.getKey() + " " + issue.self().text());
                    wiki.append(li(2, link(issue.self().link(), issue.self().text())));
                    if (status.equals("needInfo")) {
                        String replyWiki = StreamEx.of(issue.self().pendingReplyFrom())
                                .map(this::staff)
                                .joining(", ");
                        System.out.println(replyWiki);
                        wiki.append(" Ожидает ответа от ").append(replyWiki);
                    }
                    wiki.append("\n");
                }
            }
        }
        return wiki.toString();
    }

    /**
     * Относится ли тикет к дежурству
     */
    private boolean isDutyIssue(Issue issue) {
        Set<String> components = new HashSet<>();
        for (ComponentRef component : issue.getComponents()) {
            components.add(component.getDisplay());
        }
        return components.contains("Releases: JavaDirect") ||
                issue.getCreatedBy().getLogin().equals("direct-handles");
    }

    /**
     * Относится ли тикет к дежурству
     */
    private boolean isZBPIssue(Issue issue) {
        Set<String> components = new HashSet<>();
        for (ComponentRef component : issue.getComponents()) {
            components.add(component.getDisplay());
        }
        return components.contains("Поддержка: серверная часть") ||
                components.contains("Поддержка: клиентская часть") ||
                issue.getKey().startsWith("DIRECTSUP") ||
                issue.getKey().startsWith("SUPBS");
    }

    private boolean idDnaFeedback(Issue issue) {
        return issue.getTags().filter(t -> t.equals("dna_feedback")).size() > 0;
    }

    private Goal getGoal(Issue issue, Map<Long, Goal> goalsMap) {
        if (issue.getO(GOALS).isPresent()) {
            var goals = issue.getO(GOALS);
            if (goals.size() > 1) {
                System.out.println(issue.getKey());
            }
            var goalRef = (GoalRef) ((List<?>) goals.first()).get(0);
            if (goalsMap.containsKey(goalRef.getId())) {
                return goalsMap.get(goalRef.getId());
            } else {
                Goal goal = goalsClient.getGoal(goalRef.getId());
                if (goal != null) {
                    goalsMap.put(goal.getId(), goal);
                }
                return goal;
            }
        }
        return null;
    }

    /**
     * Сформировать вики-разметку для всех отсутствующих
     */
    private String absences(Set<String> logins, LocalDate startDate, LocalDate endDate) {
        Map<String, List<Gap>> gaps = staffClient.findGaps(startDate, endDate, logins);

        StringBuilder gapsWiki = new StringBuilder();

        Map<GapType, List<Gap>> byGapType = StreamEx.of(gaps.values())
                .flatCollection(Function.identity())
                .groupingBy(g -> GapType.valueOf(g.getWorkflow().toUpperCase()));

        for (GapType gapType : byGapType.keySet()) {
            gapsWiki.append(header2(GAP_HEADERS.get(gapType))).append("\n");
            for (Gap gap : byGapType.get(gapType)) {
                gapsWiki.append(
                        li(1,
                                staff(gap.getPersonLogin()) +
                                        ": c " + gap.getDateFrom().format(DAY_FORMATTER) +
                                        " по " + gap.getDateTo().minusDays(1).format(DAY_FORMATTER)))
                        .append("\n");
            }
        }
        return gapsWiki.toString();
    }

    /**
     * Запрос для получения списка логинов по группам на стафе
     * Можно использовать самое верхнее подразделение(службу, отдел)
     */
    private String buildStaffGroupRequest(Set<String> staffGroups) {
        return StreamEx.of(staffGroups)
                .map(s -> String.format("department_group.department.url==\"%s\" " +
                        "or department_group.ancestors.department.url==\"%s\"", s, s))
                .joining(" or ");
    }

    /**
     * Запрос для получения списка логинов по группам на стафе
     * Можно использовать самое верхнее подразделение(службу, отдел)
     */
    private String buildStaffHeadsRequest(Set<String> headLogins) {
        return StreamEx.of(headLogins)
                .map(s -> String.format("login==\"%s\"", s))
                .joining(" or ");
    }

    /**
     * Запрос для получения списка логинов по группам на стафе
     * Можно использовать самое верхнее подразделение(службу, отдел)
     */
    private String stGroupQuery(Set<String> staffGroups) {
        String res = StreamEx.of(staffGroups)
                .map(s -> String.format("Assignee: group( value: %s)", s))
                .joining(" or ");
        return String.format("(%s) ", res);
    }

    /**
     * Получить "главный" эпик
     * Некоторым удобно использовать эпики, но у тикета может быть только один эпик
     * Чтобы это побороть используется связка "Blocker to"
     * Если эпик блокирует другой эпик, алгоритм идёт наверх по этой связке до самого высокого эпика
     */
    private Issue mainEpic(Issue epic) {
        if (mainEpics.containsKey(epic.getKey())) {
            return mainEpics.get(epic.getKey());
        }
        Issue mainEpic = epic;
        while (true) {
            Issue epicRoot = getParentEpic(mainEpic);
            if (epicRoot == null) {
                break;
            }
            mainEpic = mainEpic(epicRoot);
        }
        mainEpics.put(epic.getKey(), mainEpic);
        return mainEpic;
    }

    /**
     * получить "родительский" эпик
     */
    private Issue getParentEpic(Issue epic) {
        var epicRoot = epic.getLinks().filter(l -> l.getType().getId().equals("depends") &&
                l.getObject().load().getType().getKey().equals("epic")
                && l.getDirection().value().equals("inward")).firstO();
        if (epicRoot.isPresent()) {
            return epicRoot.get().getObject().load();
        } else {
            return null;
        }
    }

    /**
     * Получить самый верхний parent
     */
    private static Issue getRoot(Issue issue) {
        while (issue.getParent().isPresent()) {
            try {
                issue = issue.getParent().get().load();
            } catch (ForbiddenException e) {
                break;
            }
        }
        return issue.getEpic().map(IssueRef::load).getOrElse(issue);
    }

    private String header1(String text) {
        return "===" + text;
    }

    private String header2(String text) {
        return "====" + text;
    }

    private String staff(String login) {
        return "staff:" + login;
    }

    private String li(int level, String text) {
        return Strings.repeat(" ", level * 2) + "* " + text;
    }

    private String link(String link, String text) {
        return String.format("((%s %s))", link, text);
    }

    private String collapse(String title, String text) {
        return String.format("<{%s\n%s}>", title, text);
    }
}
