package ru.yandex.direct.tools;

import java.io.IOException;
import java.nio.file.Path;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.http.client.HttpClient;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.config.DirectConfigFactory;
import ru.yandex.direct.utils.ThreadUtils;
import ru.yandex.direct.utils.io.FileUtils;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.startrek.client.model.Field;
import ru.yandex.startrek.client.model.Language;

import static ru.yandex.direct.tools.Constants.SERVICE;

@ParametersAreNonnullByDefault
public class StartrekTools {
    private static final Logger logger = LoggerFactory.getLogger(StartrekTools.class);

    private StartrekTools() {
    }

    private static String loadStartrekToken() {
        logger.debug("loading OAuth token from config");
        String tokenPath = DirectConfigFactory.getConfig()
                .getBranch(SERVICE)
                .getString("startrek_token_file");
        Path path = FileUtils.expandHome(tokenPath);
        return FileUtils.slurp(path).trim();
    }

    public static Session createClient() {
        return createClient((Language) null);
    }

    public static Session createClient(@Nullable Language language) {
        return createClient(Map.of(), language);
    }

    public static Session createClient(Map<String, Field.Schema> customFields) {
        return createClient(customFields, null);
    }

    public static Session createClient(Map<String, Field.Schema> customFields, @Nullable Language language) {
        String oauthToken = loadStartrekToken();

        var builder = StartrekClientBuilder.newBuilder()
                .uri("https://st-api.yandex-team.ru")
                .maxConnections(3)
                .httpClient(buildClient())
                .customFields(Cf.wrap(customFields));
        if (language != null) {
            builder.language(language);
        }

        return builder.build(oauthToken);
    }

    private static HttpClient buildClient() {
        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", PlainConnectionSocketFactory.getSocketFactory())
                .register("https", SSLConnectionSocketFactory.getSocketFactory())
                .build();

        HttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager(registry);

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(40000)
                .setConnectTimeout(5000)
                .setSocketTimeout(40000)
                .setCookieSpec(CookieSpecs.DEFAULT)
                .build();

        return HttpClientBuilder.create()
                .setUserAgent("direct-utils")
                .setDefaultRequestConfig(requestConfig)
                .setMaxConnPerRoute(3)
                .setMaxConnTotal(3)
                .setRetryHandler(new RetryHandler(3))
                .setConnectionManager(connManager)
                .build();
    }

    static class RetryHandler extends DefaultHttpRequestRetryHandler {
        protected RetryHandler(int retryCount) {
            super(retryCount, false, Collections.emptyList());
        }

        @Override
        public boolean retryRequest(IOException exception, int executionCount, HttpContext context) {
            ThreadUtils.sleep(5, ChronoUnit.SECONDS);
            return super.retryRequest(exception, executionCount, context);
        }
    }
}
