package ru.yandex.direct.tools.goals;

import java.time.Instant;
import java.time.ZoneId;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import ru.yandex.bolts.collection.Option;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.error.ForbiddenException;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.LinkDirection;
import ru.yandex.startrek.client.model.LocalLink;
import ru.yandex.startrek.client.model.TranslationRef;

public class GoalsClient {

    private final Session stClient;

    public GoalsClient(Session stClient) {
        this.stClient = stClient;
    }

    public Set<Goal> getRelatedGoals(Long goalsId) {
        List<Issue> issues = stClient.issues().get("GOALZ-" + goalsId)
                .getLinks()
                .stream()
                .filter(t -> t.getDirection().equals(LinkDirection.OUTWARD))
                .map(LocalLink::getObject)
                .filter(i -> i.getKey().startsWith("GOALZ"))
                .map(i -> {
                    try {
                        return i.load();
                    } catch (ForbiddenException e) { //внезапно к некоторым целям нет доступа
                        return null;
                    }
                })
                .filter(Objects::nonNull)
                .filter(i -> !i.getStatus().getKey().equals("achieved"))
                .collect(Collectors.toList());
        Set<Goal> result = new HashSet<>();
        for (Issue issue : issues) {
            result.add(toGoal(issue));
        }

        return result;
    }

    @Nullable
    public Goal getGoal(Long goalId) {
        Issue issue = stClient.issues().get("GOALZ-" + goalId).load();
        return toGoal(issue);
    }

    private Goal toGoal(Issue issue) {
        Goal goal = new Goal();
        goal.setTitle(issue.getSummary());
        var translationRef = issue.<Option<TranslationRef>>getO("goalImportance").first().get();
        goal.setImportance(translationRef.getId());
        goal.setId(Long.parseLong(issue.getKey().replace("GOALZ-", "")));
        goal.setComments(issue.getComments()
                .stream()
                .map(c -> new GoalComment()
                        .withCreated(Instant.ofEpochMilli(c.getCreatedAt().getMillis())
                                .atZone(ZoneId.systemDefault()).toLocalDateTime())
                        .withUpdated(Instant.ofEpochMilli(c.getUpdatedAt().getMillis())
                                .atZone(ZoneId.systemDefault()).toLocalDateTime())
                        .withOwnerLogin(c.getCreatedBy().getLogin())
                        .withId(c.getId())
                        .withText(c.getText().get())
                ).collect(Collectors.toList()));
        return goal;
    }
}
