package ru.yandex.direct.mirrors.preprocessor

import ru.yandex.direct.mirrors.preprocessor.PreprocessorUtils.isGzipFile
import java.io.BufferedReader
import java.io.FileInputStream
import java.util.ArrayList
import java.util.NoSuchElementException
import java.util.zip.GZIPInputStream

/**
 * парсер файлов в формате mirrors.res
 */
class MirrorsFileReader(private val filename: String) : Iterable<List<MirrorHost>> {
    companion object {
        private val LINE_REGEXP = Regex("""^[0-9]+\s+[0-9]+\s+(\S+)\s*$""")
    }

    override fun iterator(): MirrorsIterator {
        return MirrorsIterator()
    }

    /**
     * Итератор над файлом mirrors.res. Если нужно прервать чтение раньше конца файла - необходимо самому
     * вызвать close() на итераторе
     */
    inner class MirrorsIterator : Iterator<List<MirrorHost>>, AutoCloseable {
        private val reader: BufferedReader

        private var closed = false
        private var next: List<MirrorHost>? = null

        init {
            val fis = FileInputStream(filename)
            reader = try {
                val inp = if (isGzipFile(filename)) GZIPInputStream(fis) else fis
                inp.bufferedReader()
            } catch (e: Exception) {
                fis.close()
                throw e
            }
        }

        override fun hasNext() = when {
            next != null ->
                true
            closed ->
                false
            else -> {
                next = nextGroup()
                next != null
            }
        }

        override fun next(): List<MirrorHost> {
            if (hasNext()) {
                val ret = next!!
                next = null
                return ret
            }
            throw NoSuchElementException()
        }

        override fun close() {
            reader.close()
        }

        private fun nextGroup(): List<MirrorHost>? {
            val hosts = ArrayList<MirrorHost>()
            while (true) {
                val line = reader.readLine()
                if (line == null) {
                    break
                } else if (line.isBlank()) {
                    if (hosts.isNotEmpty()) {
                        return hosts
                    }
                } else {
                    val match = LINE_REGEXP.matchEntire(line)
                        ?: throw IllegalArgumentException("Can't parse line: $line")
                    hosts.add(MirrorHost.parse(match.groupValues[1]))
                }
            }

            if (hosts.isNotEmpty()) {
                return hosts
            } else {
                reader.close()
                closed = true
                return null
            }
        }
    }
}
