package ru.yandex.direct.mirrors.preprocessor

import ru.yandex.direct.mirrors.preprocessor.PreprocessorUtils.isGzipFile
import java.io.File
import java.io.FileOutputStream
import java.io.OutputStreamWriter
import java.nio.file.Files
import java.nio.file.Paths
import java.nio.file.StandardCopyOption
import java.util.HashSet
import java.util.zip.GZIPOutputStream
import kotlin.system.exitProcess

/**
 * Основной класс приложения, подробности см. в README.md
 */
class MirrorsPreprocessor(
    private val reader: Iterable<List<MirrorHost>>
) {
    internal constructor(srcFilename: String) : this(MirrorsFileReader(srcFilename))

    fun preprocess(resultFilename: String) {
        writeResultFile(resultFilename, makeHttpsSkipList())
    }

    fun writeResultFile(resultFilename: String, httpsSkiplist: Set<String>) {
        val tmpFile = "$resultFilename.tmp"
        FileOutputStream(tmpFile).use { fos ->
            OutputStreamWriter(
                if (isGzipFile(resultFilename)) GZIPOutputStream(fos)
                else fos
            ).use { out ->
                for (mirrorHosts in reader) {
                    val line = mirrorHosts
                        .filterNot { it.isHttps && it.host in httpsSkiplist }
                        .map { it.host }
                        .joinToString(" ")
                    if (line.isNotEmpty()) {
                        out.write(line)
                        out.write('\n'.toInt())
                    }
                }
            }
        }
        Files.move(
            Paths.get(tmpFile), Paths.get(resultFilename),
            StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING
        )
    }

    // получить множество хостов, для которых https зеркало не совпадает с http
    fun makeHttpsSkipList(): Set<String> {
        // сначала ищем все https в группах без соответствующего http
        val https: MutableSet<String> = HashSet()
        for (mirrorHosts in reader) {
            val groupHttps: MutableSet<String> = HashSet()
            for (mirrorHost in mirrorHosts) {
                if (mirrorHost.isHttps) {
                    groupHttps.add(mirrorHost.host)
                }
            }
            for (mirrorHost in mirrorHosts) {
                if (!mirrorHost.isHttps) {
                    groupHttps.remove(mirrorHost.host)
                }
            }
            https.addAll(groupHttps)
        }

        // собираем те домены, у которых есть http версия, смотрящая в другое место
        return reader.asSequence()
            .flatten()
            .filter { !it.isHttps && it.host in https }
            .map { it.host }
            .toSet()
    }

    companion object {
        @JvmStatic
        fun main(args: Array<String>) {
            if (args.size != 2 || !File(args[0]).exists()) {
                println(
                    "Usage: java -cp..."
                            + " " + MirrorsPreprocessor::class.java.canonicalName
                            + " " + "original_file"
                            + " " + "result_file"
                )
                exitProcess(1)
            }
            MirrorsPreprocessor(args[0]).preprocess(args[1])
        }
    }
}
