package ru.yandex.direct.mysql;

import java.nio.charset.StandardCharsets;

import org.slf4j.Logger;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;

import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.logging.LoggingInitializer;
import ru.yandex.direct.logging.LoggingInitializerParams;
import ru.yandex.direct.mysql.ytsync.configuration.YtSyncSensorRegistry;
import ru.yandex.direct.mysql.ytsync.export.components.TablesExporter;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.MysqlToYtSyncStreamerCoordinator;
import ru.yandex.direct.mysql.ytsync.task.config.DirectYtSyncConfig;
import ru.yandex.misc.sunMiscSignal.SunMiscSignal;

public class ProcessBinlogFromServer {
    private static final Logger logger = LoggingInitializer.getLogger(ProcessBinlogFromServer.class);

    public static void process(String[] args, Class<?> springConfiguration) {
        LoggingInitializerParams loggingParams = new LoggingInitializerParams();
        ParserWithHelp.parse(RunInitialImport.class.getCanonicalName(), args, loggingParams);
        LoggingInitializer.initialize(loggingParams, System.getProperty("YT_SYNC_LOG_NAME", "binlog-to-yt-dev"));

        String encoding = System.getProperty("file.encoding");
        if (!StandardCharsets.UTF_8.name().equals(encoding)) {
            throw new IllegalStateException(String.format("Started with incorrect encoding: %s, expected %s", encoding,
                    StandardCharsets.UTF_8.name()));
        }

        try (AnnotationConfigApplicationContext context =
                     new AnnotationConfigApplicationContext(springConfiguration)) {
            Thread mainThread = Thread.currentThread();
            SunMiscSignal.Handler handler = unused -> {
                if (context.isRunning()) {
                    logger.error("Trying to shutdown...");
                    context.close();
                    logger.error("Interrupting main thread...");
                    mainThread.interrupt();
                } else {
                    logger.error("Signal received twice, exiting...");
                    System.exit(1);
                }
            };
            SunMiscSignal.handle("INT", handler);
            SunMiscSignal.handle("TERM", handler);

            DirectYtSyncConfig directYtSyncConfig = context.getBean(DirectYtSyncConfig.class);
            YtSyncSensorRegistry ytSyncSensorRegistry = context.getBean(YtSyncSensorRegistry.class);

            // Запускаем начальный импорт, если это включено в конфиге
            TablesExporter tablesExporter = context.getBean(TablesExporter.class);
            ytSyncSensorRegistry.addLazyRegisteringFunc(tablesExporter::addInitialImportMetrics);
            if (directYtSyncConfig.importAutorun()) {
                tablesExporter.runInitialImport();
            } else {
                tablesExporter.requireInitialImport();
            }

            // Делаем проверку, что таблицы соответствуют тому, что ожидается после успешного импорта
            tablesExporter.verifyInitialImport();

            // Стартуем компонент синхронизации
            MysqlToYtSyncStreamerCoordinator coordinator = context.getBean(MysqlToYtSyncStreamerCoordinator.class);
            coordinator.startSync();
        } catch (Throwable e) {
            logger.error("Unexpected exception in main", e);
            throw e;
        }
    }
}
