package ru.yandex.direct.mysql.ytsync.configuration

import com.typesafe.config.Config
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.context.annotation.Import
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.env.EnvironmentType
import ru.yandex.direct.mysql.ytsync.configuration.YtSyncSpringConfiguration.YT_SYNC_CLUSTER
import ru.yandex.direct.mysql.ytsync.configuration.YtSyncSpringConfiguration.YT_SYNC_DB_NAMES
import ru.yandex.direct.mysql.ytsync.implementations.MySqlYtSyncAppDestroyer
import ru.yandex.direct.ytwrapper.chooser.WorkModeChooser
import ru.yandex.direct.ytwrapper.client.YtClusterConfig
import ru.yandex.direct.ytwrapper.client.YtClusterConfigProvider
import ru.yandex.direct.ytwrapper.model.YtCluster

@Configuration
@Import(YtSyncSpringConfiguration::class)
class M2YtSpringConfiguration {

    @Bean(YT_LOCK_CLUSTER)
    fun cluster(directConfig: DirectConfig): String {
        return directConfig.getString("yt-sync.cluster")
    }

    @Bean(YT_LOCK_CLUSTER_WORK_MODES_CONFIG)
    fun clusterWorkModesConfig(
        directConfig: DirectConfig,
        @Qualifier(YT_LOCK_CLUSTER) cluster: String
    ): Config {
        return directConfig.config.getConfig("yt-sync.yt-lock.m2yt.work_modes_per_cluster.$cluster")
    }

    @Bean(YT_LOCK_WORK_MODES)
    fun workModes(
        @Qualifier(YT_LOCK_CLUSTER_WORK_MODES_CONFIG) clusterWorkModesConfig: Config,
        @Qualifier(YT_LOCK_CLUSTER) ytLockCluster: String
    ): List<M2YtWorkMode> {
        val chunksPerInstance = clusterWorkModesConfig.getInt("dbs_per_instance")

        val dbNames = clusterWorkModesConfig.getStringList("dbnames")
        return dbNames.chunked(chunksPerInstance)
            .map { M2YtWorkMode(ytLockCluster, it) }
            .toList()
    }

    @Bean(YT_LOCK_YT_CONFIG)
    fun lockingYtConfig(
        @Qualifier(YT_LOCK_CLUSTER_WORK_MODES_CONFIG) clusterWorkModesConfig: Config,
        ytClusterConfigProvider: YtClusterConfigProvider
    ): YtClusterConfig {
        val clusterName = clusterWorkModesConfig.getString("lock_cluster")
        val cluster = YtCluster.clusterFromNameToUpperCase(clusterName)

        return ytClusterConfigProvider[cluster]
    }

    @Bean
    fun workModeChooser(
        directConfig: DirectConfig,
        @Qualifier(YT_LOCK_YT_CONFIG) lockingYtConfig: YtClusterConfig,
        environmentType: EnvironmentType,
        appDestroyer: MySqlYtSyncAppDestroyer,
        @Qualifier(YT_LOCK_WORK_MODES) workModes: List<M2YtWorkMode>,
        @Qualifier(YT_LOCK_CLUSTER) ytLockCluster: String
    ): WorkModeChooser<M2YtWorkMode> {
        var syncState = "prestable"
        if (directConfig.hasPath("yt-sync.sync-state")) {
            val state = directConfig.getString("yt-sync.sync-state")
            if ("stable" == state) {
                syncState = "stable"
            }
        }

        val workModeLocksSubpath = directConfig.getString("yt-sync.yt-lock.m2yt.work_mode_locks_subpath")
        val totalWorkModeLocksSubpath = "$workModeLocksSubpath/$ytLockCluster/$syncState"

        return WorkModeChooser(
            lockingYtConfig,
            totalWorkModeLocksSubpath,
            environmentType,
            appDestroyer,
            workModes
        )
    }

    @Bean
    fun ytLockWorkMode(workModeChooser: WorkModeChooser<M2YtWorkMode>): M2YtWorkMode {
        return workModeChooser.getOrRequestWorkMode()
    }

    @Bean(YT_SYNC_DB_NAMES)
    fun dbShards(workMode: M2YtWorkMode): List<String> {
        return workMode.dbNames
    }

    @Bean(YT_SYNC_CLUSTER)
    fun ytSyncCluster(workMode: M2YtWorkMode): String {
        // используется дальше в yt-sync. Лучше возвращать кластер воркмода, а не yt-sync.cluster
        // т.к. тогда гарантируется начало работы только после получения воркмода
        return workMode.cluster
    }

    companion object {
        const val YT_LOCK_YT_CONFIG = "ytLockYtConfig"
        const val YT_LOCK_CLUSTER = "ytLockCluster"
        const val YT_LOCK_CLUSTER_WORK_MODES_CONFIG = "ytLockClusterWorkModesConfig"
        const val YT_LOCK_WORK_MODES = "ytLockWorkModes"
    }
}

class M2YtWorkMode(
    val cluster: String,
    val dbNames: List<String>
) {
    override fun toString(): String {
        return "group-${dbNames.joinToString(separator = "-")}"
    }
}
