package ru.yandex.direct.mysql.ytsync.configuration;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.impl.TableImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.binlogbroker.replicatetoyt.SchemaManager;
import ru.yandex.direct.binlogbroker.replicatetoyt.YtReplicator;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.mysql.MySQLDataType;
import ru.yandex.direct.mysql.schema.ColumnSchema;
import ru.yandex.direct.mysql.schema.KeyColumn;
import ru.yandex.direct.mysql.schema.ServerSchema;
import ru.yandex.direct.mysql.schema.TableSchema;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupportViaBasic;
import ru.yandex.direct.mysql.ytsync.common.keys.PivotKeys;
import ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil;
import ru.yandex.direct.mysql.ytsync.export.task.ExportConfig;
import ru.yandex.direct.mysql.ytsync.task.builders.JooqTaskBuilder;
import ru.yandex.direct.mysql.ytsync.task.builders.SyncTableConnections;
import ru.yandex.direct.mysql.ytsync.task.provider.TaskProvider;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.summingLong;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.mysql.ytsync.synchronizator.util.YtSyncUtil.getSourceColumnValue;

class YtSyncConfigUtil {

    private static final Logger logger = LoggerFactory.getLogger(YtSyncCommonUtil.class);

    static TaskProvider getTaskProvider(TableSchema table, Map<String, Long> tableSizesMap, Long biggestTableSize,
                                        String source, String dbName) {
        List<KeyColumn> keyColumns = table.getPrimaryKey().get().getColumns();
        // таблица как есть, без join-ов
        SyncTableConnections syncTableConnections =
                new SyncTableConnections(table.getName(), keyColumns.get(0).getName());
        // путь к таблице = <базовый путь версии синхронизатора>/straight/<имя таблицы MySQL>
        Long tableSize = tableSizesMap.getOrDefault(table.getName(), 0L);
        int partitions = (int) (63 * tableSize / biggestTableSize + 1);
        PivotKeys pivotKeys = PivotKeys.hashModPartitions(partitions);
        JooqTaskBuilder builder =
                new JooqTaskBuilder(syncTableConnections,
                        conf -> YPath.simple(conf.rootPath()).child(dbName).child("straight").child(table.getName()).toString(), pivotKeys);
        // формирование ключа таблицы из первичного ключа MySQL
        // первый элемент ключа - farm_hash от первой ключевой колонке MySQL mod кол-во таблетов
        List<String> keyColumnNames = keyColumns.stream().map(KeyColumn::getName).collect(toList());
        builder.expressionKey(YtReplicator.HASH_COLUMN_NAME,
                SchemaManager.getHashColumnExpression(keyColumnNames, partitions));

        Set<String> keyFieldNames = keyColumns.stream().map(KeyColumn::getName).collect(Collectors.toSet());
        for (KeyColumn key : keyColumns) {
            ColumnSchema keyColumn = table.findColumn(key.getName());
            switch (MySQLDataType.byName(keyColumn.getDataType())) {
                case INT:
                case SMALLINT:
                case TINYINT:
                    builder.longKey(table.getName(), keyColumn.getName());
                    break;

                case BIGINT:
                    if (keyColumn.getColumnType().trim().toLowerCase().endsWith("unsigned")) {
                        builder.uLongKey(table.getName(), keyColumn.getName());
                    } else {
                        builder.longKey(table.getName(), keyColumn.getName());
                    }
                    break;

                default:
                    builder.stringKey(table.getName(), keyColumn.getName());
            }
        }
        for (ColumnSchema column : table.getColumns()) {
            if (!keyFieldNames.contains(column.getName())) {
                switch (MySQLDataType.byName(column.getDataType())) {
                    case INT:
                    case SMALLINT:
                    case TINYINT:
                        builder.longField(table.getName(), column.getName());
                        break;

                    case BIGINT:
                        if (column.getColumnType().trim().toLowerCase().endsWith("unsigned")) {
                            builder.uLongField(table.getName(), column.getName());
                        } else {
                            builder.longField(table.getName(), column.getName());
                        }
                        break;

                    case BLOB:
                    case LONGBLOB:
                    case MEDIUMBLOB:
                    case VARBINARY:

                    case CHAR:
                    case ENUM:
                    case JSON:
                    case MEDIUMTEXT:
                    case SET:
                    case TEXT:
                    case TINYTEXT:
                    case VARCHAR:

                    case DATE:

                    case DATETIME:
                    case TIMESTAMP:

                    case DECIMAL:
                        builder.stringField(table.getName(), column.getName());
                        break;

                    case DOUBLE:
                    case FLOAT:
                        builder.doubleField(table.getName(), column.getName());
                        break;

                    default:
                        throw new IllegalArgumentException("Can't handle " + column.getColumnType());
                }
            }
        }
        // Добавляем ещё колонку __source__
        // Сама по себе она не очень полезна, но её удобно использовать в YQL,
        // которые мёрджат таблицы с разных шардов, а также это предотвращает ошибку
        // 'Table doesn't have data columns', которая возникает, если в таблице mysql нет неключевых колонок
        builder.constField(YtReplicator.SOURCE_COLUMN_NAME, source);
        // таблицы сортируем по общему размеру данных, чтобы самые большие читались первыми
        return new TaskProvider(builder.build(), tableSizesMap.getOrDefault(table.getName(), 0L));
    }

    public static TaskProvider getTaskProvider(String tableName, ru.yandex.yt.ytclient.tables.TableSchema table,
                                               String source, String dbName) {
        String sourceColumnName = YtReplicator.SOURCE_COLUMN_NAME;
        String hashKeyColumnName = YtReplicator.HASH_COLUMN_NAME;
        List<ru.yandex.yt.ytclient.tables.ColumnSchema> keyColumns = table.toKeys().getColumns().stream()
                .filter(c -> !c.getName().equals(sourceColumnName) && !c.getName().equals(hashKeyColumnName))
                .collect(toList());
        // таблица как есть, без join-ов
        SyncTableConnections syncTableConnections =
                new SyncTableConnections(tableName, keyColumns.get(0).getName());
        // фейковые таблеты, чтобы не падать с NPE
        PivotKeys pivotKeys = PivotKeys.signedHashPartitions(16);
        JooqTaskBuilder builder =
                new JooqTaskBuilder(syncTableConnections,
                        conf -> YPath.simple(conf.rootPath()).child(dbName).child("straight").child(tableName).toString(), pivotKeys);
        // ожидаем на первой позиции ключ __hash__
        builder.expressionKey(hashKeyColumnName, table.toKeys().getColumns().get(0).getExpression());
        for (ru.yandex.yt.ytclient.tables.ColumnSchema keyColumn : keyColumns) {
            switch (keyColumn.getType()) {
                case INT64:
                    builder.longKey(tableName, keyColumn.getName());
                    break;
                case UINT64:
                    builder.uLongKey(tableName, keyColumn.getName());
                    break;
                default:
                    builder.stringKey(tableName, keyColumn.getName());
                    break;
            }
        }
        for (ru.yandex.yt.ytclient.tables.ColumnSchema column : table.toValues().getColumns()) {
            if (column.getName().equals(YtReplicator.SOURCE_COLUMN_NAME)) {
                builder.constField(YtReplicator.SOURCE_COLUMN_NAME, source);
            } else {
                switch (column.getType()) {
                    case INT64:
                        builder.longField(tableName, column.getName());
                        break;
                    case UINT64:
                        builder.uLongField(tableName, column.getName());
                        break;
                    case DOUBLE:
                        builder.doubleField(tableName, column.getName());
                        break;
                    default:
                        builder.stringField(tableName, column.getName());
                        break;
                }
            }
        }
        return new TaskProvider(builder.build(), 0L);
    }

    /**
     * Список TaskProvider для всех MySQL таблиц Директа
     * <p>
     * Формируется из метаданных MySQL таблиц
     * <p>
     * При этом некоторые таблицы игнорируются: {@link ExportConfig#getExcludeTableNames}
     * <p>
     * Или же можно ограничить список синхронизируемых таблиц с помощью {@link ExportConfig#getIncludeTableNames}
     * <p>
     * Состав колонок ключа таблицы: '__hash__', {@link TableImpl#getPrimaryKey}
     * <p>
     * Если перичный ключ MySQL таблицы содержит все колонки таблицы,
     * в соответствующую YT таблицу добавляется фейковая колонка '__dummy'
     * <p>
     * Путь к таблице = <базовый путь версии синхронизатора>/<имя БД>/straight/<имя таблицы MySQL>
     */
    static List<TaskProvider> getAllTablesImportTaskProviders(DatabaseWrapperProvider databaseWrapperProvider,
                                                              ExportConfig exportConfig, Yt yt,
                                                              EnvironmentType environmentType,
                                                              YtSupportViaBasic ytSupportViaBasic) {
        logger.info("Generating TaskProviders started");

        // соответствие <название таблицы> -> <размер данных>
        // для приоритизации начальной загрузки по убыванию размера
        Map<String, Long> tableSizesMap = exportConfig.getDbNames().stream()
                .map(dbname -> {
                    final DatabaseWrapper wrapper = databaseWrapperProvider.get(dbname);
                    final String querySortedTablesBySize = "select TABLE_NAME, DATA_LENGTH "
                            + "from information_schema.tables "
                            + "where TABLE_SCHEMA in ('ppc', 'ppcdict') "
                            + "order by DATA_LENGTH desc";
                    return wrapper.query(querySortedTablesBySize,
                            (rs, rowNum) -> Pair.of(rs.getString(1), rs.getLong(2)));
                })
                .flatMap(Collection::stream)
                .collect(groupingBy(Pair::getKey, summingLong(Pair::getRight)));

        Long biggestTableSize = tableSizesMap.values().stream()
                .sorted(Comparator.reverseOrder())
                .limit(1)
                .findFirst()
                .orElseThrow(IllegalStateException::new);

        // Множество таблиц, исключаемых из синхронизации
        // Допускается использование регулярных выражений
        Set<String> excludeTableNames = exportConfig.getExcludeTableNames();
        // Множество таблиц, включаемых в синхронизацию
        // Если множество пустое, параметр игнорируется
        Set<String> includeTableNames = exportConfig.getIncludeTableNames();

        YPath rootPath = YPath.simple(exportConfig.rootPath());
        String dbNames = String.join("-", exportConfig.getDbNames());
        YPath initialImportFinishedPath = rootPath.child(dbNames).attribute("initial-import-finished");
        boolean initialImportFinished = yt.cypress().exists(initialImportFinishedPath) &&
                yt.cypress().get(initialImportFinishedPath).boolValue();

        List<TaskProvider> result = exportConfig.getDbNames().stream()
                .map(dbName -> {
                    // имя БД, префиксированное названием окружения, например "production:ppc:15"
                    String source = getSourceColumnValue(environmentType, dbName);

                    YPath dbNameRootPath = rootPath.child(dbName);
                    // если начальный импорт завершен,
                    // то список TaskProvider формируется из схем YT таблиц
                    // чтобы не падать от несоответствия схем
                    if (initialImportFinished) {
                        YPath tablesRootPath = dbNameRootPath.child("straight");
                        List<TaskProvider> ytTableSchemaProviders = yt.cypress().list(tablesRootPath, Cf.set("type", "dynamic", "schema")).stream()
                                .filter(node -> "table".equals(node.getAttribute("type").map(YTreeNode::stringValue).orElse(null))
                                        && node.getAttribute("dynamic").get().boolValue())
                                .filter(node -> excludeTableNames.stream()
                                        .noneMatch(exclude -> node.stringValue().toLowerCase().matches(exclude)))
                                .filter(node -> includeTableNames.isEmpty()
                                        || includeTableNames.contains(node.stringValue().toLowerCase()))
                                .map(node -> getTaskProvider(node.stringValue(), node.getAttribute("schema")
                                        .map(ru.yandex.yt.ytclient.tables.TableSchema::fromYTree).get(), source, dbName))
                                .sorted(Comparator.comparing(TaskProvider::getTaskSize).reversed())
                                .collect(toList());
                        return ytTableSchemaProviders;
                    } else {
                        ServerSchema serverSchema;
                        DatabaseWrapper databaseWrapper = databaseWrapperProvider.get(dbName);
                        try (Connection connection = databaseWrapper.getDataSource().getConnection()) {
                            serverSchema = ServerSchema.dump(connection);
                        } catch (SQLException e) {
                            throw new RuntimeException(e);
                        }
                        List<TableSchema> tables = serverSchema.getDatabases().stream()
                                .filter(db -> db.getName().equals(dbName.split(":")[0]))
                                .findFirst().get()
                                .getTables();
                        List<TaskProvider> tableSchemaTaskProviders = tables.stream()
                                .filter(t -> excludeTableNames.stream()
                                        .noneMatch(exclude -> t.getName().toLowerCase().matches(exclude)))
                                .filter(t -> includeTableNames.isEmpty() || includeTableNames.contains(t.getName().toLowerCase()))
                                .map(table -> getTaskProvider(table, tableSizesMap, biggestTableSize, source, dbName))
                                .sorted(Comparator.comparing(TaskProvider::getTaskSize).reversed())
                                .collect(toList());
                        return tableSchemaTaskProviders;
                    }
                })
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
        logger.info("Generating TaskProviders finished");
        return result;
    }
}
