package ru.yandex.direct.mysql.ytsync.configuration;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.PreDestroy;

import ru.yandex.direct.mysql.ytsync.synchronizator.monitoring.SyncState;
import ru.yandex.direct.mysql.ytsync.synchronizator.monitoring.SyncStateChecker;
import ru.yandex.direct.mysql.ytsync.task.config.DirectYtSyncConfig;
import ru.yandex.direct.version.DirectVersion;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.monlib.metrics.JvmGc;
import ru.yandex.monlib.metrics.JvmMemory;
import ru.yandex.monlib.metrics.JvmRuntime;
import ru.yandex.monlib.metrics.JvmThreads;
import ru.yandex.monlib.metrics.Metric;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeDouble;
import ru.yandex.monlib.metrics.registry.MetricId;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.util.Collections.synchronizedList;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Специальная обёртка над SensorRegistry, которая позволяет добавлять сенсоры,
 * одна из меток которого (sync_state) может меняться динамически в зависимости от того,
 * какую роль выполняет именно этот инстанс приложения. При переключении sync_state
 * все сенсоры пересоздаются автоматически, и поставщику метрик не нужно специально обрабатывать этот сценарий.
 * <p>
 * Некоторое неудобство заключается в том, что поддержать все factory-методы SensorRegistry не удалось,
 * и при необходимости нужно будет добавлять недостающие методы вручную.
 */
public class YtSyncSensorRegistry {
    private final SyncStateChecker syncStateChecker;

    enum MetricType {
        GAUGE_DOUBLE,
        GAUGE_INT64,
    }

    private YtCluster ytCluster;
    private Labels subRegistryLabels;
    private MetricRegistry subRegistry;
    private Map<MetricId, Metric> metrics = new ConcurrentHashMap<>();
    private Map<MetricId, MetricType> metricTypes = new ConcurrentHashMap<>();
    private List<Consumer<MetricRegistry>> lazySensorsRegistrars = synchronizedList(new ArrayList<>());
    private Consumer<SyncState> handler;

    YtSyncSensorRegistry(DirectYtSyncConfig directYtSyncConfig,
                         SyncStateChecker syncStateChecker) {
        this.syncStateChecker = syncStateChecker;
        this.ytCluster = directYtSyncConfig.getCluster();

        // JVM-related метрики собираем всегда
        addLazyRegisteringFunc((metricRegistry) -> {
            JvmGc.addMetrics(metricRegistry);
            JvmRuntime.addMetrics(metricRegistry);
            JvmThreads.addMetrics(metricRegistry);
            JvmMemory.addMetrics(metricRegistry);

            metricRegistry.lazyCounter("sensors_total", SOLOMON_REGISTRY::estimateCount);

            metricRegistry.lazyGaugeInt64("major_version", DirectVersion::getMajorVersion);
            metricRegistry.lazyGaugeInt64("minor_version", DirectVersion::getMinorVersion);
        });
        //
        onSyncStateChanged(syncStateChecker.getSyncState());
        //
        this.handler = this::onSyncStateChanged;
        syncStateChecker.subscribe(handler);
    }

    synchronized private void onSyncStateChanged(SyncState syncState) {
        if (subRegistryLabels != null) {
            SOLOMON_REGISTRY.removeSubRegistry(subRegistryLabels);
        }
        subRegistryLabels = Labels.of("sync_state", syncState.getName(), "yt_cluster", ytCluster.getName());
        subRegistry = SOLOMON_REGISTRY.subRegistry(subRegistryLabels);

        // Пересоздаём обычные (not lazy) сенсоры
        for (Map.Entry<MetricId, MetricType> entry : metricTypes.entrySet()) {
            MetricId metricId = entry.getKey();
            MetricType sensorType = entry.getValue();
            Metric metric;
            switch (sensorType) {
                case GAUGE_DOUBLE: {
                    metric = subRegistry.gaugeDouble(metricId.getName(), metricId.getLabels());
                    break;
                }
                case GAUGE_INT64: {
                    metric = subRegistry.gaugeInt64(metricId.getName(), metricId.getLabels());
                    break;
                }
                default: {
                    throw new IllegalStateException("Type not supported yet");
                }
            }
            metrics.put(metricId, metric);
        }

        // Пересоздаём lazy-сенсоры
        for (Consumer<MetricRegistry> lazySensorsRegistrar : lazySensorsRegistrars) {
            lazySensorsRegistrar.accept(subRegistry);
        }
    }

    /**
     * Добавить сенсор
     */
    synchronized public void gaugeDouble(MetricId metricId) {
        GaugeDouble gaugeDouble = subRegistry.gaugeDouble(metricId.getName(), metricId.getLabels());
        if (metrics.containsKey(metricId)) {
            throw new IllegalStateException(String.format("Metric '%s' already registered", metricId));
        }
        metrics.put(metricId, gaugeDouble);
        metricTypes.put(metricId, MetricType.GAUGE_DOUBLE);
    }

    /**
     * Получить актуальный сенсор по ключу metricId
     */
    @Nullable
    synchronized public GaugeDouble getGaugeDouble(MetricId metricId) {
        return (GaugeDouble) metrics.get(metricId);
    }

    /**
     * Добавить функцию, которая будет вызываться всякий раз при изменении syncState (и пересоздании registry).
     * В этом колбеке можно зарегистрировать lazy-сенсоры, и они будут пересоздаваться.
     * Обычные сенсоры (not lazy) регистрировать таким образом не имеет смысла, потому что ссылка на
     * созданный сенсор со временем может стать неактуальной (после очередного изменения syncState).
     * Для регистрации обычных сенсоров можно пользоваться отдельными методами.
     */
    synchronized public void addLazyRegisteringFunc(Consumer<MetricRegistry> lazySensorsRegistrar) {
        lazySensorsRegistrars.add(lazySensorsRegistrar);
        if (subRegistry != null) {
            lazySensorsRegistrar.accept(subRegistry);
        }
    }

    @PreDestroy
    synchronized public void preDestroy() {
        syncStateChecker.unsubscribe(handler);
        if (subRegistryLabels != null) {
            SOLOMON_REGISTRY.removeSubRegistry(subRegistryLabels);
        }
    }
}
