package ru.yandex.direct.mysql.ytsync.configuration;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.common.configuration.CommonConfiguration;
import ru.yandex.direct.common.configuration.MetricsConfiguration;
import ru.yandex.direct.common.configuration.MonitoringHttpServerConfiguration;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.config.EssentialConfiguration;
import ru.yandex.direct.dbutil.configuration.DbUtilConfiguration;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.mysql.ytsync.common.compatibility.BasicYtSupport;
import ru.yandex.direct.mysql.ytsync.common.compatibility.BasicYtSupportViaKosher;
import ru.yandex.direct.mysql.ytsync.common.compatibility.BasicYtSupportViaYtClient;
import ru.yandex.direct.mysql.ytsync.common.compatibility.BasicYtSupportWithChunking;
import ru.yandex.direct.mysql.ytsync.common.compatibility.BasicYtSupportWithRetries;
import ru.yandex.direct.mysql.ytsync.common.compatibility.KosherCompressorFactory;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupportViaBasic;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupportWithRunTxRetries;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesConfig;
import ru.yandex.direct.mysql.ytsync.export.configuration.MySqlToYtExportConfiguration;
import ru.yandex.direct.mysql.ytsync.export.task.ExportConfig;
import ru.yandex.direct.mysql.ytsync.export.task.TableExportTask;
import ru.yandex.direct.mysql.ytsync.export.task.TableExportTemplate;
import ru.yandex.direct.mysql.ytsync.synchronizator.configuration.MySqlToYtSyncronizatorConfiguration;
import ru.yandex.direct.mysql.ytsync.synchronizator.monitoring.SyncStateChecker;
import ru.yandex.direct.mysql.ytsync.synchronizator.tableprocessors.TableProcessor;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SyncConfig;
import ru.yandex.direct.mysql.ytsync.task.config.DirectYtSyncConfig;
import ru.yandex.direct.mysql.ytsync.task.config.ExportConfigImpl;
import ru.yandex.direct.mysql.ytsync.task.config.SyncConfigImpl;
import ru.yandex.direct.mysql.ytsync.task.provider.TaskProvider;
import ru.yandex.direct.mysql.ytsync.task.provider.TaskProviderUtil;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtClusterConfigProvider;
import ru.yandex.direct.ytwrapper.client.YtClusterTypesafeConfigProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtConfiguration;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.yt.ytclient.bus.BusConnector;
import ru.yandex.yt.ytclient.bus.DefaultBusConnector;
import ru.yandex.yt.ytclient.proxy.ApiServiceClient;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.rpc.RpcCredentials;
import ru.yandex.yt.ytclient.rpc.RpcOptions;
import ru.yandex.yt.ytclient.tables.TableSchema;

@Configuration
@Import({
        EssentialConfiguration.class,
        DbUtilConfiguration.class,
        // Нужен NetAcl для monitoringServlet
        CommonConfiguration.class,
        MySqlToYtExportConfiguration.class,
        MySqlToYtSyncronizatorConfiguration.class,
        MetricsConfiguration.class,
        MonitoringHttpServerConfiguration.class
})
@ComponentScan(
        basePackages = {
                "ru.yandex.direct.mysql.ytsync.implementations",
                "ru.yandex.direct.mysql.ytsync.task",
        }
)
public class YtSyncSpringConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(YtSyncSpringConfiguration.class);

    private static final String TASK_PROVIDERS_LIST_BEAN_NAME = "task-providers";
    public static final String YT_SYNC_DB_NAMES = "ytSyncDbNames";
    public static final String YT_SYNC_CLUSTER = "ytSyncCluster";

    private final AtomicInteger asyncThreadCount = new AtomicInteger(0);
    private final AtomicInteger requestThreadCount = new AtomicInteger(0);
    private final AtomicInteger cachedThreadCount = new AtomicInteger(0);

    @Bean(YT_SYNC_DB_NAMES)
    List<String> dbShards() {
        throw new IllegalStateException("override me in another configuration");
    }

    @Bean(YT_SYNC_CLUSTER)
    String ytSyncCluster() {
        throw new IllegalStateException("override me in another configuration");
    }

    @Bean
    YtClusterConfigProvider ytClusterConfigProvider(DirectConfig config) {
        return new YtClusterTypesafeConfigProvider(config.getBranch("yt").getConfig());
    }

    @Bean
    YtClusterConfig ytClusterConfig(@Qualifier(YT_SYNC_CLUSTER) String ytSyncCluster,
                                    YtClusterConfigProvider ytClusterConfigProvider) {
        YtCluster cluster = YtCluster.clusterFromNameToUpperCase(ytSyncCluster);
        return ytClusterConfigProvider.get(cluster);
    }

    @Bean
    ExportConfigImpl exportConfigImpl(DirectConfig config, YtClusterConfig ytClusterConfig,
                                      @Qualifier(YT_SYNC_DB_NAMES) List<String> dbNames) {
        return new ExportConfigImpl(config, ytClusterConfig, dbNames);
    }

    @Bean
    ExportConfig exportConfig(ExportConfigImpl exportConfigImpl) {
        return exportConfigImpl;
    }

    @Bean
    SyncConfig syncConfig(DirectConfig config, @Qualifier(YT_SYNC_DB_NAMES) List<String> dbNames) {
        return new SyncConfigImpl(config, dbNames);
    }

    @Bean
    SyncStatesConfig syncStatesConfig(ExportConfigImpl exportConfig) {
        return exportConfig;
    }

    @Bean
    DirectYtSyncConfig ytSyncConfig(DirectConfig config, YtClusterConfig ytClusterConfig) {
        return new DirectYtSyncConfig(config, ytClusterConfig);
    }

    @Bean(TASK_PROVIDERS_LIST_BEAN_NAME)
    List<TaskProvider> taskProviders(ApplicationContext context, DirectYtSyncConfig directYtSyncConfig,
                                     DatabaseWrapperProvider databaseWrapperProvider, ExportConfigImpl exportConfig,
                                     Yt yt, EnvironmentType environmentType, YtSupportViaBasic ytSupportViaBasic) {
        return getTaskProviders(context, directYtSyncConfig, databaseWrapperProvider, exportConfig, yt, environmentType,
                ytSupportViaBasic);
    }

    public List<TaskProvider> getTaskProviders(ApplicationContext context, DirectYtSyncConfig directYtSyncConfig,
                                               DatabaseWrapperProvider databaseWrapperProvider,
                                               ExportConfigImpl exportConfig, Yt yt,
                                               EnvironmentType environmentType, YtSupportViaBasic ytSupportViaBasic) {
        if (directYtSyncConfig.importAllTables()) {
            return YtSyncConfigUtil.getAllTablesImportTaskProviders(databaseWrapperProvider, exportConfig, yt,
                    environmentType, ytSupportViaBasic);
        } else {
            // Нам важен порядок по убыванию размера, чтобы максимально эффективно использовать квоту по месту в YT
            // Каждая таблица для выгрузки требует 2х места, поэтому таблицы побольше мы выгружаем в начале
            return context.getBeansOfType(TaskProvider.class).values().stream()
                    .sorted(Comparator.comparing(TaskProvider::getTaskSize).reversed())
                    .collect(Collectors.toList());
        }
    }

    @Bean
    Map<Pair<String, String>, TableExportTask> tableNameAndPathToTemplate(
            @Qualifier(TASK_PROVIDERS_LIST_BEAN_NAME) List<TaskProvider> taskProviders,
            DirectYtSyncConfig directYtSyncConfig) {
        return StreamEx.of(taskProviders)
                .mapToEntry(tp -> Pair.of(tp.getMainTable(), tp.getSyncTask().getTablePath(directYtSyncConfig)),
                        TaskProvider::constructTableExportTask)
                .toMap();
    }

    @Bean
    List<TableExportTemplate> tableExportTemplates(
            DirectYtSyncConfig directYtSyncConfig,
            @Qualifier(TASK_PROVIDERS_LIST_BEAN_NAME) List<TaskProvider> taskProviders) {
        Map<String, Long> pathToIndex = new HashMap<>();
        for (TaskProvider taskProvider : taskProviders) {
            String path = taskProvider.getSyncTask().getTablePath(directYtSyncConfig);
            if (!pathToIndex.containsKey(path)) {
                pathToIndex.put(path, taskProvider.getTaskSize());
            }
        }

        List<TableExportTemplate> templates = taskProviders.stream()
                .map(p -> p.constructTableExportTemplate(directYtSyncConfig))
                .collect(Collectors.toList());

        Multimap<String, TableExportTemplate> pathToTemplates = HashMultimap.create();
        for (TableExportTemplate template : templates) {
            pathToTemplates.put(template.getTargetTable(), template);
        }

        // Объединяем задания и сортируем по убыванию размера
        // Нам важен порядок по убыванию размера, чтобы максимально эффективно использовать квоту по месту в YT
        // Каждая таблица для выгрузки требует 2х места, поэтому таблицы побольше мы выгружаем в начале
        return pathToTemplates.asMap().values().stream()
                .map(TableExportTemplate::merge)
                .filter(Objects::nonNull)
                .sorted(Comparator.comparing(t -> 0 - pathToIndex.get(t.getTargetTable())))
                .collect(Collectors.toList());
    }

    @Bean
    List<TableExportTemplate> tableIndexExportTemplates(
            DirectYtSyncConfig directYtSyncConfig,
            @Qualifier(TASK_PROVIDERS_LIST_BEAN_NAME) List<TaskProvider> taskProviders) {
        List<TableExportTemplate> templates = taskProviders.stream()
                .map(p -> p.constructIndexExportTemplate(directYtSyncConfig))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        // Убираем дубликаты с сохранением порядка
        // Нам важен порядок по убыванию размера, чтобы максимально эффективно использовать квоту по месту в YT
        // Каждая таблица для выгрузки требует 2х места, поэтому таблицы побольше мы выгружаем в начале
        List<TableExportTemplate> result = new ArrayList<>();
        for (TableExportTemplate template : templates) {
            if (!result.contains(template)) {
                result.add(template);
            }
        }

        return result;
    }

    @Bean
    @Lazy
    Supplier<List<TableProcessor>> tableProcessorsProvider(
            ApplicationContext context, DirectYtSyncConfig directYtSyncConfig,
            DatabaseWrapperProvider databaseWrapperProvider, ExportConfigImpl exportConfig,
            Yt yt, YtSupport ytSupport, EnvironmentType environmentType, YtSupportViaBasic ytSupportViaBasic) {
        // нельзя допускать кэширования taskProviders
        // дабы избежать просыпания бинлогов при переключении инстансов во время альтеров pt-osc
        // по умолчанию bean имеет scope singleton и кэшится при создании до рестарта приложения
        // по этой причине иньекция taskProviders в виде bean не подойдет в данном случае
        return () -> TaskProviderUtil.createTableProcessors(
                getTaskProviders(context, directYtSyncConfig, databaseWrapperProvider,
                        exportConfig, yt, environmentType, ytSupportViaBasic), ytSupport, directYtSyncConfig);
    }

    @Bean
    @Lazy
    Function<String, List<TableProcessor>> perDbNameTableProcessorsProvider(
            ApplicationContext context, DirectYtSyncConfig directYtSyncConfig,
            DatabaseWrapperProvider databaseWrapperProvider, ExportConfigImpl exportConfig,
            Yt yt, YtSupport ytSupport, EnvironmentType environmentType, YtSupportViaBasic ytSupportViaBasic) {
        // нельзя допускать кэширования taskProviders
        // дабы избежать просыпания бинлогов при переключении инстансов во время альтеров pt-osc
        // по умолчанию bean имеет scope singleton и кэшится при создании до рестарта приложения
        // по этой причине иньекция taskProviders в виде bean не подойдет в данном случае
        if (directYtSyncConfig.importAllTables()) {
            return dbName -> {
                List<TaskProvider> perDbNameTaskProviders = getTaskProviders(context, directYtSyncConfig,
                        databaseWrapperProvider,
                        exportConfig, yt, environmentType, ytSupportViaBasic).stream()
                        .filter(p -> p.getSyncTask().getTablePath(directYtSyncConfig).contains("/" + dbName + "/"))
                        .collect(Collectors.toList());
                return TaskProviderUtil.createTableProcessors(perDbNameTaskProviders, ytSupport, directYtSyncConfig);
            };
        } else {
            return dbName -> TaskProviderUtil.createTableProcessors(
                    getTaskProviders(context, directYtSyncConfig, databaseWrapperProvider,
                            exportConfig, yt, environmentType, ytSupportViaBasic), ytSupport, directYtSyncConfig);
        }
    }

    /**
     * Функция преобразования пути к существующей YT таблице и имени БД в {@link TableProcessor}
     * Нужна для динамического обновление схемы после применения Alter-ов
     */
    @Bean
    @Lazy
    BiFunction<YPath, String, TableProcessor> yPathToTaskProcessorConverter(
            YtSupport ytSupport,
            YtSupportViaBasic ytSupportViaBasic,
            DirectYtSyncConfig directYtSyncConfig, EnvironmentType environmentType) {

        return (yPath, dbName) -> {
            // имя БД, префиксированное названием окружения, например "production:ppc:15"
            String source = environmentType.name().toLowerCase() + ":" + dbName;

            TableSchema table;
            try {
                table = ytSupportViaBasic.getTableSchema(yPath.toString()).get();
            } catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException(e);
            }
            TaskProvider taskProvider = YtSyncConfigUtil.getTaskProvider(yPath.name(), table, source, dbName);
            return TaskProviderUtil.createSimpleTableProcessor(ytSupport, directYtSyncConfig, taskProvider);
        };
    }

    @Bean
    @Lazy
    ScheduledExecutorService ytAsyncExecutor(DirectYtSyncConfig config) {
        // Пул тредов для таймеров и прочей лёгкой обработки
        return Executors.newScheduledThreadPool(config.asyncWorkerThreads(), r -> {
            Thread thread = new Thread(r);
            thread.setName(String.format("yt-sync-async-%02d", asyncThreadCount.incrementAndGet()));
            thread.setDaemon(true);
            return thread;
        });
    }

    @Bean
    @Lazy
    Executor ytSyncExecutor(DirectYtSyncConfig config) {
        // Пул тредов для различных синхронных запросов
        return Executors.newFixedThreadPool(config.syncRequestThreads(), r -> {
            Thread thread = new Thread(r);
            thread.setName(String.format("yt-sync-request-%02d", requestThreadCount.incrementAndGet()));
            thread.setDaemon(true);
            return thread;
        });
    }

    @Bean
    @Lazy
    Executor ytCachedExecutor() {
        // Пул тредов для синхронной обработки без ограничений
        return Executors.newCachedThreadPool(r -> {
            Thread thread = new Thread(r);
            thread.setName(String.format("yt-cached-thread-%03d", cachedThreadCount.incrementAndGet()));
            thread.setDaemon(true);
            return thread;
        });
    }

    @Bean
    @Lazy
    Yt yt(DirectYtSyncConfig config) {
        String host = config.getYtProxyHost();
        String token = config.getYtToken();
        logger.info("Creating Yt kosher client for {}", host);

        YtConfiguration.Builder builder = YtConfiguration.builder()
                .withApiHost(host)
                .withToken(token)
                .withHeavyCommandsRetries(config.getKosherRetries())
                .withHeavyCommandsTimeout(Duration.ofSeconds(config.getKosherHeavyTimeoutSeconds()))
                .withSimpleCommandsRetries(config.getKosherRetries())
                .withWriteChunkSize(config.getKosherWriteChunkSize());

        return YtUtils.http(builder.build());
    }

    @Bean(destroyMethod = "close")
    @Lazy
    BusConnector ytBusConnector() {
        return new DefaultBusConnector();
    }

    @Bean(destroyMethod = "close")
    @Lazy
    ApiServiceClient ytClient(DirectYtSyncConfig config, BusConnector busConnector) {
        YtCluster cluster = config.getCluster();
        String httpProxyHost = config.getYtProxyHost();
        int httpProxyPort = config.getYtProxyPort();
        String user = config.getYtUser();
        String token = config.getYtToken();
        //
        logger.info("Creating Yt rpc client for httpProxy={} and user={}", httpProxyHost, user);
        //
        YtClient ytClient = null;
        try {
            ytClient = new YtClient(busConnector,
                    new ru.yandex.yt.ytclient.proxy.YtCluster(cluster.getName(), httpProxyHost, httpProxyPort),
                    new RpcCredentials(user, token),
                    new RpcOptions().setGlobalTimeout(Duration.ofSeconds(30)));
            ytClient.waitProxies().join(); // IGNORE-BAD-JOIN DIRECT-149116
        } catch (RuntimeException e) {
            if (ytClient != null) {
                ytClient.close();
            }
            throw e;
        }
        return ytClient;
    }

    @Bean
    @Lazy
    KosherCompressorFactory kosherCompressorFactory(DirectYtSyncConfig config) {
        switch (config.getKosherCompression()) {
            case "empty":
                return KosherCompressorFactory.emptyCompressor();
            case "lzop":
                return KosherCompressorFactory.lzopCompressor();
            case "deflate-1":
                return KosherCompressorFactory.deflateCompressor(1);
            default:
                throw new IllegalStateException("Unsupported kosher compression: " + config.getKosherCompression());
        }
    }

    @Bean
    @Lazy
    YtSupport ytSupport(DirectYtSyncConfig config, YtSupportViaBasic ytSupportViaBasic) {
        YtSupport ytSupport = ytSupportViaBasic;
        if (config.useRpcClient()) {
            // Добавляем надёжности через повторы целых транзакций
            ytSupport = new YtSupportWithRunTxRetries(ytSupport, Duration.ofSeconds(config.outerRetryTimeoutSeconds()));
        }
        return ytSupport;
    }

    @Bean
    @Lazy
    YtSupportViaBasic ytSupportViaBasic(DirectYtSyncConfig config, Yt yt, KosherCompressorFactory compressorFactory) {
        boolean useRpcClient = config.useRpcClient();
        Duration innerTimeout = Duration.ofSeconds(config.innerRetryTimeoutSeconds());
        Duration outerTimeout = Duration.ofSeconds(config.outerRetryTimeoutSeconds());
        // От кошерного клиента пока нельзя избавиться
        BasicYtSupport basicYtSupport = new BasicYtSupportViaKosher(yt,
                ytAsyncExecutor(config),
                ytSyncExecutor(config),
                ytCachedExecutor(),
                compressorFactory);
        if (useRpcClient) {
            // Добавляем сверху rpc клиент для работы с таблицами
            basicYtSupport = new BasicYtSupportViaYtClient(basicYtSupport,
                    ytClient(config, ytBusConnector()),
                    ytAsyncExecutor(config));
        }
        // Добавляем стабильности базовым операциям через повторы в случае их провала
        basicYtSupport = new BasicYtSupportWithRetries(basicYtSupport, innerTimeout, outerTimeout);
        // Нарезаем данные на мелкие параллельные кусочки
        basicYtSupport = new BasicYtSupportWithChunking(basicYtSupport, config.maxWriteChunkSize());
        // Оборачиваем базовую реализацию с добавлением сахара
        return new YtSupportViaBasic(basicYtSupport);
    }

    @Bean
    @Lazy
    YtSyncSensorRegistry ytSyncSensorRegistry(DirectYtSyncConfig config, SyncStateChecker syncStateChecker) {
        return new YtSyncSensorRegistry(config, syncStateChecker);
    }
}
