package ru.yandex.direct.mysql.ytsync.implementations;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.PostConstruct;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.db.config.DbConfigEvent;
import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.db.config.DbConfigListener;
import ru.yandex.direct.mysql.ytsync.common.util.MySqlConnectionProvider;
import ru.yandex.direct.mysql.ytsync.common.util.MySqlConnectionSettings;
import ru.yandex.direct.mysql.ytsync.common.util.MySqlSettingsChangedListener;
import ru.yandex.direct.mysql.ytsync.task.config.DirectYtSyncConfig;
import ru.yandex.direct.mysql.ytsync.util.MySqlUtils;

/**
 * Компонент для получения соединений к базе Директа с целью экспорта данных в YT
 * Отслеживает изменение dbConfig'а и позволяет подписаться на эти события.
 */
@Lazy
@Component
public class ConnectionProvider implements MySqlConnectionProvider, DbConfigListener, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionProvider.class);

    private final DbConfigFactory dbConfigFactory;
    private final DirectYtSyncConfig directYtSyncConfig;
    private final List<MySqlSettingsChangedListener> listeners = new ArrayList<>();

    @Autowired
    public ConnectionProvider(DbConfigFactory dbConfigFactory, DirectYtSyncConfig directYtSyncConfig) {
        this.dbConfigFactory = dbConfigFactory;
        this.directYtSyncConfig = directYtSyncConfig;
    }

    @PostConstruct
    private void afterConstruction() {
        // После выполнения конструктора передаем ссылку на полностью сконструированный объект
        dbConfigFactory.addListener(this);
    }

    @Override
    public void update(DbConfigEvent event) {
        List<MySqlSettingsChangedListener> listenersToNotify;
        synchronized (listeners) {
            listenersToNotify = new ArrayList<>(listeners);
        }
        for (MySqlSettingsChangedListener listener : listenersToNotify) {
            try {
                listener.settingsChanged();
            } catch (RuntimeException e) {
                logger.error("Error when notifying about DB settings has been changed", e);
            }
        }
    }

    public void addListener(MySqlSettingsChangedListener listener) {
        synchronized (listeners) {
            listeners.add(listener);
        }
    }

    public void removeListener(MySqlSettingsChangedListener listener) {
        synchronized (listeners) {
            listeners.remove(listener);
        }
    }

    private DbConfig getDbConfigForImport(String dbName) throws SQLException {
        DbConfig dbConfig = dbConfigFactory.get(dbName);
        if (directYtSyncConfig.importUsingSlaves(dbName)) {
            dbConfig = MySqlUtils.selectSlaveAuto(directYtSyncConfig, dbConfig);
        }
        return dbConfig;
    }

    private DbConfig getDbConfigForSync(String dbName) throws SQLException {
        DbConfig dbConfig = dbConfigFactory.get(dbName);
        if (directYtSyncConfig.syncUsingSlaves(dbName)) {
            dbConfig = MySqlUtils.selectSlaveAuto(directYtSyncConfig, dbConfig);
        }
        return dbConfig;
    }

    /**
     * Создаёт новое соединение к базе dbName (например ppc:1) с целью импорта данных
     * TODO : поддержать изменение dbConfig'а
     */
    @Override
    public Connection createConnectionForImport(String dbName) throws SQLException {
        DbConfig dbConfig = getDbConfigForImport(dbName);
        logger.info("Connecting to {}", dbConfig.getHosts().get(0));
        Connection conn = MySqlUtils.mysqlConnect(dbConfig);
        try {
            logger.info("Switching to catalog {}", dbConfig.getDb());
            conn.setCatalog(dbConfig.getDb());
        } catch (Throwable e) {
            conn.close();
            throw e;
        }
        return conn;
    }

    @Override
    public MySqlConnectionSettings getSettingsForSync(String dbName) throws SQLException {
        return new ConnectionSettings(getDbConfigForSync(dbName));
    }

    @Override
    public void close() {
        dbConfigFactory.removeListener(this);
    }
}
