package ru.yandex.direct.mysql.ytsync.implementations;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil;
import ru.yandex.direct.mysql.ytsync.configuration.YtSyncSensorRegistry;
import ru.yandex.direct.mysql.ytsync.synchronizator.monitoring.SyncState;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SyncConfig;
import ru.yandex.direct.utils.TimeConvertUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeDouble;
import ru.yandex.monlib.metrics.registry.MetricId;

/**
 * Отправка отставания синхронизации с YT-кластером
 * Отставания для кластера по всем шардам заливаются в Juggler
 * Отправка происходит если состояние синхронизатора равно {@link SyncState#STABLE}
 * Под отставанием имеется ввиду разница между текущим моментом на момент вызова метода send
 * и значением lastTimestamp для шарда, которое берется из таблицы mysql-sync-states в YT'е
 * <p>
 * Класс используется из разных потоков и потому нужно следить за потокобезопасностью!!!
 */
@Lazy
@Component("timeLagConsumer")
public class TimeLagYtClusterSyncSender implements Consumer<Map<Integer, Long>> {

    private static final Logger logger = LoggerFactory.getLogger(TimeLagYtClusterSyncSender.class);

    private final List<String> dbNames;

    private final YtSyncSensorRegistry ytSyncSensorRegistry;

    private final Map<Integer, MetricId> shardGauges = new HashMap<>();

    @Autowired
    TimeLagYtClusterSyncSender(SyncConfig syncConfig,
                               YtSyncSensorRegistry ytSyncSensorRegistry) {
        this.dbNames = syncConfig.getDbNames();
        this.ytSyncSensorRegistry = ytSyncSensorRegistry;
        //
        initShardGauges();
    }

    private void initShardGauges() {
        for (String dbName : dbNames) {
            int shard = YtSyncCommonUtil.extractShard(dbName);
            MetricId metricId = new MetricId("time_lag", Labels.of("shard", dbName));
            ytSyncSensorRegistry.gaugeDouble(metricId);
            shardGauges.put(shard, metricId);
        }
    }

    private void sendShardsTimeLagToSolomon(Map<Integer, Long> shardsLastTimestamp, long currentTimeMillis) {
        for (Integer shard : shardsLastTimestamp.keySet()) {
            GaugeDouble gauge = ytSyncSensorRegistry.getGaugeDouble(shardGauges.get(shard));
            if (gauge != null) {
                Long lastTimestamp = shardsLastTimestamp.get(shard);
                gauge.set(TimeConvertUtils.millisToSecond(currentTimeMillis - lastTimestamp));
            }
        }
    }

    /**
     * Отправка отставания синхронизации с YT-кластером
     */
    @Override
    public void accept(Map<Integer, Long> shardsLastTimestamp) {
        long currentTimeMillis = System.currentTimeMillis();
        sendShardsTimeLagToSolomon(shardsLastTimestamp, currentTimeMillis);
    }
}
