package ru.yandex.direct.mysql.ytsync.task.builders;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record;
import org.jooq.TableField;
import org.jooq.types.UNumber;

import ru.yandex.direct.mysql.MySQLSimpleRow;
import ru.yandex.direct.mysql.ytsync.common.keys.PivotKeys;
import ru.yandex.direct.mysql.ytsync.common.model.JsonString;
import ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil;
import ru.yandex.direct.mysql.ytsync.task.config.DirectYtSyncConfig;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.lang.number.UnsignedLong;
import ru.yandex.yt.ytclient.tables.ColumnValueType;

import static ru.yandex.direct.mysql.ytsync.task.builders.JooqTaskBuilderUtil.emptyOrBooleanNode;
import static ru.yandex.direct.mysql.ytsync.task.builders.JooqTaskBuilderUtil.emptyOrLongNode;
import static ru.yandex.direct.mysql.ytsync.task.builders.JooqTaskBuilderUtil.emptyOrStringNode;
import static ru.yandex.direct.mysql.ytsync.task.builders.JooqTaskBuilderUtil.emptyOrUnsignedLongNode;
import static ru.yandex.direct.mysql.ytsync.task.builders.JooqTaskBuilderUtil.hasInSet;

@ParametersAreNonnullByDefault
public class JooqTaskBuilder {
    private static final BigDecimal MONEY_SCALE = BigDecimal.valueOf(1000000L);

    private final List<SyncTaskItem> taskItems = new ArrayList<>();
    private final SyncTableConnections syncTableConnections;
    private final Function<DirectYtSyncConfig, String> pathExtractor;
    private final PivotKeys pivotKeys;

    public JooqTaskBuilder(SyncTableConnections syncTableConnections,
                           Function<DirectYtSyncConfig, String> pathExtractor, PivotKeys pivotKeys) {
        this.syncTableConnections = syncTableConnections;
        this.pathExtractor = pathExtractor;
        this.pivotKeys = pivotKeys;
    }

    private <V, R> JooqTaskBuilder field(@Nullable String table, String srcName, String dstName, ColumnValueType type,
                                         @Nullable Function<String, YTreeNode> defaultValueProvider,
                                         @Nullable Function<V, R> transformer, @Nullable String expression,
                                         @Nullable Integer indexPriority, @Nullable FieldExtractorsPair<V> extractorsPair,
                                         boolean isKey) {
        return field(table, table, srcName, dstName, type, defaultValueProvider, transformer, expression,
                indexPriority, extractorsPair, isKey);
    }

    private <V, R> JooqTaskBuilder field(@Nullable String table, @Nullable String tableAlias, String srcName,
                                         String dstName, ColumnValueType type,
                                         @Nullable Function<String, YTreeNode> defaultValueProvider,
                                         @Nullable Function<V, R> transformer, @Nullable String expression,
                                         @Nullable Integer indexPriority,
                                         @Nullable FieldExtractorsPair<V> extractorsPair,
                                         boolean isKey) {
        taskItems.add(new SyncTaskItem<>(table, tableAlias, srcName, dstName, dstName, type, defaultValueProvider,
                transformer, expression, indexPriority, isKey, extractorsPair));
        return this;
    }

    public <R extends Record> JooqTaskBuilder longKey(TableField<R, Long> tableField) {
        return longKey(tableField.getTable().getName(), tableField);
    }

    public <R extends Record> JooqTaskBuilder longKey(String tableAlias, TableField<R, Long> tableField) {
        return field(tableField.getTable().getName(), tableAlias, tableField.getName(),
                tableField.getName(), ColumnValueType.INT64, null, null, null,
                null, FieldExtractorsPair.LONG_EXTR, true);
    }

    public <R extends Record> JooqTaskBuilder uLongKey(TableField<R, Long> tableField) {
        return uLongKey(tableField, tableField.getName(), null);
    }

    public <R extends Record> JooqTaskBuilder longKey(TableField<R, Long> tableField, String dstName) {
        return longKey(tableField, dstName, null);
    }

    public <R extends Record> JooqTaskBuilder longKey(TableField<R, Long> tableField, @Nullable Integer indexPriority) {
        return longKey(tableField, tableField.getName(), indexPriority);
    }

    public <R extends Record> JooqTaskBuilder longKey(TableField<R, Long> tableField, String dstName,
                                                      @Nullable Integer indexPriority) {
        return field(tableField.getTable().getName(), tableField.getName(), dstName,
                ColumnValueType.INT64, null, null, null, indexPriority,
                FieldExtractorsPair.LONG_EXTR, true);
    }

    public JooqTaskBuilder longKey(String tableName, String srcName) {
        return field(tableName, srcName, srcName,
                ColumnValueType.INT64, null, null, null, null,
                FieldExtractorsPair.LONG_EXTR, true);
    }

    public JooqTaskBuilder longField(String tableName, String srcName) {
        return field(tableName, srcName, srcName, ColumnValueType.INT64, FieldExtractorsPair.LONG_EXTR, null);
    }

    public <R extends Record> JooqTaskBuilder uLongKey(TableField<R, Long> tableField, String dstName,
                                                      @Nullable Integer indexPriority) {
        return field(tableField.getTable().getName(), tableField.getName(), dstName,
                ColumnValueType.UINT64, null, null, null, indexPriority,
                FieldExtractorsPair.ULONG_EXTR, true);
    }

    public JooqTaskBuilder uLongKey(String tableName, String srcName) {
        return field(tableName, srcName, srcName,
                ColumnValueType.UINT64, null, null, null, null,
                FieldExtractorsPair.ULONG_EXTR, true);
    }

    public JooqTaskBuilder uLongField(String tableName, String srcName) {
        return field(tableName, srcName, srcName, ColumnValueType.UINT64, FieldExtractorsPair.ULONG_EXTR, null);
    }

    public JooqTaskBuilder doubleField(String tableName, String srcName) {
        return field(tableName, srcName, srcName, ColumnValueType.DOUBLE, FieldExtractorsPair.DOUBLE_EXTR, null);
    }

    public <R extends Record> JooqTaskBuilder stringKey(TableField<R, String> tableField, String dstName) {
        return stringKey(tableField, dstName, null);
    }

    public <R extends Record> JooqTaskBuilder stringKey(TableField<R, String> tableField, String dstName,
                                                        @Nullable Function<String, String> transformer) {
        return field(tableField.getTable().getName(), tableField.getName(), dstName, ColumnValueType.STRING,
                null, transformer, null, null, FieldExtractorsPair.STRING_EXTR, true);
    }

    public JooqTaskBuilder stringKey(String tableName, String srcName) {
        return field(tableName, srcName, srcName,
                ColumnValueType.STRING, null, null, null, null,
                FieldExtractorsPair.STRING_EXTR, true);
    }

    public JooqTaskBuilder expressionKey(String name, String expression) {
        return field(null, name, name, ColumnValueType.INT64, null, null, expression, null, null, true);
    }

    public JooqTaskBuilder shardKey(String name) {
        return shardKey(name, null);
    }

    public JooqTaskBuilder shardKey(String name, @Nullable Integer indexPriority) {
        return field(null, name, name, ColumnValueType.INT64, dbName -> YTree.integerNode(YtSyncCommonUtil.extractShard(dbName)), null, null, indexPriority, null,
                true);
    }

    public <V, R> JooqTaskBuilder field(@Nullable String table, String srcName, String dstName, ColumnValueType type,
                                        FieldExtractorsPair<V> extractorsPair, @Nullable Function<V, R> transformer) {
        return field(table, table, srcName, dstName, type, extractorsPair, transformer);
    }

    public <V, R> JooqTaskBuilder field(@Nullable String table, @Nullable String tableAlias, String srcName,
                                        String dstName, ColumnValueType type, FieldExtractorsPair<V> extractorsPair,
                                        @Nullable Function<V, R> transformer) {
        return field(table, tableAlias, srcName, dstName, type, null, transformer, null, null, extractorsPair, false);
    }

    public <R extends Record, T> JooqTaskBuilder field(TableField<R, T> tableField, String dstName) {
        return field(tableField.getTable().getName(), tableField, dstName);
    }

    public <R extends Record, T> JooqTaskBuilder field(String tableAlias, TableField<R, T> tableField, String dstName) {
        Class<T> cls = tableField.getType();

        if (UNumber.class.isAssignableFrom(cls)) {
            return field(tableField.getTable().getName(), tableAlias, tableField.getName(), dstName,
                    ColumnValueType.UINT64, FieldExtractorsPair.ULONG_EXTR, null);
        } else if (Stream.of(Long.class, Integer.class, Short.class, Byte.class)
                .anyMatch(clazz -> clazz.isAssignableFrom(cls))) {
            return field(tableField.getTable().getName(), tableAlias, tableField.getName(), dstName,
                    ColumnValueType.INT64, FieldExtractorsPair.LONG_EXTR, null);
        } else if (Double.class.isAssignableFrom(cls) || BigDecimal.class.isAssignableFrom(cls)) {
            return field(tableField.getTable().getName(), tableAlias, tableField.getName(), dstName,
                    ColumnValueType.DOUBLE, FieldExtractorsPair.DOUBLE_EXTR, null);
        } else {
            return field(tableField.getTable().getName(), tableAlias, tableField.getName(), dstName,
                    ColumnValueType.STRING, null, null, null, null, FieldExtractorsPair.STRING_EXTR, false);
        }
    }

    public <R extends Record, T> JooqTaskBuilder ysonField(TableField<R, T> tableField, String dstName,
                                                           @Nullable Function<String, JsonString> transformer) {
        return ysonField(tableField.getTable().getName(), tableField, dstName, transformer);
    }

    public <R extends Record, T> JooqTaskBuilder ysonField(String tableAlias, TableField<R, T> tableField,
                                                           String dstName,
                                                           @Nullable Function<String, JsonString> transformer) {
        return field(tableField.getTable().getName(), tableAlias, tableField.getName(), dstName,
                ColumnValueType.ANY, null, transformer, null,
                null, FieldExtractorsPair.STRING_EXTR, false);
    }

    public <R extends Record, T> JooqTaskBuilder field(TableField<R, T> tableField) {
        return field(tableField, tableField.getName());
    }

    public <R extends Record, T> JooqTaskBuilder field(String tableAlias, TableField<R, T> tableField) {
        return field(tableAlias, tableField, tableField.getName());
    }

    public <R extends Record> JooqTaskBuilder moneyField(TableField<R, BigDecimal> tableField) {
        return moneyField(tableField, tableField.getName());
    }

    public <R extends Record> JooqTaskBuilder moneyField(TableField<R, BigDecimal> tableField, String dstName) {
        return moneyField(tableField.getTable().getName(), tableField, dstName);
    }

    public <R extends Record> JooqTaskBuilder moneyField(String tableAlias, TableField<R, BigDecimal> tableField,
                                                         String dstName) {
        return field(tableField.getTable().getName(), tableAlias, tableField.getName(), dstName,
                ColumnValueType.INT64, null, s -> s != null ? s.multiply(MONEY_SCALE).longValue() : null,
                null, null, FieldExtractorsPair.DOUBLE_EXTR, false);
    }

    public JooqTaskBuilder constField(String name, @Nullable String value) {
        return field(null, name, name, ColumnValueType.STRING, i -> emptyOrStringNode(value), null, null, null, null,
                false);
    }

    private JooqTaskBuilder constField(String name, @Nullable Long value, Boolean isKey,
                                       @Nullable Integer indexPriority) {
        return field(null, name, name, ColumnValueType.INT64, i -> emptyOrLongNode(value), null, null, indexPriority,
                null,
                isKey);
    }

    private JooqTaskBuilder constField(String name, @Nullable String value, Boolean isKey,
                                       @Nullable Integer indexPriority) {
        return field(null, name, name, ColumnValueType.STRING, i -> emptyOrStringNode(value), null, null, indexPriority,
                null,
                isKey);
    }

    public JooqTaskBuilder constField(String name, @Nullable Long value) {
        return constField(name, value, false, null);
    }

    public JooqTaskBuilder constField(String name, @Nullable Boolean value) {
        return field(null, name, name, ColumnValueType.BOOLEAN, i -> emptyOrBooleanNode(value), null, null, null,
                null,
                false);
    }

    private JooqTaskBuilder constField(String name, @Nullable UnsignedLong value, Boolean isKey,
                                       @Nullable Integer indexPriority) {
        return field(null, name, name, ColumnValueType.UINT64, i -> emptyOrUnsignedLongNode(value), null, null,
                indexPriority,
                null,
                isKey);
    }

    public JooqTaskBuilder constField(String name, @Nullable UnsignedLong value) {
        return constField(name, value, false, null);
    }

    public JooqTaskBuilder constKey(String name, Long value, @Nullable Integer indexPriority) {
        return constField(name, value, true, indexPriority);
    }

    public JooqTaskBuilder constKey(String name, UnsignedLong value, @Nullable Integer indexPriority) {
        return constField(name, value, true, indexPriority);
    }

    public JooqTaskBuilder constKey(String name, String value) {
        return constField(name, value, true, null);
    }

    public JooqTaskBuilder stringField(@Nullable String table, String name) {
        return stringField(table, name, name);
    }

    public JooqTaskBuilder stringField(@Nullable String table, String srcName, String dstName) {
        return stringField(table, srcName, dstName, null);
    }

    public <R extends Record> JooqTaskBuilder stringField(TableField<R, String> field, String dstName) {
        return stringField(field, dstName, null);
    }

    public <R extends Record> JooqTaskBuilder stringField(TableField<R, String> field, String dstName,
                                                          @Nullable Function<String, String> transformer) {
        return field(field.getTable().getName(), field.getName(), dstName, ColumnValueType.STRING,
                FieldExtractorsPair.STRING_EXTR, transformer);
    }

    public JooqTaskBuilder stringField(@Nullable String table, String srcName, String dstName,
                                       @Nullable Function<String, String> transformer) {
        return field(table, srcName, dstName, ColumnValueType.STRING, FieldExtractorsPair.STRING_EXTR, transformer);
    }

    public <R extends Record, E extends Enum<E>> JooqTaskBuilder enumField(TableField<R, E> tableField,
                                                                           String dstName) {
        return stringField(tableField.getTable().getName(), tableField.getName(), dstName);
    }

    public <R extends Record, E extends Enum<E>> JooqTaskBuilder enumField(TableField<R, E> tableField) {
        return enumField(tableField.getTable().getName(), tableField);
    }

    public <R extends Record, E extends Enum<E>> JooqTaskBuilder enumField(String tableAlias,
                                                                           TableField<R, E> tableField) {
        return field(tableField.getTable().getName(), tableAlias, tableField.getName(), tableField.getName(),
                ColumnValueType.STRING, null, null, null,
                null, FieldExtractorsPair.STRING_EXTR, false);
    }

    public <R extends Record> JooqTaskBuilder dateTimeField(TableField<R, LocalDateTime> tableField, String dstName,
                                                            @Nullable Function<LocalDateTime, Long> transformer) {
        return dateTimeField(tableField.getTable().getName(), tableField, dstName, transformer);
    }

    public <R extends Record> JooqTaskBuilder dateTimeField(String tableAlias, TableField<R, LocalDateTime> tableField,
                                                            String dstName,
                                                            @Nullable Function<LocalDateTime, Long> transformer) {
        return field(tableField.getTable().getName(), tableAlias, tableField.getName(), dstName,
                ColumnValueType.INT64, null, transformer, null, null, FieldExtractorsPair.LOCALDATETIME_EXTR, false);
    }

    public <R extends Record> JooqTaskBuilder dateTimeField(TableField<R, LocalDateTime> tableField) {
        return stringField(tableField.getTable().getName(), tableField.getName());
    }

    public <R extends Record> JooqTaskBuilder dateField(String tableAlias, TableField<R, LocalDate> tableField,
                                                        String dstName,
                                                        @Nullable Function<LocalDate, Long> transformer) {
        return field(tableField.getTable().getName(), tableAlias, tableField.getName(), dstName,
                ColumnValueType.INT64, null, transformer, null, null, FieldExtractorsPair.LOCALDATE_EXTR, false);
    }

    public <R extends Record> JooqTaskBuilder dateField(TableField<R, LocalDate> tableField) {
        return dateField(tableField.getTable().getName(), tableField);
    }

    public <R extends Record> JooqTaskBuilder dateField(String tableAlias, TableField<R, LocalDate> tableField) {
        return field(tableField.getTable().getName(), tableAlias, tableField.getName(), tableField.getName(),
                ColumnValueType.STRING, null, null, null, null, FieldExtractorsPair.STRING_EXTR, false);
    }

    public <R extends Record> JooqTaskBuilder numFlagSetField(TableField<R, String> tableField, String flagName,
                                                              String dstName) {
        return field(tableField.getTable().getName(), tableField.getName(), dstName,
                ColumnValueType.INT64,
                null, hasInSet(flagName).andThen(b -> b ? 1 : 0), null, null, FieldExtractorsPair.STRING_EXTR,
                false);
    }

    public <R extends Record> JooqTaskBuilder flagSetField(TableField<R, String> tableField, String flagName,
                                                           String dstName) {
        return field(tableField.getTable().getName(), tableField.getName(), dstName,
                ColumnValueType.BOOLEAN,
                null, hasInSet(flagName), null, null, FieldExtractorsPair.STRING_EXTR, false);
    }

    public <R extends Record> JooqTaskBuilder flagSetField(TableField<R, String> tableField, String flagName) {
        return flagSetField(tableField.getTable().getName(), tableField, flagName);
    }

    public <R extends Record> JooqTaskBuilder flagSetField(String tableAlias, TableField<R, String> tableField,
                                                           String flagName) {
        return field(tableField.getTable().getName(), tableAlias, tableField.getName(),
                tableField.getName() + "_" + flagName, ColumnValueType.BOOLEAN,
                null, hasInSet(flagName), null, null, FieldExtractorsPair.STRING_EXTR, false);
    }

    public SyncTask build() {
        return new SyncTask(syncTableConnections, null, taskItems, pathExtractor, pivotKeys);
    }

    public SyncTask build(Predicate<MySQLSimpleRow> skipPredicate) {
        return new SyncTask(syncTableConnections, skipPredicate, taskItems, pathExtractor, pivotKeys);
    }
}
