package ru.yandex.direct.mysql.ytsync.task.builders;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class SyncTableConnections {
    private static class SyncJoin {
        private final String join;
        private final String table;
        private final String tableAlias;
        private final Map<String, String> keys;

        private SyncJoin(String join, String table, Map<String, String> keys) {
            this(join, table, table, keys);
        }

        private SyncJoin(String join, String table, String tableAlias, Map<String, String> keys) {
            this.join = join;
            this.table = table;
            this.keys = keys;
            this.tableAlias = tableAlias;
        }

        private String buildJoin(String baseAlias) {
            var ons = new ArrayList<String>(keys.size());
            keys.forEach((baseKey, joinKey) -> {
                ons.add(String.format("`%s`.`%s` = `%s`.`%s`", baseAlias, baseKey, tableAlias, joinKey));
            });

            return String.format(" %s `%s` AS `%s` ON %s", join, table, tableAlias, String.join(" AND ", ons));
        }
    }

    private final String mainTable;
    private final String mainTableKey;
    private final String mainTableAlias;
    private final List<SyncJoin> joinList;
    private String where;

    public SyncTableConnections(String mainTable, String mainTableKey) {
        this(mainTable, mainTable, mainTableKey);
    }

    public SyncTableConnections(String mainTable, String mainTableAlias, String mainTableKey) {
        this.mainTable = mainTable;
        this.mainTableAlias = mainTableAlias;
        this.mainTableKey = mainTableKey;

        joinList = new ArrayList<>();
    }

    public String getMainTable() {
        return mainTable;
    }

    public String getMainTableAlias() {
        return mainTableAlias;
    }

    public String getMainTableKey() {
        return mainTableKey;
    }

    public String getFromSqlPart() {
        StringBuilder sb = new StringBuilder(String.format("`%s` AS `%s`", mainTable, mainTableAlias));

        for (SyncJoin syncJoin : joinList) {
            sb.append(syncJoin.buildJoin(mainTableAlias));
        }

        return sb.toString();
    }

    public SyncTableConnections where(String fieldName, String value) {
        where = String.format("`%s`.`%s` = '%s'", mainTableAlias, fieldName, value);
        return this;
    }

    public SyncTableConnections whereIn(String fieldName, Collection<String> values) {
        where = String.format("`%s`.`%s` IN (%s)", mainTableAlias, fieldName,
                values.stream()
                        .map(v -> "'" + v + "'")
                        .collect(Collectors.joining(","))
        );
        return this;
    }

    /**
     * LEFT JOIN по ключу с одинаковым названием в объединяемых таблицах
     *
     * @param tableName - таблица, с которой происходит join
     * @param using     - колонка, используемая для сравнения и в исходной, и в присоединяемой таблицах
     * @return модель объединенной таблицы для запроса
     */
    public SyncTableConnections leftJoin(String tableName, String using) {
        return leftJoin(tableName, tableName, using);
    }

    public SyncTableConnections leftJoin(String tableName, String tableAlias, String using) {
        joinList.add(new SyncJoin("LEFT JOIN", tableName, tableAlias, Map.of(using, using)));
        return this;
    }

    /**
     * LEFT JOIN по ключу с различными названиями в объединяемых таблицах
     *
     * @param tableName  - таблица, с которой происходит join
     * @param tableAlias - alias таблицы, с которой происходит join
     * @param using      - колонка, используемая для сравнения в присоединяемой таблице
     * @param baseUsing  - колонка, используемая для сравнения в исходной таблице
     * @return модель объединенной таблицы для запроса
     */
    public SyncTableConnections leftJoin(String tableName, String tableAlias, String using, String baseUsing) {
        joinList.add(new SyncJoin("LEFT JOIN", tableName, tableAlias, Map.of(baseUsing, using)));
        return this;
    }

    public String getWhere() {
        return where;
    }
}
