package ru.yandex.direct.mysql.ytsync.task.builders;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

import ru.yandex.direct.mysql.MySQLSimpleRow;
import ru.yandex.direct.mysql.ytsync.common.keys.PivotKeys;
import ru.yandex.direct.mysql.ytsync.export.util.valueprocessing.ExportFieldProcessor;
import ru.yandex.direct.mysql.ytsync.synchronizator.tableprocessors.SyncFlatRowCreator;
import ru.yandex.direct.mysql.ytsync.synchronizator.tableprocessors.SyncFlatRowCreatorBase;
import ru.yandex.direct.mysql.ytsync.synchronizator.tableprocessors.SyncFlatRowFieldProcessor;
import ru.yandex.direct.mysql.ytsync.task.config.DirectYtSyncConfig;
import ru.yandex.yt.ytclient.tables.ColumnSchema;
import ru.yandex.yt.ytclient.tables.TableSchema;

@ParametersAreNonnullByDefault
public class SyncTask {
    private final Predicate<MySQLSimpleRow> skipPredicate;
    private final SyncTableConnections syncTableConnections;
    private final List<SyncTaskItem> taskItems;
    private final Function<DirectYtSyncConfig, String> pathExtractor;
    private final PivotKeys pivotKeys;

    private final TableSchema tableSchema;
    private final TableSchema indexSchema;

    SyncTask(SyncTableConnections syncTableConnections, @Nullable Predicate<MySQLSimpleRow> skipPredicate,
             List<SyncTaskItem> taskItems, Function<DirectYtSyncConfig, String> pathExtractor, PivotKeys pivotKeys) {
        this.syncTableConnections = syncTableConnections;
        this.taskItems = ImmutableList.copyOf(taskItems);
        this.pathExtractor = pathExtractor;
        this.pivotKeys = pivotKeys;
        this.skipPredicate = skipPredicate;

        tableSchema = createTableSchema();
        indexSchema = createIndexSchema();
    }

    private TableSchema createIndexSchema() {
        List<SyncTaskItem> indexItems = taskItems.stream()
                .filter(ti -> ti.isKey() && ti.getDstExpression() == null)
                .sorted(Comparator.comparing(SyncTaskItem::getIndexPriority))
                .collect(Collectors.toList());
        indexItems = Lists.reverse(indexItems);

        if (indexItems.isEmpty() || indexItems.get(0).getIndexPriority() == Integer.MIN_VALUE) {
            return null;
        }
        TableSchema.Builder builder = new TableSchema.Builder();
        for (SyncTaskItem indexItem : indexItems) {
            if (indexItem.getIndexPriority() > Integer.MIN_VALUE) {
                builder = builder.addKey(indexItem.getDstFieldName(), indexItem.getDstType());
            } else {
                builder = builder.addValue(indexItem.getDstFieldName(), indexItem.getDstType());
            }
        }

        return builder.build();
    }

    private TableSchema createTableSchema() {
        TableSchema.Builder builder = new TableSchema.Builder();
        for (SyncTaskItem taskItem : taskItems) {
            if (taskItem.isKey()) {
                if (taskItem.getDstExpression() != null) {
                    builder = builder.addKeyExpression(taskItem.getDstFieldName(), taskItem.getDstType(),
                            taskItem.getDstExpression());
                } else {
                    builder = builder.addKey(taskItem.getDstFieldName(), taskItem.getDstType());
                }
            } else {
                builder = builder.addValue(taskItem.getDstFieldName(), taskItem.getDstType());
            }
        }
        return builder.build();
    }

    public TableSchema getTableSchema() {
        return tableSchema;
    }

    public TableSchema getIndexSchema() {
        return indexSchema;
    }

    public boolean hasIndexTable() {
        return indexSchema != null;
    }

    public String getIndexTablePath(DirectYtSyncConfig config) {
        return getTablePath(config) + ".index";
    }

    public String getTablePath(DirectYtSyncConfig config) {
        return pathExtractor.apply(config);
    }

    public String getSqlTemplate() {
        StringBuilder sb = new StringBuilder("SELECT ");
        Set<String> fields = new HashSet<>();
        for (SyncTaskItem taskItem : taskItems) {
            if (taskItem.getSrcTableAlias() != null) {
                fields.add(String.format("`%s`.`%s` AS `%s`", taskItem.getSrcTableAlias(), taskItem.getSrcFieldName(),
                        taskItem.getSrcFieldAlias()));
            }
        }
        sb.append(String.join(", ", fields));
        sb.append(" FROM ").append(syncTableConnections.getFromSqlPart()).append(" ");
        sb.append("WHERE ");
        if (syncTableConnections.getWhere() != null) {
            sb.append(syncTableConnections.getWhere()).append(" AND ");
        }
        sb.append("`").append(getMainTableAlias()).append("`.`").append(getMainIdColumn())
                .append("` BETWEEN {MIN_ID} AND {MAX_ID}");
        return sb.toString();
    }

    public String getMainTableName() {
        return syncTableConnections.getMainTable();
    }

    public String getMainTableAlias() {
        return syncTableConnections.getMainTableAlias();
    }

    public String getMainIdColumn() {
        return syncTableConnections.getMainTableKey();
    }

    public PivotKeys getTablePivotKeys() {
        return pivotKeys;
    }

    public Map<String, Map<String, List<ExportFieldProcessor<?>>>> getExportTableFieldProcessors() {
        Map<String, Map<String, List<ExportFieldProcessor<?>>>> result = new HashMap<>();
        for (SyncTaskItem taskItem : taskItems) {
            if (taskItem.getSrcTable() == null) {
                continue;
            }
            Map<String, List<ExportFieldProcessor<?>>> tableResult =
                    result.computeIfAbsent(taskItem.getSrcTable(), i -> new HashMap<>());
            List<ExportFieldProcessor<?>> fieldProcessors =
                    tableResult.computeIfAbsent(taskItem.getSrcFieldName(), i -> new ArrayList<>());
            fieldProcessors.add(taskItem.createExportFieldProcessor());
        }

        return result;
    }

    public List<ExportFieldProcessor<?>> getExportAdditionalFields() {
        return taskItems.stream()
                .filter(i -> i.getSrcTable() == null && i.getDstExpression() == null)
                .map(SyncTaskItem::createExportFieldProcessor)
                .map(fp -> (ExportFieldProcessor<?>) fp)
                .collect(Collectors.toList());
    }

    public Set<String> getAllTables() {
        return taskItems.stream()
                .map(SyncTaskItem::getSrcTable)
                .collect(Collectors.toSet());
    }

    public Map<String, Map<String, List<SyncFlatRowFieldProcessor<?>>>> getSyncTableFieldProcessors() {
        Map<String, Map<String, List<SyncFlatRowFieldProcessor<?>>>> result = new HashMap<>();
        for (SyncTaskItem taskItem : taskItems) {
            if (taskItem.getSrcTable() == null) {
                continue;
            }
            Map<String, List<SyncFlatRowFieldProcessor<?>>> tableResult =
                    result.computeIfAbsent(taskItem.getSrcTable(), i -> new HashMap<>());
            List<SyncFlatRowFieldProcessor<?>> fieldProcessors =
                    tableResult.computeIfAbsent(taskItem.getSrcFieldName(), i -> new ArrayList<>());
            fieldProcessors.add(taskItem.createSyncFieldProcessor());
        }

        return result;
    }

    public List<SyncFlatRowFieldProcessor<?>> getSyncOptionalAdditionalFields() {
        return taskItems.stream()
                .filter(i -> i.getSrcTable() == null && i.getDstExpression() == null && !i.isKey())
                .map(SyncTaskItem::createSyncFieldProcessor)
                .map(fp -> (SyncFlatRowFieldProcessor<?>) fp)
                .collect(Collectors.toList());
    }

    public List<SyncFlatRowFieldProcessor<?>> getSyncAdditionalFields() {
        return taskItems.stream()
                .filter(i -> i.getSrcTable() == null && i.getDstExpression() == null && i.isKey())
                .map(SyncTaskItem::createSyncFieldProcessor)
                .map(fp -> (SyncFlatRowFieldProcessor<?>) fp)
                .collect(Collectors.toList());
    }

    private List<String> getMainTableKeys() {
        return tableSchema.getColumns().stream()
                .filter(c -> c.getSortOrder() != null && c.getExpression() == null)
                .map(ColumnSchema::getName)
                .collect(Collectors.toList());
    }

    public SyncFlatRowCreatorBase getSyncFlatRowCreator() {
        return new SyncFlatRowCreator(tableSchema, getMainTableName(), getMainTableKeys(),
                getSyncTableFieldProcessors(), getSyncAdditionalFields(),
                getSyncOptionalAdditionalFields());
    }

    public boolean hasSkipPredicate() {
        return skipPredicate != null;
    }

    public Predicate<MySQLSimpleRow> getSkipPredicate() {
        return skipPredicate;
    }
}
