package ru.yandex.direct.mysql.ytsync.task.config;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import com.typesafe.config.Config;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.model.YtCluster;

/**
 * Обёртка над DirectConfig для доступа к конфигурации yt-sync
 */
public class DirectYtSyncConfig extends RootPathConfig {
    private final Config primarySettings;
    private final Config pathSettings;
    private final Config importSettings;
    private final String clusterName;
    private final YtCluster cluster;
    private final YtClusterConfig clusterConfig;
    private final Config perClusterSettings;

    public DirectYtSyncConfig(DirectConfig directConfig, YtClusterConfig clusterConfig) {
        super(directConfig);

        this.primarySettings = directConfig.getBranch("yt-sync").getConfig();
        this.pathSettings = primarySettings.getConfig("path");
        this.importSettings = primarySettings.getConfig("import");
        this.clusterName = primarySettings.getString("cluster");
        this.cluster = YtCluster.clusterFromNameToUpperCase(clusterName);
        this.clusterConfig = clusterConfig;
        this.perClusterSettings = createPerClusterSettings();
    }

    private Config createPerClusterSettings() {
        Config settings = primarySettings.getConfig("default");
        if (primarySettings.hasPath(clusterName)) {
            // Настройки кластера переопределяют настройки из default
            settings = primarySettings.getConfig(clusterName).withFallback(settings);
        }
        if (primarySettings.hasPath("all")) {
            // Настройки all переопределяют настройки кластера
            settings = primarySettings.getConfig("all").withFallback(settings);
        }
        return settings;
    }

    public YtCluster getCluster() {
        return cluster;
    }

    /**
     * Кол-во потоков для работы асинхронных обработчиков
     */
    public int asyncWorkerThreads() {
        return perClusterSettings.getInt("async-threads");
    }

    /**
     * Кол-во потоков для выполнения синхронных запросов в yt
     */
    public int syncRequestThreads() {
        return perClusterSettings.getInt("request-threads");
    }

    /**
     * Хост http прокси для использования в кошерном клиенте
     */
    public String getYtProxyHost() {
        return clusterConfig.getProxyHost();
    }

    /**
     * Порт для http прокси
     */
    public int getYtProxyPort() {
        return clusterConfig.getProxyPort();
    }

    /**
     * Имя пользователя для аутентификации в клиентах
     */
    public String getYtUser() {
        return clusterConfig.getUser();
    }

    /**
     * Токен для аутентификации в клиентах
     */
    public String getYtToken() {
        return clusterConfig.getToken();
    }

    /**
     * Количество ретраев в кошерном клиенте
     */
    public int getKosherRetries() {
        return perClusterSettings.getInt("kosher-retries");
    }

    /**
     * Таймаут в секундах на тяжёлые команды в кошерном клиенте
     */
    public int getKosherHeavyTimeoutSeconds() {
        return perClusterSettings.getInt("kosher-heavy-timeout-seconds");
    }

    /**
     * Размер чанка записи данных в кошерном клиенте
     */
    public int getKosherWriteChunkSize() {
        return perClusterSettings.getInt("kosher-write-chunk-size");
    }

    /**
     * Тип компрессии для использования в кошерном клиенте
     */
    public String getKosherCompression() {
        return perClusterSettings.getString("kosher-compression");
    }

    /**
     * Разрешено ли использовать rpc клиент
     */
    public boolean useRpcClient() {
        return perClusterSettings.getBoolean("use-rpc-client");
    }

    /**
     * Максимальное кол-во секунд, которое даётся операциям на выполнение внутри транзакций
     */
    public int innerRetryTimeoutSeconds() {
        return perClusterSettings.getInt("inner-retry-timeout-seconds");
    }

    /**
     * Максимальное кол-во секунд, которое даётся операциям на выполнение вне транзакций
     */
    public int outerRetryTimeoutSeconds() {
        return perClusterSettings.getInt("outer-retry-timeout-seconds");
    }

    /**
     * Максимальный размер чанка, который можно использовать в операциях записи на yt
     */
    public int maxWriteChunkSize() {
        return perClusterSettings.getInt("max-chunk-size");
    }

    /**
     * Путь ссылки указывающую на папку из которой grid-core читает данные из YT'а
     */
    public String stableLinkPath() {
        return pathSettings.getString("stable-link");
    }

    /**
     * Путь к таблице campaigns
     */
    public String campaignsPath() {
        return rootPath() + pathSettings.getString("table-campaigns");
    }

    /**
     * Путь к таблице phrases
     */
    public String phrasesPath() {
        return rootPath() + pathSettings.getString("table-phrases");
    }

    /**
     * Путь к таблице bids
     */
    public String bidsPath() {
        return rootPath() + pathSettings.getString("table-bids");
    }

    /**
     * Путь к таблице retargetingConditions
     */
    public String retargetingConditionsPath() {
        return rootPath() + pathSettings.getString("table-retargeting-conditions");
    }

    /**
     * Путь к таблице bidsRetargeting
     */
    public String bidsRetargetingPath() {
        return rootPath() + pathSettings.getString("table-bids-retargeting");
    }

    /**
     * Путь к таблице banners
     */
    public String bannersPath() {
        return rootPath() + pathSettings.getString("table-banners");
    }

    /**
     * Путь к таблице perf_creatives
     */
    public String perfCreativesPath() {
        return rootPath() + pathSettings.getString("table-perf-creatives");
    }

    /**
     * Путь к таблице recommendations-status
     */
    public String recommendationsStatusPath() {
        return rootPath() + pathSettings.getString("table-recommendations-status");
    }

    /**
     * Путь к таблице recommendations-online
     */
    public String recommendationsOnlinePath() {
        return rootPath() + pathSettings.getString("table-recommendations-online");
    }

    /**
     * Путь к таблице tag_group
     */
    public String tagGroupPath() {
        return rootPath() + pathSettings.getString("table-tag-group");
    }

    /**
     * Нужно ли использовать slave'ы для первоначального импорта данных
     * @param dbName имя БД, например "ppc:1" или "ppcdict"
     */
    public boolean importUsingSlaves(String dbName) {
        return importSettings.getBoolean("use-slaves")
                // временное исключение для ppcdict, пока не станет понятно, откуда брать раба для нее
                // на данный момент команда direct-sql pr:ppcdict "SHOW SLAVE HOSTS;"
                // отдает пустой список
                && !dbName.equalsIgnoreCase(PPCDICT_DBNAME);

    }

    /**
     * Нужно ли использовать слейвы для синхронизации (инкрементального импорта)
     * Обычно предполагается, что инсталляции в STABLE читают из мастера, а PRESTABLE могут пользоваться слейвами
     * @param dbName имя БД, например "ppc:1" или "ppcdict"
     */
    public boolean syncUsingSlaves(String dbName) {
        return perClusterSettings.getBoolean("use-slaves")
                // временное исключение для ppcdict, пока не станет понятно, откуда брать раба для нее
                // на данный момент команда direct-sql pr:ppcdict "SHOW SLAVE HOSTS;"
                // отдает пустой список
                && !dbName.equalsIgnoreCase(PPCDICT_DBNAME);
    }

    /**
     * Импортировать все таблицы из mysql
     */
    public boolean importAllTables() {
        return perClusterSettings.getBoolean("import-all-tables");
    }

    /**
     * Запускать ли начальный импорт автоматически
     */
    public boolean importAutorun() {
        return importSettings.getBoolean("autorun");
    }

    /**
     * Маска хостов, которые можно использовать в качестве слейвов
     */
    public Pattern slaveHostPattern() {
        List<String> patterns = new ArrayList<>();
        for (String mask : primarySettings.getStringList("slave-host-masks")) {
            patterns.add(regexFromMask(mask));
        }
        if (patterns.isEmpty()) {
            patterns.add("\\A.*\\z");
        }
        return Pattern.compile(String.join("|", patterns), Pattern.DOTALL);
    }

    /**
     * Конвертирует простую маску с символами * и ? в regex
     */
    private static String regexFromMask(String mask) {
        StringBuilder sb = new StringBuilder();
        sb.append("\\A");
        for (int i = 0; i < mask.length(); i++) {
            char ch = mask.charAt(i);
            switch (ch) {
                case '*':
                    sb.append(".*");
                    break;
                case '?':
                    sb.append(".");
                    break;
                case '+':
                case '{':
                case '}':
                case '$':
                case '.':
                case '^':
                case '(':
                case ')':
                case '[':
                case ']':
                case '|':
                case '\\':
                    sb.append('\\');
                    // fall through
                default:
                    sb.append(ch);
                    break;
            }
        }
        sb.append("\\z");
        return sb.toString();
    }
}
