package ru.yandex.direct.mysql.ytsync.task.config;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import com.typesafe.config.Config;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesConfig;
import ru.yandex.direct.mysql.ytsync.export.task.ExportConfig;
import ru.yandex.direct.utils.io.FileUtils;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.model.attributes.OptimizeForAttr;

/**
 * Обёртка над DirectConfig для доступа к конфигурации yt-sync
 */
public class ExportConfigImpl extends RootPathConfig implements ExportConfig, SyncStatesConfig {
    private static final long DEFAULT_CHUNKS_LIMIT = Long.MAX_VALUE;
    private static final OptimizeForAttr OPTIMIZE_FOR = OptimizeForAttr.SCAN;
    private static final int PPCDICT_MAX_IMPORT_THREADS = 4;

    private final DirectConfig directConfig;
    private final Config primarySettings;
    private final Config pathSettings;
    private final Config importSettings;
    private final String clusterName;
    private final YtClusterConfig clusterConfig;
    private final Config perClusterSettings;
    private final List<String> dbNames;

    public ExportConfigImpl(
            DirectConfig directConfig,
            YtClusterConfig clusterConfig,
            List<String> dbNames) {
        super(directConfig);

        this.directConfig = directConfig;
        this.primarySettings = directConfig.getBranch("yt-sync").getConfig();
        this.pathSettings = primarySettings.getConfig("path");
        this.importSettings = primarySettings.getConfig("import");
        this.clusterName = primarySettings.getString("cluster");
        this.clusterConfig = clusterConfig;
        this.perClusterSettings = createPerClusterSettings();
        this.dbNames = dbNames;
    }

    private Config createPerClusterSettings() {
        Config settings = primarySettings.getConfig("default");
        if (primarySettings.hasPath(clusterName)) {
            // Настройки кластера переопределяют настройки из default
            settings = primarySettings.getConfig(clusterName).withFallback(settings);
        }
        if (primarySettings.hasPath("all")) {
            // Настройки all переопределяют настройки кластера
            settings = primarySettings.getConfig("all").withFallback(settings);
        }
        return settings;
    }

    /**
     * Импортировать все таблицы из mysql
     */
    @Override
    public boolean importAllTables() {
        return perClusterSettings.getBoolean("import-all-tables");
    }

    /**
     * Список баз для синхронизации
     * Если не задано в конфиге, получаем на основе данных о количестве шардов
     */
    @Override
    public List<String> getDbNames() {
        return dbNames;
    }

    /**
     * Множество таблиц, исключаемых из синхронизации
     * Допускается использование регулярных выражений
     */
    @Override
    public Set<String> getExcludeTableNames() {
        return ImmutableSet.copyOf(perClusterSettings.getStringList("exclude-table-names").stream()
                .map(String::toLowerCase)
                .collect(Collectors.toSet()));
    }

    /**
     * Множество таблиц, включаемых в синхронизацию
     * Если множество пустое, параметр игнорируется
     */
    @Override
    public Set<String> getIncludeTableNames() {
        return ImmutableSet.copyOf(perClusterSettings.getStringList("include-table-names").stream()
                .map(String::toLowerCase)
                .collect(Collectors.toSet()));
    }

    /**
     * Хост http прокси для использования в кошерном клиенте
     */
    @Override
    public String getYtProxy() {
        return clusterConfig.getProxy();
    }

    /**
     * Путь до файла с токеном для аутентификаци в клиентах
     */
    @Override
    public String getYtTokenPath() {
        String tokenPath = directConfig.findString("yt." + clusterName + ".token_file")
                .orElse(YtClusterConfig.DEFAULT_YT_TOKEN_PATH);
        return FileUtils.expandHome(tokenPath).toString();
    }

    @Override
    public String getTablesPrefix() {
        return pathSettings.getString("static-import-tables-prefix");
    }

    /**
     * Указывает, какой тип хранилища следует использовать для таблиц на yt
     */
    @Override
    public String ytTablesMedium() {
        if (perClusterSettings.hasPath("tables-medium")) {
            return perClusterSettings.getString("tables-medium");
        } else {
            return "default";
        }
    }

    @Override
    public OptimizeForAttr getOptimizeFor() {
        return OPTIMIZE_FOR;
    }

    /**
     * Ограничение на кол-во чанков при импорте
     */
    @Override
    public long chunksLimit() {
        if (importSettings.hasPath("chunks-limit")) {
            return importSettings.getLong("chunks-limit");
        } else {
            return DEFAULT_CHUNKS_LIMIT;
        }
    }

    /**
     * Размер одного чанка для импорта входных таблиц
     */
    @Override
    public int chunkSize() {
        return importSettings.getInt("chunk-size");
    }

    /**
     * Кол-во потоков для начального импорта данных
     *
     * @param dbNames список имен импортируемых БД
     */
    @Override
    public int threadsNum(List<String> dbNames) {
        boolean ppcDictInDbNames = dbNames.stream().anyMatch(PPCDICT_DBNAME::equalsIgnoreCase);
        int defaultThreadsNum = importSettings.getInt("threads");
        return ppcDictInDbNames ? Math.min(defaultThreadsNum, PPCDICT_MAX_IMPORT_THREADS) : defaultThreadsNum;
    }

    @Override
    public int flushIntervalMinutes() {
        return perClusterSettings.getInt("flush-interval-minutes");
    }

    @Override
    public boolean profilingByTag() {
        return perClusterSettings.hasPath("profiling-by-tag")
                ? perClusterSettings.getBoolean("profiling-by-tag")
                : false;
    }

    @Override
    public boolean dynamicStoreRead() {
        return perClusterSettings.hasPath("dynamic-store-read")
                ? perClusterSettings.getBoolean("dynamic-store-read")
                : false;
    }

    /**
     * Путь к таблице sync-states
     */
    @Override
    public String syncStatesPath() {
        return rootPath() + pathSettings.getString("table-sync-states");
    }

    /**
     * primary_medium таблицы sync-states
     */
    @Override
    public String syncTableMedium() {
        // Используем те же настройки medium, что и для экспорта
        return ytTablesMedium();
    }
}
